import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  decodeSingle_statement,
  deleteRecord,
  deleteSettingRecord,
  encodeAndStringifyFields,
  encodeSingle_statement,
  getOrganizationAccordingToDepartment,
  getRecord,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  settingsUpdater,
  uniqueIdGenerator,
  updateQueryBuilder,
  uploadFile,
  whereCondition,
} from "../helper/general.js";

import { sendResponse } from "../helper/wrapper.js";
import Asset from "../sequelize/AssetSchema.js";

export const createUpdateAsset = async (req, res) => {
  let { id, name, department, organization } = req.body;
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  const [checkNameWithOrganization] = await getRecord("asset", "name", name);
  if (checkNameWithOrganization) {
    if (id && checkNameWithOrganization.id != id) {
      return sendResponse(res, 400, "Asset name already exist");
    } else if (!id) {
      return sendResponse(res, 400, "Asset name already exist");
    }
  }

  if (!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "AM",
      "asset",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  /**If id comes in body then it will update the query */
  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id
    ? updateQueryBuilder(Asset, req.body)
    : createQueryBuilder(Asset, req.body);
  const [result] = await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Asset",
    id ? id : result.insertId
  );
  return sendResponse(res, 200, `Record ${status} Successfully`);
};

export const getAsset = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "asset",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });

  const searchTableName = [
    "department.name",
    "organization.name",
    "location.name",
    "asset_type.name",
    "asset.asset_name",
    "asset.asset_description",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = asset.location",
    },
    {
      type: "left",
      targetTable: "asset_type",
      onCondition: "asset_type.id = asset.asset_type",
    },
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = asset.created_by",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = asset.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = asset.organization",
    },
    {
      type: "left",
      targetTable: "department as asset_department",
      onCondition: "asset_department.id = asset.asset_department",
    },
    {
      type: "left",
      targetTable: "currency as purchase_currency",
      onCondition: "purchase_currency.id = asset.purchase_currency",
    },
    {
      type: "left",
      targetTable: "currency as lease_currency",
      onCondition: "lease_currency.id = asset.lease_currency",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const assetQuery = `SELECT asset.* , CONCAT(createdUser.name , ' ' , createdUser.surname) AS created_by,
  location.name AS location_name, asset_type.name AS asset_type_name, asset_department.name as asset_department_name, department.name AS department_name, purchase_currency.name as purchase_currency_name, lease_currency.name as lease_currency_name,  organization.name AS organization_name FROM asset ${joinCondition} WHERE asset.deleted = 0 ${searchCondition} ${condition}`;
  let [asset] = await db.query(assetQuery);
  asset = await decodeAndParseFields(asset);
  const totalRecord = await countQueryCondition(assetQuery);
  return sendResponse(res, 200, asset, totalRecord);
};

export const deleteAsset = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(Asset, id);
  await insertActivityLog(req.user.sessionid, "delete", "Asset", id);
  return sendResponse(res, 200, "Record deleted successfully");
};
