import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  pointsCalculate,
  processesSingleDDRMDocument,
  searchConditionRecord,
  updateQueryBuilder,
  uploadFile,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import AuditExecution from "../sequelize/AuditExecutionSchema.js";
import AuditRecord from "../sequelize/AuditRecordSchema.js";

/**Function to create and update Audit Execution  */
export const createUpdateAuditExecution = async (req, res) => {
  const { id, scheduled_audit, organization, department, sidebar_id = 155 } = req.body;

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization = await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  if (req.files && req.files.report) {
    const file = req.files.report;
    // req.body.report = await uploadFile("audit_execution", req.files.report);
    const ddrm_id = await processesSingleDDRMDocument("audit_execution", sidebar_id, file, req);
    req.body.ddrm_id = ddrm_id;
    req.body.evaluation_status = "Executed";
  }

  /** Check that for this audit Scheduled audit execution already exist or not */
  const [checkAuditExecutionWithOrganization] = await db.query(
    "SELECT id FROM audit_execution WHERE scheduled_audit = ? AND deleted = '0' AND id != ? AND organization = ?",
    [scheduled_audit, id, organizationId]
  );
  if (checkAuditExecutionWithOrganization.length > 0) {
    return sendResponse(res, 400, `Record with Scheduled Audit ${scheduled_audit} already exists.`);
  }

  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;
    /**Check that this organization have that particular Scheduled Audit or not*/
    const [checkScheduledAuditWithOrganization] = await db.query(
      "SELECT scheduled_audit FROM audit_execution WHERE scheduled_audit = ? AND deleted = '0' AND id != ? AND organization = ?",
      [scheduled_audit, id, organizationId]
    );
    if (checkScheduledAuditWithOrganization.length > 0) {
      return sendResponse(res, 400, `Record with Scheduled Audit ${scheduled_audit} already exists.`);
    }

    const { query, values } = updateQueryBuilder(AuditExecution, req.body);
    await db.query(query, values);
    /**Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "update",
      "Audit Execution",
      `This user Update Audit Execution which id is ${id}`
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    /** Check if this organization already has the particular scheduled_audit */
    const [checkScheduledAuditWithOrganization] = await db.query(
      `SELECT scheduled_audit FROM audit_execution WHERE scheduled_audit = ? AND deleted = '0' AND organization = ?`,
      [scheduled_audit, organizationId]
    );
    if (checkScheduledAuditWithOrganization.length > 0) {
      return sendResponse(res, 400, `Record with Scheduled Audit ${scheduled_audit} already exists.`);
    }
    req.body.created_by = req.user.sessionid;
    req.body.status = req.body.status ?? "Scheduled";
    /** Insert Audit Execution details */
    const { query, values } = createQueryBuilder(AuditExecution, req.body);

    await db.query(query, values);

    /** Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "create",
      "Audit Execution",
      `This user created a new Audit Execution for audit scheduled ${scheduled_audit} for organization ${organizationId}`
    );

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all Audit Execution */
export const viewAllAuditExecution = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "audit_execution",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });

  const searchTableName = [
    "audit_scheduling.audit_name",
    "audit_execution.status",
    "audit_scheduling.audit_type",
    "audit_template.name",
    "organization.name",
    "CONCAT(users.name , ' ' , users.surname)",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = audit_execution.created_by",
    },
    {
      type: "left",
      targetTable: "audit_scheduling",
      onCondition: "audit_scheduling.id = audit_execution.scheduled_audit",
    },
    {
      type: "left",
      targetTable: "audit_template",
      onCondition: "audit_template.id = audit_execution.audit_protocol",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = audit_execution.organization",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = audit_execution.ddrm_id",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  /**Record of all alert */
  const auditExecutionDataFetchQuery = `SELECT audit_execution.*, 
    audit_scheduling.planned_start_date, 
    audit_scheduling.planned_end_date, 
    audit_execution.created_by as created_by_id,
    audit_template.name as audit_template_name,
    audit_scheduling.name as scheduled_audit_name,
    repository.url as report,
    users.name as created_by_name,audit_execution.organization,organization.name as organization_name
    FROM audit_execution  
    ${joinsRecord} 
    where audit_execution.deleted = 0 ${searchCondition} ${condition}`;

  const [auditExecutionDataFetch] = await db.query(auditExecutionDataFetchQuery);

  /**Count all Audit Execution */
  const totalRecord = await countQueryCondition(auditExecutionDataFetchQuery);

  return sendResponse(res, 200, auditExecutionDataFetch, totalRecord);
};

/**Function to delete a specific Audit Execution */
export const deleteAuditExecution = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("audit_execution", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(req.user.sessionid, "delete", "Audit Execution", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

/** function for adding audit record using audit template */
export const saveAuditRecord = async (req, res) => {
  const { id, organization, department, sections, audit_execution } = req.body;
  /** Check for audit execution is already executed or not  (by checking status)*/

  const [checkAuditExecution] = await db.query("SELECT id, status FROM audit_scheduling WHERE id = ? AND deleted = 0", [
    audit_execution,
  ]);

  if (checkAuditExecution[0]?.status == "Executed") {
    return sendResponse(res, 400, "Audit Execution is already executed");
  }

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization = await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  const points = await pointsCalculate(sections);
  req.body.total_point = points.totalPoints;
  req.body.earned_points = points.pointsEarned;
  req.body.audit_record = req.body.sections;
  if (id) {
    const { query, values } = updateQueryBuilder(AuditRecord, req.body);
    await db.query(query, values);
    return sendResponse(res, 200, "Record updated successfully");
  }
  req.body.created_by = req.user.sessionid;
  const { query, values } = createQueryBuilder(AuditRecord, req.body);
  const [createAuditRecord] = await db.query(query, values);
  if (createAuditRecord.insertId > 0) {
    /** update status of audit execution  */
    const updateAuditExecutionQuery = `UPDATE audit_scheduling SET status=? , audit_record=? WHERE id=? AND deleted = 0`;
    await db.query(updateAuditExecutionQuery, ["Executed", createAuditRecord.insertId, audit_execution]);
    /**Insert record for activity log */
    await insertActivityLog(req.user.sessionid, "create", "Audit Execution Record", createAuditRecord.insertId);
    return sendResponse(res, 200, "Record created successfully");
  } else {
    return sendResponse(res, 200, "Record not created");
  }
};

/**Function to view all Audit Record */
export const viewAllAuditRecord = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "audit_record",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    columnName: "audit_execution",
  });

  const searchTableName = [
    "audit_record.name",
    "audit_record.description",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = audit_record.created_by",
    },
    {
      type: "left",
      targetTable: "audit_execution",
      onCondition: "audit_execution.id = audit_record.audit_execution",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = audit_record.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  /**Record of all Audit Execution */
  const auditExecutionDataFetchQuery = `SELECT 
    audit_record.*, audit_record.created_by as created_by_id, users.name as created_by_name,organization.name as organization_name
    FROM audit_record  
    ${joinsRecord} 
    where audit_record.deleted = 0 ${searchCondition} ${condition}`;

  const [auditExecutionDataFetch] = await db.query(auditExecutionDataFetchQuery);
  auditExecutionDataFetch.forEach((element) => {
    element.audit_record = JSON.parse(element.audit_record);
  });
  /**Count all Audit Execution */
  const totalRecord = await countQueryCondition(auditExecutionDataFetchQuery);

  return sendResponse(res, 200, auditExecutionDataFetch, totalRecord);
};
