import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  createQueryBuilder,
  whereCondition,
  makeJoins,
  countQueryCondition,
  searchConditionRecord,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  decodeAndParseFields,
  encodeAndStringifyFields,
  insertActivityLog,
  getRecord,
  getBcpComment,
  getUserListByIds,
  getEmployeeDetails,
  uploadFile,
  processUploadDocuments,
  uniqueIdGenerator,
} from "../helper/general.js";
import BcpManagement from "../sequelize/BcpManagementSchema.js";
import BcpTesting from "../sequelize/BcpTestingSchema.js";
import { updateQueryBuilder } from "../helper/queryBuilder.js";
import BcpManagementComment from "../sequelize/BcpManagementCommentSchema.js";
import BcpDocs from "../sequelize/BcpDocsSchema.js";

export const createUpdateBcpManagement = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];
  const {
    id,
    department,
    reviewer,
    testing_start_date,
    testing_cycles,
    sidebar_id,
  } = req.body;
  req.body.reviewer = reviewer?.map((e) => e?.user_id);
  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }

  const documents = await processUploadDocuments(req, sidebar_id);

  if (id && documents.length) {
    const deleteQuery = `UPDATE bcp_docs SET deleted = 1 where deleted = 0 AND bcp_id = ?`;
    await db.query(deleteQuery, [id]);
  }

  const allFiles = [];
  documents.forEach((doc) => {
    allFiles.push(doc.filepath);
  });

  req.body = await encodeAndStringifyFields(req.body);

  if (!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "BCPM",
      "bcp_management",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  const { query, values } = id
    ? updateQueryBuilder(BcpManagement, req.body)
    : createQueryBuilder(BcpManagement, req.body);
  const [result] = await db.query(query, values);

  if (documents || Array.isArray(documents)) {
    // Iterate through uploaded files
    for (const file of documents) {
      const title = file.title; // Use file name as title
      const ddrm_id = file?.ddrm_id;
      const id_doc = file.id;
      // Move the file to the public folder
      // Save file details for database
      const { query: docQuery, values: docValues } = createQueryBuilder(
        BcpDocs,
        {
          doc_no: id_doc,
          bcp_id: id ? id : result.insertId,
          title,
          ddrm_id,
        }
      );

      await db.query(docQuery, docValues);
    }
  }
  await insertActivityLog(
    req.user.sessionid,
    status,
    "BCP Management",
    `This user ${status} a new BCP Management Record`
  );

  // Initial dates for first record in the cycle
  let planned_start_date = new Date(testing_start_date);
  let data = {
    bcp_management_id: result.insertId ? result.insertId : id,
    organization: req.body.organization,
    department: req.body.department,
    schedule_type: req.body.schedule_type,
    planned_testing_date: planned_start_date,
    testing_frequency: req.body.testing_frequency,
    testing_cycles: testing_cycles,
  };
  const type = id ? "update" : "create";
  await createUpdateBcpTestingFunction(data, type);
  return sendResponse(res, 200, `Record ${status} successfully`);
};

const allEmployeeFields = [
  "crisis_management_team_leader",
  "deputy_crisis_management_team_leader",
  "emergency_response_coordinator",
  "technical_recovery_lead",
  "communication_officer",
  "legal_and_compliance_officer",
  "hr_representative",
  "operation_recovery_lead",
  "finance_and_administration_lead",
  "facilities_and_physical_infrastructure",
  "supply_chain_and_procurement_lead",
  "external_liaison_officer",
  "security_lead",
  "risk_management_officer",
  "crisis_documentation_officer",
];

export const getBcpManagement = async (req, res) => {
  const { id } = req.params;

  // Prepare conditions
  const condition = await whereCondition({
    table: "bcp_management",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "bcp_management.plan_name",
    "bcp_management.plan_reference_number",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );
  let BcpComments = [];
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = bcp_management.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = bcp_management.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = bcp_management.department",
    },
    {
      type: "left",
      targetTable: "bcp_requirement_analysis",
      onCondition:
        "bcp_requirement_analysis.id = bcp_management.bcp_requirement_analysis_id",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = bcp_requirement_analysis.location",
    },
    {
      type: "left",
      targetTable: "users as approver_user",
      onCondition: "approver_user.id = bcp_management.approver",
    },
    {
      type: "left",
      targetTable: "roles as approver_role",
      onCondition: "approver_role.id = approver_user.role",
    },
    {
      type: "left",
      targetTable: "department as approver_department",
      onCondition: "approver_department.id = approver_user.department",
    },
  ];
  const joinCondition = await makeJoins(joins);

  // Base query for BCP Management
  const bcpManagementQuery = `
      SELECT 
        bcp_management.*,
        bcp_requirement_analysis.name as activity_name,
        bcp_requirement_analysis.location as activity_location,
        location.name as location_name,
        bcp_requirement_analysis.aggregate_activity_score,
        bcp_requirement_analysis.tasks,
        bcp_requirement_analysis.rto_value AS activity_recovery_time_objective,
        bcp_requirement_analysis.mtpod_value AS activity_maximum_tolerable,
        CONCAT(approver_user.name , ' ' , approver_user.surname) as approver_name,
        approver_user.profile as approver_profile,
        approver_role.name as approver_role,
        approver_department.name as approver_department, 
        organization.name AS organization_name, 
        department.name AS department_name, 
        users.name AS created_by_name, 
        users.surname AS created_by_surname, 
        users.profile AS created_by_profile 
      FROM bcp_management 
      ${joinCondition} 
      WHERE bcp_management.deleted = 0 
      ${searchCondition} 
      ${condition}
    `;

  // Execute base query
  let [BcpManagement] = await db.query(bcpManagementQuery);
  BcpManagement = await decodeAndParseFields(BcpManagement);
  for (let i = 0; i < BcpManagement.length; i++) {
    let { reviewer } = BcpManagement[i];
    reviewer = reviewer.filter((e) => e);
    let user = [];
    if (reviewer.length > 0) {
      [user] =
        await db.query(`SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
        LEFT JOIN department ON department.id  = users.department WHERE users.id IN(${reviewer})`);
    }

    BcpManagement[i].reviewer = user;
    const plan_external_emergency_numbers_arr =
      BcpManagement[i]?.plan_external_emergency_numbers || [];

    const { userIds, contactMap } = plan_external_emergency_numbers_arr.reduce(
      (acc, e) => {
        if (e.user_id) {
          acc.userIds.push(e.user_id);
          acc.contactMap[e.user_id] = e.contact_number; // Store contact_number mapped to user_id
        }
        return acc;
      },
      { userIds: [], contactMap: {} }
    );

    BcpManagement[i].plan_external_emergency_numbers = userIds.length
      ? (await getUserListByIds(userIds)).map((user) => ({
          ...user,
          contact_number: contactMap[user.user_id] || null, // Merge contact_number
        }))
      : [];

    const id = BcpManagement[i].id;

    const [files] = await db.query(
      `SELECT bcp_docs.*, repository.url as file, document_name as title FROM bcp_docs LEFT JOIN repository ON repository.id = bcp_docs.ddrm_id LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE bcp_docs.deleted = 0 AND bcp_id = ${id}`
    );

    BcpManagement[i].upload_documents = files.length
      ? files
      : [{ title: "", ddrm_id: null, file: null }];
  }
  if (id) BcpComments = await getBcpComment(id);
  BcpManagement = BcpManagement.map((item) => {
    let isCommented = false;
    return {
      ...item,
      comments: BcpComments.filter((comment) => {
        if (req.user.sessionid === comment.created_by) {
          isCommented = true;
        }
        if (
          req.user.sessionid === item.policy_owner ||
          req.user.sessionid === item.policy_approver ||
          req.user.sessionid === item.created_by ||
          req.user.isSuperAdmin == 1
        ) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        } else if (req.user.sessionid === comment.created_by) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        }
      }),
      isCommented,
    };
  });

  // Fetch employee data in a single query
  // if (employeeFieldIds.length > 0) {
  //   const uniqueIds = [...new Set(employeeFieldIds)];
  //   const leaderQuery = `
  //       SELECT
  //         users.id AS user_id,
  //         CONCAT(users.name, ' ', users.surname) AS name,
  //         users.profile AS profile,
  //         users.phone AS phone_number,
  //         department.name AS department,
  //         roles.name AS role,
  //         users.email AS email_address
  //       FROM users
  //       LEFT JOIN roles ON roles.id = users.role
  //       LEFT JOIN department ON department.id = users.department
  //       WHERE users.id IN (${uniqueIds.join(",")})
  //     `;

  //   const [leaderData] = await db.query(leaderQuery);

  //   // Map employee data by user_id
  //   const leaderDataMap = leaderData.reduce((map, leader) => {
  //     map[leader.user_id] = leader;
  //     return map;
  //   }, {});

  //   // Merge leader data back into BcpManagement
  //   for (let i = 0; i < BcpManagement.length; i++) {
  //     const item = BcpManagement[i];
  //     let all_members = [];
  //     for (let key of allEmployeeFields) {
  //       if (item[key] && leaderDataMap[item[key]]) {
  //         const leader = leaderDataMap[item[key]];
  //         const memberDetails = {
  //           [`${key}_name`]: leader.name || null,
  //           [`${key}_phone_number`]: leader.phone_number || null,
  //           [`${key}_department`]: leader.department || null,
  //           [`${key}_role`]: leader.role || null,
  //           [`${key}_email_address`]: leader.email_address || null,
  //           [`${key}_profile`]: leader.profile || null,
  //         };
  //         BcpManagement[i] = {
  //           ...BcpManagement[i],
  //           ...memberDetails,
  //         };
  //         all_members.push({value: leader.user_id, title: leader.name});
  //       }
  //     }
  //     BcpManagement[i].all_members = all_members;
  //   }
  // }

  const { leaderDataMap, all_members, bcpManagementData } =
    await getEmployeeDetails(BcpManagement);

  // Total records count
  const totalRecord = await countQueryCondition(bcpManagementQuery);

  // Send response
  return sendResponse(res, 200, bcpManagementData, totalRecord);
};
export const deleteBcpManagement = async (req, res) => {
  const { id } = req.params;
  const [record] = await getRecord("bcp_management", "id", id);
  if (record?.status === "approved") {
    return sendResponse(res, 400, "Approved record cannot be deleted.");
  }
  await deleteRecord("bcp_management", id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "bcp_management",
    `This user deleted a new bcp_management Record for organization ${id}`
  );
  return sendResponse(res, 200, "Record deleted successfully");
};

/**Function to create comments on bcp management */
export const bcpComment = async (req, res) => {
  let { status, approver, bcp_management_id, comments } = req.body;
  const [record] = await getRecord("bcp_management", "id", bcp_management_id);
  if (!record) {
    return sendResponse(res, 400, "Record not found");
  }
  approver = approver ? approver : record?.approver;
  // comment is required
  if (status == "rejected" && !comments) {
    return sendResponse(res, 400, "Comments are required");
  }
  const reviewerList = JSON.parse(record?.reviewer || "[]");
  // permission check for approver or reviewer
  console.log(
    record?.approver,
    "record?.approver",
    req.user.sessionid,
    reviewerList,
    req.user.sessionid
  );
  console.log(
    record?.approver == req.user.sessionid,
    !reviewerList.includes(req.user.sessionid)
  );
  if (
    record?.approver != req.user.sessionid &&
    !reviewerList.includes(req.user.sessionid) &&
    req.user.isSuperAdmin != 1
  ) {
    return sendResponse(res, 400, "Access Denied");
  } else {
    let newStatus =
      status == "rejected"
        ? "rejected"
        : record.approver == req.user.sessionid
        ? "approved"
        : "approval";

    // if comment added by super admin and status is approved then change status to approved
    if (req.user.isSuperAdmin && status == "approved") {
      newStatus = "approved";
    }
    await db.query(
      `Update bcp_management set approver = ? , status = '${newStatus}' where id = ?`,
      [approver, req.body.bcp_management_id]
    );
  }
  req.body.created_by = req.user.sessionid;
  const { query, values } = createQueryBuilder(BcpManagementComment, req.body);
  const insertedData = await db.query(query, values);
  return sendResponse(res, 200, "Status Updated Successfully");
};

export const getBcpManagementApprovalWorkflow = async (req, res) => {
  const { id } = req.params;

  // Prepare conditions
  const condition = await whereCondition({
    table: "bcp_management",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "bcp_management.plan_name",
    "bcp_management.plan_reference_number",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );
  let BcpComments = [];
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = bcp_management.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = bcp_management.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = bcp_management.department",
    },
    {
      type: "left",
      targetTable: "bcp_requirement_analysis",
      onCondition:
        "bcp_requirement_analysis.id = bcp_management.bcp_requirement_analysis_id",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = bcp_requirement_analysis.location",
    },
    {
      type: "left",
      targetTable: "users as approver_user",
      onCondition: "approver_user.id = bcp_management.approver",
    },
    {
      type: "left",
      targetTable: "roles as approver_role",
      onCondition: "approver_role.id = approver_user.role",
    },
    {
      type: "left",
      targetTable: "department as approver_department",
      onCondition: "approver_department.id = approver_user.department",
    },
  ];
  const joinCondition = await makeJoins(joins);
  const approvalCondition = req.user.isSuperAdmin
    ? ""
    : `AND ((JSON_CONTAINS(bcp_management.reviewer, '${req.user.sessionid}')) OR  bcp_management.approver = ${req.user.sessionid})`;

  // Base query for BCP Management
  const bcpManagementQuery = `
  SELECT 
  bcp_management.*,
        bcp_requirement_analysis.name as activity_name,
        bcp_requirement_analysis.location as activity_location,
        location.name as location_name,
        bcp_requirement_analysis.aggregate_activity_score,
        bcp_requirement_analysis.tasks,
        bcp_requirement_analysis.rto_value AS activity_recovery_time_objective,
        bcp_requirement_analysis.mtpod_value AS activity_maximum_tolerable,
        CONCAT(approver_user.name , ' ' , approver_user.surname) as approver_name,
        approver_user.profile as approver_profile,
        approver_role.name as approver_role,
        approver_department.name as approver_department, 
        organization.name AS organization_name, 
        department.name AS department_name, 
        users.name AS created_by_name, 
        users.surname AS created_by_surname, 
        users.profile AS created_by_profile 
        FROM bcp_management 
        ${joinCondition} 
        WHERE bcp_management.deleted = 0 
        ${approvalCondition}
        ${searchCondition} 
        ${condition}
        `;

  // Execute base query
  let [BcpManagement] = await db.query(bcpManagementQuery);
  BcpManagement = await decodeAndParseFields(BcpManagement);
  for (let i = 0; i < BcpManagement.length; i++) {
    let { reviewer } = BcpManagement[i];
    reviewer = reviewer.filter((e) => e);
    let user = [];
    if (reviewer.length > 0) {
      [user] =
        await db.query(`SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
        LEFT JOIN department ON department.id  = users.department WHERE users.id IN(${reviewer})`);
    }

    BcpManagement[i].reviewer = user;
    const plan_external_emergency_numbers_arr =
      BcpManagement[i]?.plan_external_emergency_numbers || [];
    const plan_external_emergency_numbers_id =
      plan_external_emergency_numbers_arr
        ?.filter((e) => e.emergency_service)
        .map((e) => e.emergency_service);
    BcpManagement[i].plan_external_emergency_numbers =
      plan_external_emergency_numbers_id?.length
        ? await getUserListByIds(plan_external_emergency_numbers_id)
        : [];
  }
  if (id) BcpComments = await getBcpComment(id);
  BcpManagement = BcpManagement.map((item) => {
    let isCommented = false;
    return {
      ...item,
      comments: BcpComments.filter((comment) => {
        if (req.user.sessionid === comment.created_by) {
          isCommented = true;
        }
        if (
          req.user.sessionid === item.policy_owner ||
          req.user.sessionid === item.policy_approver ||
          req.user.sessionid === item.created_by ||
          req.user.isSuperAdmin == 1
        ) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        } else if (req.user.sessionid === comment.created_by) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        }
      }),
      isCommented,
    };
  });

  // Batch process employee fields
  const employeeFieldIds = [];
  const employeeFieldMap = {};

  for (let item of BcpManagement) {
    for (let key of allEmployeeFields) {
      if (item[key]) {
        employeeFieldIds.push(item[key]);
        employeeFieldMap[item[key]] = { key, id: item[key] };
      }
    }
  }

  // Fetch employee data in a single query
  if (employeeFieldIds.length > 0) {
    const uniqueIds = [...new Set(employeeFieldIds)];
    const leaderQuery = `
        SELECT 
          users.id AS user_id,
          CONCAT(users.name, ' ', users.surname) AS name,
          users.profile AS profile,
          users.phone AS phone_number,
          department.name AS department,
          roles.name AS role,
          users.email AS email_address 
        FROM users 
        LEFT JOIN roles ON roles.id = users.role 
        LEFT JOIN department ON department.id = users.department 
        WHERE users.id IN (${uniqueIds.join(",")})
      `;

    const [leaderData] = await db.query(leaderQuery);

    // Map employee data by user_id
    const leaderDataMap = leaderData.reduce((map, leader) => {
      map[leader.user_id] = leader;
      return map;
    }, {});

    // Merge leader data back into BcpManagement
    for (let i = 0; i < BcpManagement.length; i++) {
      const item = BcpManagement[i];
      for (let key of allEmployeeFields) {
        if (item[key] && leaderDataMap[item[key]]) {
          const leader = leaderDataMap[item[key]];
          BcpManagement[i] = {
            ...BcpManagement[i],
            [`${key}_name`]: leader.name || null,
            [`${key}_phone_number`]: leader.phone_number || null,
            [`${key}_department`]: leader.department || null,
            [`${key}_role`]: leader.role || null,
            [`${key}_email_address`]: leader.email_address || null,
            [`${key}_profile`]: leader.profile || null,
          };
        }
      }
    }
  }

  // Total records count
  const totalRecord = await countQueryCondition(bcpManagementQuery);

  // Send response
  return sendResponse(res, 200, BcpManagement, totalRecord);
};

// Function to check user permissions
async function checkUserPermissions(
  planReviewer,
  planApproval,
  sessionid,
  reviewPermission
) {
  if (planReviewer.includes(sessionid) && reviewPermission) {
    return true;
  }
  if (planApproval === sessionid) {
    return true;
  }
  return false;
}

async function createUpdateBcpTestingFunction(data, type) {
  // console.log("data", "type", data, type);
  let {
    id,
    testing_cycles,
    testing_frequency,
    planned_testing_date,
    bcp_management_id,
  } = data;
  if (type === "create") {
    for (let i = 0; i < testing_cycles; i++) {
      // Set dates for this cycle's testing entry
      data.planned_testing_date = planned_testing_date
        .toISOString()
        .split("T")[0];
      // add unique_id to data
      data.unique_id = await uniqueIdGenerator(
        data.organization,
        data.department,
        "BCPT",
        "bcp_testing",
        "unique_id",
        "unique_id"
      );
      const { query, values } = createQueryBuilder(BcpTesting, data);
      const [result] = await db.query(query, values);

      // Calculate the next cycle's dates based on `when` frequency
      switch (testing_frequency) {
        case "Daily":
          planned_testing_date.setDate(planned_testing_date.getDate() + 1);
          break;
        case "Weekly":
          planned_testing_date.setDate(planned_testing_date.getDate() + 7);
          break;
        case "Bi-Monthly":
          planned_testing_date.setDate(planned_testing_date.getDate() + 15);
          break;
        case "Monthly":
          planned_testing_date.setMonth(planned_testing_date.getMonth() + 1);
          break;
        case "Quarterly":
          planned_testing_date.setMonth(planned_testing_date.getMonth() + 3);
          break;
        case "Bi-Annually":
          planned_testing_date.setMonth(planned_testing_date.getMonth() + 6);
          break;
        case "Annually":
          planned_testing_date.setFullYear(
            planned_testing_date.getFullYear() + 1
          );
          break;
        default:
          break;
      }
    }
  } else if (type === "update") {
    // fetch the existing data in Bcp testing with this bcp_management_id;
    const [existingData] = await db.query(
      `SELECT * FROM bcp_testing WHERE bcp_management_id = ?`,
      [bcp_management_id]
    );

    for (let i = 0; i < testing_cycles || i < existingData.length; i++) {
      // Set dates for this cycle's testing entry
      data.planned_testing_date = planned_testing_date
        .toISOString()
        .split("T")[0];
      const { query, values } = existingData[i]
        ? updateQueryBuilder(BcpTesting, { ...data, id: existingData[i].id })
        : createQueryBuilder(BcpTesting, data);
      const [result] = await db.query(query, values);
      // Calculate the next cycle's dates based on `when` frequency
      switch (testing_frequency) {
        case "Daily":
          planned_testing_date.setDate(planned_testing_date.getDate() + 1);
          break;
        case "Weekly":
          planned_testing_date.setDate(planned_testing_date.getDate() + 7);
          break;
        case "Bi-Monthly":
          planned_testing_date.setDate(planned_testing_date.getDate() + 15);
          break;
        case "Monthly":
          planned_testing_date.setMonth(planned_testing_date.getMonth() + 1);
          break;
        case "Quarterly":
          planned_testing_date.setMonth(planned_testing_date.getMonth() + 3);
          break;
        case "Bi-Annually":
          planned_testing_date.setMonth(planned_testing_date.getMonth() + 6);
          break;
        case "Annually":
          planned_testing_date.setFullYear(
            planned_testing_date.getFullYear() + 1
          );
          break;
        default:
          break;
      }
    }
  }
}
