import {
  countQueryCondition,
  decodeAndParseFields,
  deleteFile,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  whereCondition,
} from "../helper/general.js";
import ContractorInvestigation from "../sequelize/ContractorInvestigationSchema.js";
import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import ContractorIncident from "../sequelize/ContractorIncidentSchema.js";
import ContractorInvestigationDocs from "../sequelize/ContractorInvestigationDocsSchema.js"; // for sync purpose only not to be removed
/** Function to create and update contractor investigation  */
export const createUpdateContractorInvestigation = async (req, res) => {
  let { id, department, incident_id } = req.body;
  let status = id ? "update" : "create";

  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0]?.organization;
  }

  req.body.is_done = true;

  req.body = await encodeAndStringifyFields(req.body);

  let result = id
    ? await ContractorInvestigation.update(req.body, {
        where: { id },
      })
    : await ContractorInvestigation.create(req.body);

  await ContractorIncident.update(
    {
      is_done: true,
    },
    {
      where: { id: incident_id },
    }
  );
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Stakeholder Identification",
    id ? id : result.id
  );

  return sendResponse(res, 200, `Record ${status} successfully`);
};

/** Function to get contractor investigation */
export const getContractorInvestigation = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "contractor_investigation",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "contractor_investigation.incident_name",
    "contractor_investigation.initial_incident_severity",
    "contractor_investigation.incident_category",
    "contractor_investigation.incident_description",
    "contractor_investigation.incident_location",
    "contractor_investigation.role_in_incident",
    "contractor_investigation.investigator_name",
    "contractor_investigation.specialty",
    "contractor_investigation.organization_external",
    "contractor_investigation.person_involved_internal_external",
    "contractor_investigation.involved_person",
    "contractor_investigation.involved_role",
    "contractor_investigation.involved_department",
    "contractor_investigation.involved_role_in_incident",
    "contractor_investigation.investigation_enquiry_outcome",
    "contractor_investigation.recall_required",
    "contractor_investigation.applicable_sop_created",
    "contractor_investigation.affected_employees_retrained",
    "contractor_investigation.recommended_can_be_applied",
    "contractor_investigation.role_in_incident_external",
    "contractor_investigation.investigation_organization",
    "contractor_investigation.assets_damage_assets_property_type",
    "contractor_investigation.assets_currency",
    "contractor_investigation.assets_vehicle_incident_type",
    "contractor_investigation.quality_incident_type",
    "contractor_investigation.customer_impact",
    "contractor_investigation.product_impact",
    "contractor_investigation.quality_currency",
    "contractor_investigation.quality_vehicle_incident_type",
    "contractor_investigation.environment_consequence_type",
    "contractor_investigation.origin",
    "contractor_investigation.injured_person",
    "contractor_investigation.injury_classification",
    "contractor_investigation.part_body_affected",
    "contractor_investigation.reportable_classification",
    "contractor_investigation.exposed_person",
    "contractor_investigation.diseases_illnesses_type",
    "contractor_investigation.exposure_route",
    "contractor_investigation.security_incident_type",
    "contractor_investigation.security_currency",
    "contractor_investigation.line_manager_type",
    "(created_user.name , ' ' , created_user.surname)",
    "organization.name",
  ];

  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      targetTable: "incident_category",
      onCondition:
        "contractor_investigation.incident_category = incident_category.id",
      type: "LEFT",
    },
    {
      targetTable: "department",
      onCondition: "contractor_investigation.department = department.id",
      type: "LEFT",
    },
    {
      targetTable: "organization",
      onCondition: "contractor_investigation.organization = organization.id",
      type: "LEFT",
    },
    {
      targetTable: "users as employeeUser",
      onCondition: "contractor_investigation.employee = employeeUser.id",
      type: "LEFT",
    },

    {
      targetTable: "users as created_user",
      onCondition: "contractor_investigation.created_by = created_user.id",
      type: "LEFT",
    },
    {
      type: "LEFT",
      targetTable: "department AS employee_user_department",
      onCondition: "employeeUser.department = employee_user_department.id",
    },
    {
      type: "LEFT",
      targetTable: "roles AS employee_user_role",
      onCondition: "employeeUser.role = employee_user_role.id",
    },
    {
      type: "left",
      targetTable: "incident",
      onCondition: "incident.id = contractor_investigation.incident_id",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  const contractorInvestigationGetQuery = `
    SELECT contractor_investigation.*, 
           incident.incident_title AS incident_title_name,
           employeeUser.name AS employee_name, 
           employeeUser.role AS employee_role,
           employeeUser.department AS employee_department,
           employee_user_department.name AS employee_department_name,
           employee_user_role.name AS employee_role_name,
           created_user.name AS created_by_name, 
           created_user.surname AS created_by_surname, 
           organization.name AS organization_name,
           department.name AS department_name 
    FROM contractor_investigation 
    ${joinsRecord}
    WHERE contractor_investigation.deleted = 0 ${searchCondition} ${condition}`;

  let [getContractorInvestigation] = await db.query(
    contractorInvestigationGetQuery
  );

  const totalCount = await countQueryCondition(contractorInvestigationGetQuery);
  getContractorInvestigation = await decodeAndParseFields(
    getContractorInvestigation
  );

  // console.log(getContractorInvestigation);
  return sendResponse(res, 200, getContractorInvestigation, totalCount);
};

/**Function to delete a specific Investigation */
export const deleteContractorInvestigation = async (req, res) => {
  const { id } = req.params;
  const [filePaths] = await db.query(
    "SELECT * FROM contractor_investigation_docs WHERE contractor_investigation_id = ?",
    [id]
  );

  if (filePaths?.length > 0) {
    for (const file of filePaths) {
      if (file?.file !== undefined) {
        await deleteFile(file?.file);
      }
    }
  }
  await db.query(
    "UPDATE contractor_investigation_docs SET deleted = 1 WHERE investigation_id = ?",
    [id]
  );
  const deleteRecordQuery = await deleteRecord("contractor_investigation", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "ContractorInvestigation",
      id
    );
    return sendResponse(
      res,
      200,
      "Contractor Investigation Report Record deleted Successfully"
    );
  }
};
