import db from "../db-config.js";
import {
  createQueryBuilder,
  insertActivityLog,
  makeJoins,
  whereCondition,
  searchConditionRecord,
  countQueryCondition,
  updateQueryBuilder,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  decodeAndParseFields,
  deleteRecord,
} from "../helper/general.js";
import ContractorRiskRegister from "../sequelize/ContractorRiskRegisterSchema.js";
import { sendResponse } from "../helper/wrapper.js";

export const createUpdateContractorRiskRegister = async (req, res) => {
  const { id, department } = req.body;
  let status = id ? "Updated" : "Created";

  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }

  req.body = await encodeAndStringifyFields(req.body);

  const { query, values } = id
    ? updateQueryBuilder(ContractorRiskRegister, req.body)
    : createQueryBuilder(ContractorRiskRegister, req.body);

  await db.query(query, values);

  await insertActivityLog(
    req.user.sessionid,
    status,
    "Contractor Risk Assessment",
    id
  );

  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getContractorRiskRegister = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "contractor_risk_register",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "contractor_risk_register.risk_name",
    "contractor_risk_register.risk_description",
    "contractor_risk_register.impact_description",
    "contractor_risk_register.causes_description",
    "contractor_risk_register.inherent_likelihood",
    "contractor_risk_register.inherent_consequence",
    "contractor_risk_register.control_name",
    "contractor_risk_register.control_design_intent",
    "contractor_risk_register.control_effectiveness",
    "contractor_risk_register.control_description",
    "CONCAT(createdUser.name , ' ' , createdUser.surname)",
    "organization.name",
    "department.name",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = contractor_risk_register.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = contractor_risk_register.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = contractor_risk_register.department",
    },
    // {
    //   type: "left",
    //   targetTable: "users AS riskOwner",
    //   onCondition: "riskOwner.id = contractor_risk_register.risk_owner",
    // },
    // {
    //   type: "left",
    //   targetTable: "users AS controlOwner",
    //   onCondition: "controlOwner.id = contractor_risk_register.control_owner",
    // },
  ];

  const joinCondition = await makeJoins(joins);

  const contractorRiskAssessmentQuery = `SELECT contractor_risk_register.* , organization.name AS organization_name , department.name AS department_name , CONCAT(createdUser.name , ' ' , createdUser.surname) AS created_by_name, createdUser.profile AS created_by_profile    FROM contractor_risk_register ${joinCondition} WHERE contractor_risk_register.deleted = 0 ${searchCondition} ${condition} `;
  let [contractorRiskAssessment] = await db.query(
    contractorRiskAssessmentQuery
  );
  contractorRiskAssessment = await decodeAndParseFields(
    contractorRiskAssessment
  );
  const totalRecord = countQueryCondition(contractorRiskAssessmentQuery);

  return sendResponse(res, 200, contractorRiskAssessment, totalRecord);
};

export const deleteContractorRiskRegister = async (req, res) => {
  const { id } = req.params;

  await deleteRecord(ContractorRiskRegister, id);

  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Contractor Risk Assessment",
    id
  );

  return sendResponse(res, 200, "Record deleted successfully");
};
