import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import FeedbackCollectionMethod from "../sequelize/FeedbackCollectionMethodSchema.js";

export const createUpdateFeedbackCollection = async (req, res) => {
  const { id } = req.body;

  let status = id ? "Updated" : "Created";

  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  req.body = await encodeAndStringifyFields(req.body);

  const { query, values } = id
    ? updateQueryBuilder(FeedbackCollectionMethod, req.body)
    : createQueryBuilder(FeedbackCollectionMethod, req.body);
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Feedback Collection Identification",
    id
  );
  await db.query(query, values);
  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getFeedbackCollectionMethod = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "feedback_collection",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "feedback_collection.method_name",
    "feedback_collection.method_type",
    "feedback_collection.collection_method_description",
    "feedback_collection.target_stakeholder",
    "feedback_collection.frequency",
    "organization.name",
    "department.name",
    "CONCAT(users.name , ' ' , users.surname)",
    "CONCAT(responsible.name , ' ' , responsible.surname)",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = feedback_collection.created_by",
    },
    {
      type: "left",
      targetTable: "users AS responsible",
      onCondition: "responsible.id = feedback_collection.responsible_person",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = feedback_collection.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = feedback_collection.department",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const feedbackCollectionMethodQuery = `SELECT feedback_collection.* , users.name AS created_by_name , users.surname AS created_by_surname , users.profile AS created_by_profile , organization.name AS organization_name , department.name  AS department_name , CONCAT(responsible.name , ' ' , responsible.surname) AS responsible_person_name , responsible.profile AS responsible_person_profile FROM feedback_collection ${joinCondition} WHERE feedback_collection.deleted = 0 ${searchCondition} ${condition}`;

  let [feedbackCollectionMethod] = await db.query(
    feedbackCollectionMethodQuery
  );

  feedbackCollectionMethod = await decodeAndParseFields(
    feedbackCollectionMethod
  );

  const total = await countQueryCondition(feedbackCollectionMethodQuery);
  return sendResponse(res, 200, feedbackCollectionMethod, total);
};

export const deleteFeedbackCollectionMethod = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(FeedbackCollectionMethod, id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Feedback Collection Method",
    id
  );
  return sendResponse(res, 200, "Record deleted successfully");
};
