import moment from "moment";
import db from "../db-config.js";
import {
  insertActivityLog,
  getOrganizationAccordingToDepartment,
  whereCondition,
  makeJoins,
  countQueryCondition,
  deleteRecord,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  getRecord,
  decodeAndParseFields,
  saveEquipmentHistory,
  uniqueIdGenerator,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import FireEquipmentMaintenanceInspection from "../sequelize/FireEquipmentMaintenanceInspectionSchema.js";
import { createUpdateActionFunction, getCustomActionFunction } from "./customActionCreationController.js";

const excludedFields = ["created_at", "updated_at", "created_by", "updated_by"];

/**Function to create/update Fire Equipment MaintenanceInspection   */
export const FireEquipmentMaintenanceInspectionCreateUpdate = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];

  const { id, organization, department, action_taken, sidebar_id = 313, task_data } = req.body;

  let organizationId = organization;

  if (department) {
    const recordAccordingToOrganization = await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }
  /**Check record if organization is not coming then fetch organization according to department */

  const actionTaken = [];
  for (let i = 0; i < task_data.length; i++) {
    let taskData = task_data[i];
    taskData.approved = false;
    const data = {
      task_data: [taskData],
      department,
      files: req.files,
      user: req.user,
      id: action_taken[i],
      organization: organizationId,
      sidebar_id,
    };

    if (data?.task_data && data?.task_data[0] && data?.task_data[0]?.role) {
      const { status, arr } = await createUpdateActionFunction(data, req);
      if (status) {
        const action_taken = arr[0].id;
        actionTaken.push(action_taken);
      }
    }
  }

  req.body.action_taken = actionTaken;

  // req.body.scheduled_status = req.body.scheduled_status ? req.body.scheduled_status : "Pending";
  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;

    const [oldData] = await db.query(`SELECT * FROM fire_equipment_maintenance_inspection WHERE id = ?`, [id]);

    req.body.inspection_date = moment(req.body.inspection_date).isValid() ? req.body.inspection_date : null;
    req.body.next_maintenance_date = moment(req.body.next_maintenance_date).isValid()
      ? req.body.next_maintenance_date
      : null;
    req.body.last_maintenance_date = moment(req.body.last_maintenance_date).isValid()
      ? req.body.last_maintenance_date
      : null;

    /**Update Fire Equipment MaintenanceInspection Query */
    const { query, values } = updateQueryBuilder(FireEquipmentMaintenanceInspection, req.body);
    // return
    await db.query(query, values);

    const [newData] = await db.query(`SELECT * FROM fire_equipment_maintenance_inspection WHERE id = ?`, [id]);
    const changedFields = {};

    if (oldData[0].next_maintenance_date !== newData[0].next_maintenance_date) {
      changedFields.next_maintenance_date = newData[0].next_maintenance_date;
    }

    if (oldData[0].last_maintenance_date !== newData[0].last_maintenance_date) {
      changedFields.last_maintenance_date = newData[0].last_maintenance_date;
    }

    if (oldData[0].recording_type !== newData[0].recording_type) {
      changedFields.recording_type = newData[0].recording_type;
    }
    if (oldData[0]?.inspection_checkbox !== newData[0]?.inspection_checkbox) {
      changedFields.inspection_checkbox = newData[0].inspection_checkbox;
    }

    await saveEquipmentHistory(
      "RECORDED",
      department,
      organizationId,
      "Maintenance and Inspection Updated",
      req.user.sessionid,
      req.body.equipment_id,
      id,
      changedFields
    );
    /**Insert Activity  */
    await insertActivityLog(req.user.sessionid, "update", "FireEquipmentMaintenanceInspection", id);
    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const unique_id = await uniqueIdGenerator(
      organizationId,
      department,
      "MIS",
      "fire_equipment_maintenance_inspection",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
    req.body.created_by = req.user.sessionid;

    req.body.inspection_date = moment(req.body.inspection_date).isValid() ? req.body.inspection_date : null;
    req.body.next_maintenance_date = moment(req.body.next_maintenance_date).isValid()
      ? req.body.next_maintenance_date
      : null;
    req.body.last_maintenance_date = moment(req.body.last_maintenance_date).isValid()
      ? req.body.last_maintenance_date
      : null;

    /**Insert record for Fire Equipment MaintenanceInspection */
    const { query, values } = createQueryBuilder(FireEquipmentMaintenanceInspection, req.body);
    const createFireEquipmentMaintenanceInspection = await db.query(query, values);

    const changedFields = {
      last_maintenance_date: req.body.last_maintenance_date,
      next_maintenance_date: req.body.next_maintenance_date,
    };
    await saveEquipmentHistory(
      "SCHEDULED",
      department,
      organizationId,
      "Maintenance and Inspection Scheduled",
      req.user.sessionid,
      req.body.equipment_id,
      createFireEquipmentMaintenanceInspection.insertId,
      changedFields
    );
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Fire Equipment Maintenance Inspection",
      createFireEquipmentMaintenanceInspection.insertId
    );
    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single Fire Equipment Maintenance Inspection */
export const viewFireEquipmentMaintenanceInspection = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "fire_equipment_maintenance_inspection",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = ["CONCAT(users.name , ' ' , users.surname)", "organization.name"];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = fire_equipment_maintenance_inspection.created_by",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = fire_equipment_maintenance_inspection.location",
    },
    {
      type: "left",
      targetTable: "equipment",
      onCondition: "equipment.id = fire_equipment_maintenance_inspection.equipment_type",
    },
    {
      type: "left",
      targetTable: "department as responsible_department",
      onCondition: "responsible_department.id = fire_equipment_maintenance_inspection.responsible_department",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = fire_equipment_maintenance_inspection.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = fire_equipment_maintenance_inspection.organization",
    },
    {
      type: "left",
      targetTable: "currency",
      onCondition: "currency.id = fire_equipment_maintenance_inspection.currency",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all Fire Equipment MaintenanceInspection */
  const FireEquipmentMaintenanceInspectionQuery = `
  SELECT fire_equipment_maintenance_inspection.*, 
  location.name AS location_name, 
  department.name AS department_name, 
  responsible_department.name as responsible_department_name,  
  equipment.name AS equipment_type_name, 
  users.name AS created_by, 
  organization.name AS organization_name,
  currency.name AS currency_name 
      FROM fire_equipment_maintenance_inspection 
      ${joinsRecord} where fire_equipment_maintenance_inspection.deleted = 0  ${searchCondition} ${condition}`;
  let [FireEquipmentMaintenanceInspection] = await db.query(FireEquipmentMaintenanceInspectionQuery);
  FireEquipmentMaintenanceInspection = await decodeAndParseFields(FireEquipmentMaintenanceInspection);

  const actionTakenData = [];

  FireEquipmentMaintenanceInspection = await Promise.all(
    FireEquipmentMaintenanceInspection.map(async (record) => {
      const actionTaken = record.action_taken;
      if (actionTaken) {
        for (let record of actionTaken) {
          const data = {
            params: { id: record },
            user: req.user,
            query: {},
          };
          const [customActionData] = await getCustomActionFunction(data);
          actionTakenData.push(customActionData);
        }
        record.task_data = actionTakenData;
      }
      // console.log('record.task_data: ', record.task_data);
      // const taskData = record?.task_data;
      // if (Array.isArray(taskData) && taskData.length > 0) {
      //   taskData.map((item) => {
      //     item.file = item?.attachment || "";
      //     delete item.attachment;
      //   });
      // }

      // console.log('taskData: ', taskData);
      return {
        ...record,
        schedule: JSON.parse(record?.schedule || "[]"),
        inspection_checklist: record?.inspection_checklist || [{ points: "" }],

        inspection_checkbox:
          record?.inspection_checkbox && record.inspection_checkbox !== "undefined"
            ? record?.inspection_checkbox
            : [{ checkbox: "", inspection_date: "", inspection_outcome: "" }],

        // task_data: record?.task_data || [],
        task_data: record.task_data ? (record.task_data.length > 0 ? record.task_data : []) : [],
        last_maintenance_date: moment(record?.last_maintenance_date).format("YYYY-MM-DD"),
        next_maintenance_date: moment(record?.next_maintenance_date).format("YYYY-MM-DD"),
        inspection_date: moment(record?.inspection_date).format("YYYY-MM-DD"),
        installation_date: moment(record?.installation_date).format("YYYY-MM-DD"),
      };
    })
  );

  /**Count of all Fire Equipment MaintenanceInspection */
  const totalRecord = await countQueryCondition(FireEquipmentMaintenanceInspectionQuery);
  return sendResponse(res, 200, FireEquipmentMaintenanceInspection, totalRecord);
};

/**Function to delete a specific Fire Equipment Maintenance Inspection */
export const deleteFireEquipmentMaintenanceInspection = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("fire_equipment_maintenance_inspection", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(req.user.sessionid, "delete", "Fire Equipment Maintenance Inspection", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

/**Function to view all and single Fire Equipment Register status History*/
export const viewFireEquipmentRegisterStatusHistory = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "fire_equipment_status_history",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = ["organization.name"];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = fire_equipment_status_history.created_by",
    },
    {
      type: "left",
      targetTable: "users AS notify_user",
      onCondition: "notify_user.id = fire_equipment_status_history.notify_to",
    },
    {
      type: "left",
      targetTable: "users AS reported_user",
      onCondition: "reported_user.id = fire_equipment_status_history.reported_by",
    },
    {
      type: "left",
      targetTable: "fire_equipment_register",
      onCondition: "fire_equipment_register.id = fire_equipment_status_history.equipment_id",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = fire_equipment_status_history.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = fire_equipment_status_history.organization",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all Fire Equipment Register */
  const FireEquipmentRegisterQuery = `SELECT fire_equipment_status_history.*,fire_equipment_register.equipment_name AS fire_equipment_name,fire_equipment_register.unique_equipment_id AS unique_equipment_id,fire_equipment_register.equipment_category AS fire_equipment_category, department.name AS department_name, users.name AS created_by, 
  organization.name AS organization_name, CONCAT(notify_user.name,' ',notify_user.surname) AS notify_to_name,CONCAT(reported_user.name,' ',reported_user.surname) AS reported_by_name
      FROM fire_equipment_status_history 
      ${joinsRecord} where fire_equipment_status_history.deleted = 0  ${searchCondition} ${condition}`;

  let [FireEquipmentRegister] = await db.query(FireEquipmentRegisterQuery);

  /**Count of all Fire Equipment Register */
  const totalRecord = await countQueryCondition(FireEquipmentRegisterQuery);
  return sendResponse(res, 200, FireEquipmentRegister, totalRecord);
};
