import db from "../db-config.js";
import {
  insertActivityLog,
  getOrganizationAccordingToDepartment,
  whereCondition,
  makeJoins,
  countQueryCondition,
  deleteRecord,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  decodeAndParseFields,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import FlammableStore from "../sequelize/FlammableStoreSchema.js";
import { createUpdateActionFunction, getCustomActionFunction } from "./customActionCreationController.js";

/**Function to create/update Flammable Store   */
export const FlammableStoreCreateUpdate = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];

  const { id, organization, department, task_data, action_taken, sidebar_id = 305 } = req.body;

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;

  const data = {
    task_data,
    department,
    files: req.files,
    user: req.user,
    id: action_taken,
    organization: organizationId,
    sidebar_id,
  };

  if (id && req.body?.task_data && req.body?.task_data[0] && req.body?.task_data[0]?.role) {
    const { status, arr } = await createUpdateActionFunction(data, req);
    if (status) {
      const action_taken = arr[0].id;
      req.body.action_taken = action_taken;
    }
  }
  if (department) {
    const recordAccordingToOrganization = await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }
  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;
    /**Update Flammable Store Query */
    req.body.scheduled_status = "Scheduled";
    const { query, values } = updateQueryBuilder(FlammableStore, req.body);
    await db.query(query, values);
    /**Insert Activity  */
    await insertActivityLog(req.user.sessionid, "update", "FlammableStore", id);
    return sendResponse(res, 200, "Record updated successfully");
  } else {
    req.body.created_by = req.user.sessionid;
    /**Insert record for Flammable Store */
    const { query, values } = createQueryBuilder(FlammableStore, req.body);
    const createFlammableStore = await db.query(query, values);
    /**Insert record for activity log */
    await insertActivityLog(req.user.sessionid, "create", "Flammable Store", createFlammableStore.insertId);
    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single Flammable Store */
export const viewFlammableStore = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "flammable_store",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = ["CONCAT(users.name , ' ' , users.surname)", "organization.name"];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = flammable_store.created_by",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = flammable_store.location",
    },
    {
      type: "left",
      targetTable: "equipment",
      onCondition: "equipment.id = flammable_store.equipment_type",
    },
    // {
    //   type: "left",
    //   targetTable: "department as responsible_department",
    //   onCondition:
    //     "responsible_department.id = flammable_store.responsible_department",
    // },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = flammable_store.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = flammable_store.organization",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all Flammable Store */
  const FlammableStoreQuery = `SELECT flammable_store.*, location.name AS location_name, department.name AS department_name,  equipment.name AS equipment_type_name, users.name AS created_by, 
  organization.name AS organization_name 
  FROM flammable_store 
  ${joinsRecord} where flammable_store.deleted = 0  ${searchCondition} ${condition}`;
  let [FlammableStore] = await db.query(FlammableStoreQuery);

  for (let flammable of FlammableStore) {
    // action taken
    if (flammable?.action_taken) {
      const data = {
        params: { id: flammable?.action_taken },
        user: req.user,
        query: {},
      };
      const customActionData = await getCustomActionFunction(data);
      flammable.task_data = customActionData;
    }
  }
  FlammableStore = await decodeAndParseFields(FlammableStore);
  /**Count of all Flammable Store */
  const totalRecord = await countQueryCondition(FlammableStoreQuery);
  return sendResponse(res, 200, FlammableStore, totalRecord);
};

/**Function to delete a specific Flammable Store */
export const deleteFlammableStore = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("flammable_store", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(req.user.sessionid, "delete", "Flammable Store", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};
