import db from "../db-config.js";
import {
  checkPolicyUpdated,
  countQueryCondition,
  decodePolicy_statement,
  decodeReference_appendices,
  decodeSingle_statement,
  decodeTheEditorContent,
  deleteRecord,
  encodePolicy_statement,
  encodeReference_appendices,
  encodeSingle_statement,
  encodeTheEditorContent,
  generateNextVersion,
  getLevelDetails,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  insertNotification,
  makeJoins,
  whereCondition,
  uniqueIdGenerator,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  decodeAndParseFields,
} from "../helper/general.js";
import Policy from "../sequelize/PolicySchema.js";
import { sendResponse } from "../helper/wrapper.js";
import PolicyComment from "../sequelize/PolicyCommentSchema.js";

/**Function to create Policy creation  */
export const createUpdatePolicy = async (req, res) => {
  const {
    organization,
    department,
    id,
    policy_title,
    policy_reviewer,
    policy_author,
    policy_owner,
    policy_version_no,
    status,
    send_to_employee,
  } = req.body;
  const policy_reviewer_list = [policy_reviewer];
  /** if updating record after published */
  if (req.body?.published_on) {
    // Parse the ISO datetime string to a Date object
    const date = new Date(req.body.published_on);
    // Convert to IST by adding 5 hours and 30 minutes
    const istOffset = 5 * 60 * 60 * 1000 + 30 * 60 * 1000;
    // IST is UTC+5:30
    const istDate = new Date(date.getTime() + istOffset);
    // Format the IST date to 'YYYY-MM-DD HH:MM:SS'
    req.body.published_on = istDate
      .toISOString()
      .slice(0, 19)
      .replace("T", " ");
  }
  req.body = await encodeAndStringifyFields(req.body);
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  const policyId = await uniqueIdGenerator(
    organizationId,
    department,
    "POL",
    "policy",
    "policy_id"
  );

  // const policyVersionNo = policy_version_no;
  let insertedID = id;
  let isPolicyUpdated = false;
  if (id) {
    req.body.updated_by = req.user.sessionid;
    isPolicyUpdated = await checkPolicyUpdated(id, req.body, req);
    if (!isPolicyUpdated) {
      const { query, values } = updateQueryBuilder(Policy, req.body);
      await db.query(query, values);
    }
  } else {
    req.body.policy_id = policyId;
    req.body.created_by = req.user.sessionid;
    // req.body.policy_version_no = policyVersionNo ?? generateNextVersion(policyVersionNo ?? "");
    const { query, values } = createQueryBuilder(Policy, req.body);
    const [insertedData] = await db.query(query, values);
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Policy ",
      `This user created a new policy with title ${policy_title} for organization ${organizationId}`
    );
    /**Insert record for activity log */
    insertedID = insertedData.insertId;
  }
  if (status == "complete") {
    /** check if previously status changed */
    if (id) {
      const [policy] = await db.query(
        `SELECT status FROM policy WHERE id = ?`,
        [id]
      );
      if (policy[0].status != "complete") {
        await forEmployee(policy_reviewer_list, insertedID);
        const publicUrl = `/public-view-policy/${insertedID}`;

        for (let i = 0; i < policy_reviewer_list.length; i++) {
          await insertNotification(
            "Policy Approval",
            publicUrl,
            send_to_employee[i],
            "url",
            req.user.sessionid
          );
        }
      }
    }
    await forEmployee(send_to_employee, insertedID);
    const publicUrl = `/public-view-policy/${insertedID}`;

    for (let i = 0; i < send_to_employee.length; i++) {
      await insertNotification(
        "Policy Approval",
        publicUrl,
        send_to_employee[i],
        "url",
        req.user.sessionid
      );
    }
  }
  if (status == "approval") {
    await forOwnerApproval(insertedID);
    const publicUrl = `/public-view-policy/${insertedID}`;
    await insertNotification(
      "Policy Approval",
      publicUrl,
      policy_owner,
      "url",
      req.user.sessionid
    );
  }

  return sendResponse(
    res,
    200,
    `Record ${
      isPolicyUpdated
        ? "new version added"
        : req.body.id
        ? "updated"
        : "created"
    }  successfully`
  );
};

export const viewAllPolicy = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "p",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "p.policy_title",
    "p.policy_description",
    "p.policy_purpose",
    "p.policy_id",
    "p.policy_scope",
    "p.policy_definitions",
    "p.policy_statement",
    "p.status",
    "p.current_status",
    "CONCAT(u1.name , ' ' , u1.surname)",
    "CONCAT(u2.name , ' ' , u2.surname)",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = p.created_by",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = p.policy_owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = p.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const Query = `SELECT p.*,  
      organization.name as organization_name, CONCAT(u1.name , ' ' , u1.surname) as created_by_name, CONCAT(u2.name , ' ' , u2.surname) as policy_owner_name, u2.profile as policy_owner_profile FROM policy as p 
      ${joinsRecord}
      where p.deleted = 0 ${searchCondition} ${condition}`;
  // return console.log(Query);
  let [policyDraft] = await db.query(Query);
  policyDraft = await decodeAndParseFields(policyDraft);

  // Decode and parse necessary fields
  for (let item of policyDraft) {
    const evaluated = item.policy_assigned_team;
    let evaluatorDetail = [];
    if (evaluated && evaluated.length > 0)
      evaluatorDetail = await Promise.all(
        evaluated.map(async (ids) => await getLevelDetails(ids))
      );
    item.evaluatorDetail = evaluatorDetail;
    const commented = item.comment ? JSON.parse(item.comment) : null;
    let commentDetail = [];
    if (commented && commented.length > 0)
      commentDetail = await Promise.all(
        commented.map(async (ids) => await getLevelDetails(ids))
      );
    item.commentDetail = commentDetail;
  }

  let policyComments = [];
  if (id) policyComments = await getPolicyComment(id);

  const totalRecord = await countQueryCondition(Query);

  // Merge comments into each object in the data array

  const dataWithComments = policyDraft.map((item) => {
    let isCommented = false;
    return {
      ...item,
      comments: policyComments.filter((comment) => {
        if (req.user.sessionid === comment.created_by) {
          isCommented = true;
        }
        if (
          req.user.sessionid === item.policy_owner ||
          req.user.sessionid === item.created_by ||
          req.user.isSuperAdmin == 1
        ) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        } else if (req.user.sessionid === comment.created_by) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        }
      }),
      isCommented,
    };
  });

  return sendResponse(res, 200, dataWithComments, totalRecord);
};

export const viewAllPolicyWorkflow = async (req, res) => {
  const { id } = req.params;
  // console.log(req.query.filter)
  const condition = await whereCondition({
    table: "p",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "p.policy_title",
    "p.policy_id",
    "p.policy_description",
    "p.policy_purpose",
    "p.policy_scope",
    "p.policy_definitions",
    "p.status",
    "p.current_status",
    "CONCAT(u1.name , ' ' , u1.surname)",
    "CONCAT(u2.name , ' ' , u2.surname)",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  // const searchCondition = req.query.search

  const joins = [
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = p.created_by",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = p.policy_owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = p.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const Query = `SELECT p.*,  
      organization.name as organization_name, CONCAT(u1.name , ' ' , u1.surname) as created_by_name, CONCAT(u2.name , ' ' , u2.surname) as policy_owner_name, u2.profile as policy_owner_profile FROM policy as p 
      ${joinsRecord}
      where p.deleted = 0 ${searchCondition} ${condition}`;
  // return console.log(Query);
  let [policyDraft] = await db.query(Query);
  policyDraft = await decodeAndParseFields(policyDraft);
  // Decode and parse necessary fields
  for (let item of policyDraft) {
    const evaluated = item.policy_assigned_team;
    let evaluatorDetail = [];
    if (evaluated && evaluated.length > 0)
      evaluatorDetail = await Promise.all(
        evaluated.map(async (ids) => await getLevelDetails(ids))
      );
    item.evaluatorDetail = evaluatorDetail;
    const commented = item.comment ? JSON.parse(item.comment) : null;
    let commentDetail = [];
    if (commented && commented.length > 0)
      commentDetail = await Promise.all(
        commented.map(async (ids) => await getLevelDetails(ids))
      );
    item.commentDetail = commentDetail;
  }

  let policyComments = [];
  if (id) policyComments = await getPolicyComment(id);

  const totalRecord = await countQueryCondition(Query);

  const dataWithComments = policyDraft.map((item) => {
    let isCommented = false;
    return {
      ...item,
      comments: policyComments.filter((comment) => {
        if (req.user.sessionid === comment.created_by) {
          isCommented = true;
        }
        if (
          req.user.sessionid === item.policy_owner ||
          req.user.sessionid === item.created_by ||
          req.user.isSuperAdmin == 1
        ) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        } else if (req.user.sessionid === comment.created_by) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        }
      }),
      isCommented,
    };
  });

  return sendResponse(res, 200, dataWithComments, totalRecord);
};

/**function to decode the stored policy statement html content of text editor */
export const forEmployee = async (team, id) => {
  const queryUpdate = `UPDATE policy SET  policy_assigned_team=? WHERE id=?`;
  const valuesUpdate = [JSON.stringify(team), id];
  await db.query(queryUpdate, valuesUpdate);
};

/**function to decode the stored policy statement html content of text editor */
export const forOwnerApproval = async (id) => {
  const [queryUpdate] = await db.query(
    `UPDATE policy SET status='approval'WHERE id= ${id}`
  );
};

/**Function to create comments on policy */
export const policyComment = async (req, res) => {
  const { query, values } = createQueryBuilder(PolicyComment, req.body);
  const insertedData = await db.query(query, values);
  return sendResponse(res, 200, "Comment on policy stored");
};

/**Function to get comments on that policy */
// export const getPolicyComment = async (policy_id) => {
//   const [getPolicyComments] = await db.query(
//     "SELECT policy_comments.created_by,policy_comments.created_at,policy_comments.status,policy_comments.comments,users.name,users.profile  FROM policy_comments left join users on users.id = policy_comments.created_by WHERE policy_comments.policy_id = ?",
//     [policy_id]
//   );
//   // AND policy_comments.status = '0'
//   return getPolicyComments;
// };

/**Function to update policy */
export const changePolicyStatus = async (req, res) => {
  const { status, policy_id, comments } = req.body;
  const getPolicyQuery = `SELECT policy_owner,created_by, current_status FROM policy WHERE id = ?`;
  const getPolicyQueryValue = [policy_id];
  const [policy] = await db.query(getPolicyQuery, getPolicyQueryValue);
  const policy_owner = policy[0].policy_owner;
  // if (req.user.sessionid === policy_owner) {
  //   await db.query(
  //     `UPDATE policy SET current_status= ? WHERE id= ${policy_id}`,
  //     [status]
  //   );
  // }

  if (status == "reject") {
    /** Check comment is not empty */
    if (!comments) {
      return res.status(400).json({
        status: false,
        message: "Please enter comment",
      });
    }
    /**After reject comment on policy store */
    const insertPolicyCommentsQuery = `INSERT INTO policy_comments (policy_id, comments, created_by, status)  VALUES (?, ?, ?, ?)`;
    const insertPolicyCommentsQueryValue = [
      policy_id,
      comments,
      req.user.sessionid,
      0,
    ];
    const [insertedData] = await db.query(
      insertPolicyCommentsQuery,
      insertPolicyCommentsQueryValue
    );
  } else if (status == "approved") {
    const insertPolicyCommentsQuery = `INSERT INTO policy_comments (policy_id, comments, created_by, status)  VALUES (?, ?, ?, ?)`;
    const insertPolicyCommentsQueryValue = [
      policy_id,
      comments,
      req.user.sessionid,
      1,
    ];
    const [insertedData] = await db.query(
      insertPolicyCommentsQuery,
      insertPolicyCommentsQueryValue
    );
  } else if (status === "publish") {
    if (policy[0]?.current_status == "reject") {
      return res.status(400).json({
        status: false,
        message: "Policy has been rejected Can't Publish",
      });
    }
    const insertPolicyCommentsQuery = `INSERT INTO policy_comments (policy_id, comments, created_by, status)  VALUES (?, ?, ?, ?)`;
    const insertPolicyCommentsQueryValue = [
      policy_id,
      comments,
      req.user.sessionid,
      2,
    ];
    const [insertedData] = await db.query(
      insertPolicyCommentsQuery,
      insertPolicyCommentsQueryValue
    );
    if (policy[0]?.current_status !== "reject") {
      const [queryUpdate] = await db.query(
        `UPDATE policy SET publish = 1 ,  published_on = now() WHERE id= ${policy_id}`
      );
    }
  }

  const publicUrl = `/public-view-policy/${policy_id}`;
  await insertNotification(
    `Policy has been ${status} by this user`,
    comments,
    policy[0]?.created_by,
    "text",
    req.user.sessionid
  );

  if (policy[0]?.current_status !== "reject") {
    await db.query(
      `UPDATE policy SET current_status= ? WHERE id= ${policy_id}`,
      [status]
    );
  }

  return sendResponse(res, 200, "Comment on policy stored");
};

/**Function to delete a specific Policy */
export const deletePolicy = async (req, res) => {
  const { id } = req.params;
  /**Delete record of Alert-type*/
  await deleteRecord("policy", id);
  /**Insert record for activity log */
  await insertActivityLog(req.user.sessionid, "delete", "Policy", id);
  return sendResponse(res, 200, "Record Deleted successfully");
};

export const viewPolicyForApproval = async (req, res) => {
  const { id } = req.params;
  // console.log(req.query.filter)

  const condition = await whereCondition({
    table: "p",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "p.policy_title",
    "p.policy_id",
    "p.policy_description",
    "p.policy_purpose",
    "p.policy_scope",
    "p.policy_definitions",
    "p.status",
    "p.current_status",
    "CONCAT(u1.name , ' ' , u1.surname)",
    "CONCAT(u2.name , ' ' , u2.surname)",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const approvalCondition = req.user.isSuperAdmin
    ? ""
    : `(p.policy_assigned_team IN (${req.user.sessionid}) OR p.policy_owner = ${req.user.sessionid}) AND`;

  const joins = [
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = p.created_by",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = p.policy_owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = p.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const Query = `SELECT p.id,p.policy_id,p.policy_title,p.policy_description,p.policy_purpose,p.policy_scope,p.policy_definitions,p.policy_establishment_date,p.policy_owner,p.policy_version_no,p.next_review_date,p.role_responsibilities,p.policy_statement,p.compliance_enforcement,p.reference_appendices,p.policy_assigned_team,p.comment,p.status,p.current_status,p.organization,p.department,p.publish,p.published_on,p.created_by as created_by_id,   
      organization.name as organization_name, u1.name as created_by , u1.surname as created_by_surname , u1.profile as created_by_profile, u2.name as policy_owner_name,u2.surname as policy_owner_surname, u2.profile as policy_owner_profile FROM policy as p 
      ${joinsRecord}
      where p.deleted = 0 AND ${approvalCondition} (p.status = 'employee' OR p.status = 'approval') ${searchCondition} ${condition}`;
  // return console.log(Query);
  const [policyDraft] = await db.query(Query);

  // Decode and parse necessary fields
  for (let item of policyDraft) {
    item.policy_description = await decodeSingle_statement(
      item.policy_description
    );
    item.reference_appendices = item.reference_appendices
      ? JSON.parse(item.reference_appendices)
      : null;
    item.policy_definitions = item.policy_definitions
      ? JSON.parse(item.policy_definitions)
      : null;

    item.role_responsibilities = item.role_responsibilities
      ? await decodeTheEditorContent(
          item.role_responsibilities,
          "role_description"
        )
      : null;

    item.publicUrl = `/public-view-policy/${item.id}`;
    // if (item.role_responsibilities) {
    //   for (let i = 0; i < item.role_responsibilities.length; i++) {
    //     if (item.role_responsibilities[i].role_name) {
    //       const role = await db.query(
    //         `SELECT name FROM roles WHERE id = ${item.role_responsibilities[i].role_name}`
    //       );
    //       item.role_responsibilities[i].role = role[0].name;
    //     }
    //   }
    // }
    item.compliance_enforcement = item.compliance_enforcement
      ? JSON.parse(item.compliance_enforcement)
      : null;
    if (item.policy_statement)
      item.policy_statement = await decodePolicy_statement(
        item.policy_statement
      );
    if (item.reference_appendices)
      item.reference_appendices = await decodeReference_appendices(
        item.reference_appendices
      );
    const evaluated = item.policy_assigned_team
      ? JSON.parse(item.policy_assigned_team || "[]")
      : null;
    let evaluatorDetail = [];
    if (evaluated && evaluated.length > 0)
      evaluatorDetail = await Promise.all(
        evaluated.map(async (ids) => await getLevelDetails(ids))
      );
    item.evaluatorDetail = evaluatorDetail;
    const commented = item.comment ? JSON.parse(item.comment) : null;
    let commentDetail = [];
    if (commented && commented.length > 0)
      commentDetail = await Promise.all(
        commented.map(async (ids) => await getLevelDetails(ids))
      );
    item.commentDetail = commentDetail;
  }

  let policyComments = [];
  if (id) policyComments = await getPolicyComment(id);

  const totalRecord = await countQueryCondition(Query);

  const dataWithComments = policyDraft.map((item) => {
    let isCommented = false;
    return {
      ...item,
      comments: policyComments.filter((comment) => {
        if (req.user.sessionid === comment.created_by) {
          isCommented = true;
        }
        if (
          req.user.sessionid === item.policy_owner ||
          req.user.sessionid === item.created_by ||
          req.user.isSuperAdmin == 1
        ) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        } else if (req.user.sessionid === comment.created_by) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        }
      }),
      isCommented,
    };
  });

  return sendResponse(res, 200, dataWithComments, totalRecord);
};
