import asyncHandler from "express-async-handler";
import db from "../db-config.js";
import {
  checkNameAlreadyExits,
  checkNameInOrganization,
  countQueryCondition,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  whereCondition,
  recordIdExist,
  encodeSingle_statement,
  decodeSingle_statement,
  storeError,
  dateValidator,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  decodeAndParseFields,
  uniqueIdGenerator,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import Role from "../sequelize/RoleSchema.js";

/**Function to create new roles for employee  */
export const createUpdateRole = async (req, res) => {
  const {
    id,
    level,
    name,
    description,
    organization,
    license,
    skills,
    department,
    role_accountability,
  } = req.body;

  if (typeof level === "string" && level.length === 0) {
    req.body.level = null;
  }
  req.body.license = JSON.stringify(license);
  req.body.skills = JSON.stringify(skills);
  req.body.role_accountability =
    JSON.stringify(role_accountability || "") ?? null;
  req.body.description = await encodeSingle_statement(description);

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }
  /**Check that this organization have that particular name or not*/
  const checkNameWithOrganization = await checkNameInOrganization(
    "roles",
    name,
    organizationId
  );

  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;
    if (
      checkNameWithOrganization.length > 0 &&
      checkNameWithOrganization[0].id != id
    ) {
      return sendResponse(res, 500, `Record with ${name} already exists`);
    }
    // const encodedDescription
    /**Update Roles Query */

    const { query, values } = updateQueryBuilder(Role, req.body);
    await db.query(query, values);

    /**Insert Activity  */
    await insertActivityLog(req.user.sessionid, "update", "Roles", id);

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    if (checkNameWithOrganization.length > 0) {
      return res
        .status(200)
        .json({ status: false, message: "This record is already exists" });
    }
    const unique_id = await uniqueIdGenerator(
      organizationId,
      department,
      "ROL",
      "roles",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
    req.body.created_by = req.user.sessionid;

    /**Insert record for roles */
    const { query, values } = createQueryBuilder(Role, req.body);
    // console.log(responsibilities);
    const [createRoles] = await db.query(query, values);

    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Roles",
      createRoles.insertId
    );

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single role */
export const viewRole = async (req, res) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "roles",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "roles.name",
    "roles.description",
    "roles.hierarchy",
    "roles.qualification",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  // let searchCondition = req.query.search
  //   ? `AND (roles.name LIKE '%${req.query.search}%' OR roles.description LIKE '%${req.query.search}%' OR organization.name LIKE '%${req.query.search}%' OR roles.hierarchy LIKE '%${req.query.search}%' OR roles.qualification LIKE '%${req.query.search}%' OR users.name LIKE '%${req.query.search}%' )`
  //   : "";

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = roles.created_by",
    },
    {
      type: "left",
      targetTable: "roles as r1",
      onCondition: "r1.id = roles.report_to_role",
    },
    {
      type: "left",
      targetTable: "license",
      onCondition: "license.id = roles.license",
    },
    {
      type: "left",
      targetTable: "role_hierarchy",
      onCondition: "role_hierarchy.id = roles.hierarchy",
    },
    {
      type: "left",
      targetTable: "highest_qualification",
      onCondition: "highest_qualification.id = roles.qualification",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = roles.organization",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all roles */
  const rolesQuery = `SELECT roles.unique_id,roles.responsibilities,roles.pre_employment,roles.post_employment,roles.report_to_role,r1.name as report_to_role_name,
  roles.hierarchy,roles.qualification,roles.skills,roles.year_of_experience,roles.license , license.name as license_name,roles.other_requirements,roles.select_other_requirements,roles.id,roles.id,roles.created_by as created_by_id,roles.level,roles.name,roles.description,roles.organization,organization.name as organization_name, organization.header_image , organization.footer_image , organization.business_logo,users.name as created_by,users.profile as created_by_profile ,users.surname as created_by_surname , role_hierarchy.name as hierarchy_name, highest_qualification.name as qualification_name,  roles.organization 
  FROM roles 
  ${joinsRecord} where roles.deleted = 0  ${searchCondition} ${condition}`;
  let [roles] = await db.query(rolesQuery);
  roles = await decodeAndParseFields(roles);
  for (const role of roles) {
    role.description = await decodeSingle_statement(role.description);
  }
  for (let role of roles) {
    let skills = [];
    let responsibilities = [];
    let license = [];
    if (role.skills?.length > 0 && role.skills[0]) {
      [skills] = await db.query(
        `SELECT id , name FROM skills WHERE id IN (${role.skills})`
      );
    }
    if (role.license?.length > 0 && role.license[0]) {
      [license] = await db.query(
        `SELECT name FROM license WHERE id IN (${role.license})`
      );
    }
    if (role.responsibilities?.length > 0 && role.responsibilities[0]) {
      const responsibilitiesIds = role.responsibilities.filter(
        (id) => Number.isInteger(id) && id > 0
      ); // Remove empty and invalid IDs
      if (responsibilitiesIds.length > 0) {
        const placeholders = responsibilitiesIds.map(() => "?").join(", "); // Create placeholders
        [responsibilities] = await db.query(
          `SELECT name FROM responsibility WHERE id IN (${placeholders})`,
          responsibilitiesIds
        );
      } else {
        responsibilities = []; // Return empty array if no valid IDs
      }
      // [responsibilities] = await db.query(
      //   `SELECT name FROM responsibility WHERE id IN (${responsibilitiesIds})`
      // );
    }
    role.skillsDetail = skills.map((ele) => {
      return {
        id: ele.id,
        name: ele.name,
      };
    });
    role.licenseDetail = license.map((ele) => {
      return {
        id: ele.id,
        name: ele.name,
      };
    });
    role.responsibilityDetail = responsibilities.map((ele) => {
      return {
        id: ele.id,
        name: ele.name,
      };
    });
  }
  const totalRecord = await countQueryCondition(rolesQuery);

  return sendResponse(res, 200, roles, totalRecord);
};

/**Function to delete a specific role */
export const deleteRole = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("roles", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(req.user.sessionid, "delete", "roles", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

/**function to import role */
export const importRoleData = async (req, res) => {
  const { organization, data } = req.body;
  const alreadyExistData = [];

  for (let i = 0; i < data.length; i++) {
    let role = data[i];

    // Check if the role already exists
    const rolesCheck = await checkNameAlreadyExits(
      "roles",
      "name",
      role.name,
      organization
    );

    /** Check if role dependent data exists or not */
    const organizationCheck = await recordIdExist("organization", organization);

    const skillsCheck = await recordIdExist("skills", role.skills);
    /**Check if that these fields is missing then show error that this field is missing */
    const { error } = rolesCreateUpdateValidationSchema.validate(
      { ...role, ...organization },
      { abortEarly: false }
    );
    if (rolesCheck) {
      alreadyExistData.push({
        message: `Role already exists at index ${i + 1}`,
      });
    } else if (!organizationCheck) {
      alreadyExistData.push({
        message: `Organization not found at index ${i + 1}`,
      });
    } else if (skillsCheck) {
      alreadyExistData.push({
        message: `Skills not found at index ${i + 1}`,
      });
    } else if (error) {
      const errors = error.details.map((detail) => detail.message);
      alreadyExistData.push({
        message: `Error at index ${i + 1}: ${errors[0]}`,
      });
    } else {
      role.created_by = req.user.sessionid;
      role.organization = organization;
      // Prepare values for insertion
      const { query, values } = createQueryBuilder(Role, role);
      await db.query(query, values);
      // Insert record for activity log
      await insertActivityLog(
        req.user.sessionid,
        "create",
        "Roles",
        `This user create new role with name ${role.name}`
      );
    }
  }
  if (alreadyExistData.length === data.length) {
    return sendResponse(res, 500, "File data is not valid", alreadyExistData);
  } else {
    return sendResponse(res, 200, "Record created successfully", errorMessage);
  }
};

// /**validate the transaction data */
// const validateRoleData = async (row, organization) => {
//   const rolesCheckQuery = `SELECT id,name,description FROM roles WHERE deleted = '0' AND name = ? AND organization = ?`;
//   const rolesCheck = await db.query(rolesCheckQuery, [row.name, organization]);
//   return rolesCheck.length>0;
// };
