import {
  countQueryCondition,
  decodeAndParseFields,
  decodeSingle_statement,
  deleteRecord,
  encodeSingle_statement,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import db from "../db-config.js";
import { createQueryBuilder } from "../helper/queryBuilder.js";
import { sendResponse } from "../helper/wrapper.js";
import SecurityIncident from "../sequelize/SecurityIncidentFormSchema.js";

/**Function to create Safety Incident Form  */
export const createSecurityIncidentForm = async (req, res) => {
  const { id } = req.body;
  const sessionId = req.user.sessionid;
  let organizationId = req.body.organization;
  const { department, comments } = req.body;

  const status = id ? "Updated" : "Created";
  if (department) {
    const recordAccordingToOrganization = await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0]?.organization;
    if (organizationId) {
      req.body.organization = organizationId;
    }
  }

  req.body.comments = await encodeSingle_statement(comments);
  req.body.description_of_incident = await encodeSingle_statement(req.body.description_of_incident);
  req.body.description_of_forced_entry = await encodeSingle_statement(req.body.description_of_forced_entry);

  req.body.created_by = sessionId;
  const { query, values } = id
    ? updateQueryBuilder(SecurityIncident, req.body)
    : createQueryBuilder(SecurityIncident, req.body);
  const [createSecurityForm] = await db.query(query, values);
  if (!id && createSecurityForm.insertId === 0) {
    return sendResponse(res, 404, "Cannot create record");
  }

  await insertActivityLog(
    req.user.sessionid,
    "create",
    "Security Incident Form",
    `This user created a new security incident Form Record for organization ${organizationId}`
  );

  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getSecurityIncidentForm = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "security_incident_form",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = [
    "security_incident_form.location_of_incident",
    "security_incident_form.incident_reference_no",
    "security_incident_form.type_of_security_incident",
    "security_incident_form.description_of_incident",
    "security_incident_form.committed_by_whom",
    "security_incident_form.specify_name",
    "security_incident_form.name_of_employee",
    "security_incident_form.employee_id",
    "security_incident_form.employee_type",
    "security_incident_form.what_kind_of_theft",
    "security_incident_form.name_of_insurer",
    "security_incident_form.is_transnet_or_third_party_insurer",
    "security_incident_form.reported_by_who_to_insurer",
    "security_incident_form.claim_number",
    "security_incident_form.who_received_the_claim",
    "security_incident_form.name_of_police_station_saps",
    "security_incident_form.case_number_saps",
    "security_incident_form.name_of_contact_person_saps",
    "security_incident_form.contact_details_saps",
    "security_incident_form.investigation_officer_saps",
    "security_incident_form.name_of_police_station_arrest",
    "security_incident_form.case_number_arrest",
    "security_incident_form.name_of_contact_person_arrest",
    "security_incident_form.contact_details_arrest",
    "security_incident_form.investigation_officer_arrest",
    "security_incident_form.recovered_items",
    "security_incident_form.description_of_forced_entry",
    "security_incident_form.asset_number",
    "security_incident_form.location_of_asset",
    "security_incident_form.person_reporting_break_in",
    "security_incident_form.witness_of_break_in",
    "security_incident_form.tools_used",
    "security_incident_form.comments",
    "security_incident_form.form_completed_by",
    "security_incident_form.designation",
    "security_incident_form.signature",
  ];

  const searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = security_incident_form.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = security_incident_form.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = security_incident_form.department",
    },
    {
      type: "left",
      targetTable: "category_type",
      onCondition: "category_type.id = security_incident_form.category_id",
    },
    {
      type: "left",
      targetTable: "users u1",
      onCondition: "u1.id = security_incident_form.person_who_reported_investigation_saps",
    },
    {
      type: "left",
      targetTable: "users u2",
      onCondition: "u2.id = security_incident_form.who_was_involved_in_arrest",
    },
    {
      type: "left",
      targetTable: "users AS reportedBy",
      onCondition: "reportedBy.id =  security_incident_form.reported_by",
    },
  ];
  const joinCondition = await makeJoins(joins);
  const securityIncidentsFormQuery = `SELECT security_incident_form.* , organization.name as organization_name ,department.name as department_name , users.name as created_by_name , users.surname as created_by_surname  , users.profile as created_by_profile ,CONCAT(u1.name , ' ' , u1.surname) as person_who_reported_investigation_saps_name , CONCAT(u2.name , ' ' , u2.surname) as who_was_involved_in_arrest_name , u1.profile as person_who_reported_investigation_saps_profile , u2.profile as who_was_involved_in_arrest_profile , category_type.name as category_name , CONCAT(reportedBy.name , ' ' , reportedBy.surname) as reported_by_name FROM security_incident_form ${joinCondition} WHERE security_incident_form.deleted = 0 ${condition} ${searchCondition}`;

  const totalRecord = await countQueryCondition(securityIncidentsFormQuery);

  let [securityIncidentForm] = await db.query(securityIncidentsFormQuery);
  securityIncidentForm = await decodeAndParseFields(securityIncidentForm);
  return sendResponse(res, 200, securityIncidentForm, totalRecord);
};

export const deleteSecurityIncidentForm = async (req, res) => {
  const { id } = req.params;
  const deleteForm = await deleteRecord("security_incident_form", id);
  if (deleteForm) {
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "Security Incidents Form",
      `This user deleted a security incidents Form Record for organization ${id}`
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
  return sendResponse(res, 404, "Cannot delete record");
};
