import db from "../db-config.js";
import {
  insertActivityLog,
  getOrganizationAccordingToDepartment,
  whereCondition,
  makeJoins,
  countQueryCondition,
  deleteRecord,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  getSopComment,
  getRecord,
  decodeAndParseFields,
  uniqueIdGenerator,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import SOPComment from "../sequelize/sopCommentSchema.js";
import SOP from "../sequelize/SOPSchema.js";
import moment from "moment";

/**Function to create/update sop   */
export const SOPCreateUpdate = async (req, res) => {
  /**Check record if organization is not coming then fetch organization according to department */
  req.body = (await decodeAndParseFields([req.body]))[0];
  let { id, organization, department, sop_status, affected_sop, sop_department } = req.body;

  department = sop_department ? sop_department : department;
  req.body.department = department;
  // console.log(req.body.sop_reviewer, "sop_reviewer before");
  // req.body.sop_reviewer = Array.isArray(req.body.sop_reviewer)
  //   ? req.body.sop_reviewer.map((sop) => sop.employee)
  //   : req.body.sop_reviewer;
  // console.log(req.body.sop_reviewer, "sop_review after");
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization = await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  if (affected_sop !== null) {
    const getAffectedSopData = await db.query(`SELECT * FROM sops WHERE id = ?`, [affected_sop]);
    if (!getAffectedSopData[0]) {
      return sendResponse(res, 400, "Affected SOP not found");
    }
    await db.query(`UPDATE sops SET is_archived = ? WHERE id = ?`, [true, affected_sop]);
  }
  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;

    const [sopRecord] = await db.query(`SELECT * FROM sops WHERE id = ?`, [id]);
    /** Policy has been rejected then only amendment sop can only be created can't edit */
    if (sopRecord[0]?.current_status === "rejected") {
      return sendResponse(res, 400, "SOP has been rejected Only Amendment SOP can be created");
    }

    if (sopRecord[0].sop_status !== "complete" && sop_status === "complete") {
      await db.query("UPDATE sops SET current_status = 'reviewing' WHERE id = ?", [id]);
    }

    /**Update sop Query */
    const { query, values } = updateQueryBuilder(SOP, req.body);
    await db.query(query, values);
    /**Insert Activity  */
    await insertActivityLog(req.user.sessionid, "update", "SOP", id);
    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const unique_id = await uniqueIdGenerator(organizationId, department, "SOP", "sops", "unique_id", "unique_id");

    req.body.unique_id = unique_id;
    req.body.created_by = req.user.sessionid;
    /**Insert record for sop */

    const { query, values } = createQueryBuilder(SOP, req.body);
    const createSOP = await db.query(query, values);
    if (sop_status === "complete") {
      /** update the current_status */
      await db.query("UPDATE sops SET current_status = 'reviewing' WHERE id = ?", [createSOP?.insertId]);
    }
    /**Insert record for activity log */
    await insertActivityLog(req.user.sessionid, "create", "sop", createSOP.insertId);
    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single sop */
export const viewSOP = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "sops",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = [ "sops.type_of_sop","sops.sop_name","organization.name"];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  /**Make Joins according to tables */
  const joins = [
    // Joining created_by user
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = sops.created_by",
    },
    {
      type: "left",
      targetTable: "roles as r1",
      onCondition: "u1.role = r1.id", // Role of created_by
    },
    {
      type: "left",
      targetTable: "department as d1",
      onCondition: "d1.id = r1.department", // Department of created_by role
    },

    // Joining sop_owner user
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = sops.sop_owner",
    },
    {
      type: "left",
      targetTable: "roles as r2",
      onCondition: "u2.role = r2.id", // Role of sop_owner
    },
    {
      type: "left",
      targetTable: "department as d2",
      onCondition: "d2.id = r2.department", // Department of sop_owner role
    },

    // Joining sop_approver user
    {
      type: "left",
      targetTable: "users as u3",
      onCondition: "u3.id = sops.sop_approver",
    },
    {
      type: "left",
      targetTable: "roles as r3",
      onCondition: "u3.role = r3.id", // Role of sop_approver
    },
    {
      type: "left",
      targetTable: "department as d3",
      onCondition: "d3.id = r3.department", // Department of sop_approver role
    },

    // Joining sop_author user
    {
      type: "left",
      targetTable: "users as u4",
      onCondition: "u4.id = sops.sop_author",
    },
    {
      type: "left",
      targetTable: "roles as r4",
      onCondition: "u4.role = r4.id", // Role of sop_author
    },
    {
      type: "left",
      targetTable: "department as d4",
      onCondition: "d4.id = r4.department", // Department of sop_author role
    },

    // Joining other tables
    {
      type: "left",
      targetTable: "sop_category",
      onCondition: "sop_category.id = sops.sop_category",
    },
    {
      type: "left",
      targetTable: "department as d5",
      onCondition: "d5.id = sops.sop_department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = sops.organization",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = sops.ddrm_id",
    },
    {
      type: "left",
      targetTable: "document_creation",
      onCondition: "document_creation.id = repository.document_creation_id",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all sop */
  const SOPQuery = `
    SELECT sops.*, 
      repository.url AS file,
      document_creation.document_name AS title,
      d1.name AS created_by_department,  
      d2.name AS sop_owner_department,  
      d3.name AS sop_approver_department,  
      d4.name AS sop_author_department,  
      CONCAT(u1.name, ' ', u1.surname) AS created_by_name, 
      CONCAT(u2.name, ' ', u2.surname) AS sop_owner_name, 
      CONCAT(u3.name, ' ', u3.surname) AS sop_approver_name, 
      CONCAT(u4.name, ' ', u4.surname) AS sop_author_name, 
      organization.name AS organization_name,  
      sop_category.name AS sop_category_name,
      r1.name AS created_by_role, 
      r2.name AS sop_owner_role, 
      r3.name AS sop_approver_role, 
      r4.name AS sop_author_role
    FROM sops 
    ${joinsRecord} 
    WHERE sops.deleted = 0 
       
      ${searchCondition} 
      ${condition}
  `;

  let [SOP] = await db.query(SOPQuery);
  SOP = await decodeAndParseFields(SOP);
  for (let item of SOP) {
    delete item.updated_at;
    delete item.created_at;
    let sop_reviewer = [];
    if (Array.isArray(item.sop_reviewer)) {
      for (const i of item.sop_reviewer) {
        if (i?.employee) sop_reviewer.push(i?.employee);
        else break;
      }
    }
    // sop_reviewer = item.sop_reviewer;
    // if (Array.isArray(sop_reviewer) && sop_reviewer.length) {
    //   const [usersList] = await db.query(
    //     `SELECT users.id, CONCAT(users.name , ' ' , users.surname) as name, profile, roles.name as role_name, department.name as sop_reviewer_department FROM users LEFT JOIN roles ON users.role = roles.id LEFT JOIN department ON roles.department = department.id WHERE users.id IN (${sop_reviewer})`
    //   );
    //   item.sop_reviewer_details = usersList;
    // }

    if (Array.isArray(item.sop_reviewer) && item.sop_reviewer.length) {
      for (let i of item.sop_reviewer) {
        const [employee] = await db.query(
          `SELECT CONCAT(users.name , ' ' , users.surname) as name FROM users WHERE id = ${i.employee}`
        );
        i.employee_name = employee[0].name;
      }
    }
    if (Array.isArray(item.with_access_items) && item.with_access_items.length && item.with_access_items[0].employee) {
      for (let i of item.with_access_items) {
        const [employee] = await db.query(
          `SELECT CONCAT(users.name , ' ' , users.surname) as name FROM users WHERE id = ${i.employee}`
        );
        i.employee_name = employee[0].name;
      }
    }
    if (
      Array.isArray(item.without_access_items) &&
      item.without_access_items.length &&
      item.without_access_items[0].employee
    ) {
      for (let i of item.without_access_items) {
        const [employee] = await db.query(
          `SELECT CONCAT(users.name , ' ' , users.surname) as name FROM users WHERE id = ${i.employee}`
        );
        i.employee_name = employee[0].name;
        // i.email = employee[0].email;
      }
    }
    if (
      Array.isArray(item.without_access_and_without_email) &&
      item.without_access_and_without_email.length &&
      item.without_access_and_without_email[0].employee
    ) {
      for (let i of item.without_access_and_without_email) {
        const [employee] = await db.query(
          `SELECT CONCAT(users.name , ' ' , users.surname) as name FROM users WHERE id = ${i.employee}`
        );
        i.employee_name = employee[0].name;
      }
    }
  }
  let sopComments = [];
  if (id) sopComments = await getSopComment(id);

  /**Count of all sop */
  const totalRecord = await countQueryCondition(SOPQuery);

  const dataWithComments = SOP.map((item) => {
    let isCommented = false;
    return {
      ...item,
      comments: sopComments.filter((comment) => {
        if (req.user.sessionid === comment.created_by) {
          isCommented = true;
        }
        if (
          req.user.sessionid === item.sop_owner ||
          req.user.sessionid === item.sop_approver ||
          req.user.sessionid === item.created_by ||
          req.user.isSuperAdmin == 1
        ) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        } else if (req.user.sessionid === comment.created_by) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        }
      }),
      isCommented,
    };
  });

  return sendResponse(res, 200, dataWithComments, totalRecord);
};

/**Function to delete a specific sop */
export const deleteSOP = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("sops", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(req.user.sessionid, "delete", "sop", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

export const sopComment = async (req, res) => {
  const { status, sop_approver, sop_id } = req.body;

  const [record] = await getRecord("sops", "id", sop_id);

  // if (sop_approver) {
  //   const reviewerList = JSON.parse(record?.sop_reviewer || "[]");
  //   if (!reviewerList.includes(req.user.sessionid)) {
  //     return sendResponse(res, 200, "Access Denied");
  //   } else {
  //     await db.query("Update sops set sop_approver = ? , current_status = 'approval' where id = ?", [
  //       sop_approver,
  //       req.body.sop_id,
  //     ]);
  //   }
  // }

  if (sop_approver) {
    const reviewerList = JSON.parse(record?.sop_reviewer || "[]");

    const hasAccess = reviewerList.some((reviewer) => reviewer.employee === req.user.sessionid);

    if (!hasAccess) {
      return sendResponse(res, 200, "Access Denied");
    } else {
      await db.query("UPDATE sops SET sop_approver = ?, current_status = 'approval' WHERE id = ?", [
        sop_approver,
        req.body.sop_id,
      ]);
    }
  }

  req.body.created_by = req.user.sessionid;
  const { query, values } = createQueryBuilder(SOPComment, req.body);

  // if (status == "rejected") {
  //   /** Update the sop status on the user basis if reviewer to draft or if approver then reject it*/
  //   const sop_reviewer = JSON.parse(record?.sop_reviewer || "[]");

  //   if (sop_reviewer.includes(req.user.sessionid)) {
  //     await db.query("Update sops set sop_status = 'draft', current_status = 'rejected' where id = ?", [sop_id]);
  //   }
  //   if (record?.sop_approver == req.user.sessionid || req.user.isSuperAdmin == 1) {
  //     await db.query("Update sops set current_status = 'rejected' where id = ?", [sop_id]);
  //   }
  // }

  // if (status == "approved") {
  //   /** if person is approver then update the policy status */
  //   if (record?.sop_approver == req.user.sessionid) {
  //     await db.query("Update sops set  current_status = 'approved' where id = ?", [sop_id]);
  //   }
  // }
  if (status === "rejected") {
    /** Update the SOP status based on user role (reviewer or approver) */
    const sop_reviewer = JSON.parse(record?.sop_reviewer || "[]");

    // Check if the user is in the reviewer list based on the employee field
    const isReviewer = sop_reviewer.some((reviewer) => reviewer.employee === req.user.sessionid);

    if (isReviewer || req.user.isSuperAdmin == 1) {
      await db.query("UPDATE sops SET sop_status = 'complete', current_status = 'rejected' WHERE id = ?", [sop_id]);
    }

    // Check if the user is the approver or a super admin
    if (record?.sop_approver == req.user.sessionid || req.user.isSuperAdmin == 1) {
      await db.query("UPDATE sops SET current_status = 'rejected' WHERE id = ?", [sop_id]);
    }
  }

  if (status === "approved") {
    /** If the user is the approver, update the sop status to approved */
    if (record?.sop_approver == req.user.sessionid || req.user.isSuperAdmin == 1) {
      await db.query("UPDATE sops SET sop_status = 'complete', current_status = 'approved' WHERE id = ?", [sop_id]);
    }
  }

  await db.query(query, values);
  return sendResponse(res, 200, "Record on sop stored");
};

/**Function to view all approval work flow and single Sop */
export const viewSopApprovalWorkFlow = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "sops",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = ["CONCAT(users.name , ' ' , users.surname)", "organization.name"];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  // const approvalCondition = req.user.isSuperAdmin
  //   ? ""
  //   : `((JSON_CONTAINS(sops.sop_reviewer, '${req.user.sessionid}')) OR sops.sop_owner = ${req.user.sessionid} OR sops.sop_approver = ${req.user.sessionid}) AND`;
  const approvalCondition = req.user.isSuperAdmin
    ? ""
    : `(
      (
        JSON_CONTAINS(
          sops.sop_reviewer, 
          JSON_OBJECT('employee', ${req.user.sessionid})
        )
      ) 
      OR sops.sop_owner = ${req.user.sessionid} 
      OR sops.sop_approver = ${req.user.sessionid}
    ) AND`;

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = sops.created_by",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = sops.sop_owner",
    },
    {
      type: "left",
      targetTable: "users as u3",
      onCondition: "u3.id = sops.sop_approver",
    },
    {
      type: "left",
      targetTable: "users as u4",
      onCondition: "u4.id = sops.sop_author",
    },
    {
      type: "left",
      targetTable: "sop_category",
      onCondition: "sop_category.id = sops.sop_category",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = sops.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = sops.organization",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = sops.ddrm_id",
    },
    {
      type: "left",
      targetTable: "document_creation",
      onCondition: "document_creation.id = repository.document_creation_id",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all sop */
  const SOPQuery = `SELECT sops.*, department.name AS department_name,  CONCAT(u1.name, ' ',u1.surname) AS created_by, CONCAT(u2.name, ' ',u2.surname) AS sop_owner_name, CONCAT(u3.name, ' ',u3.surname) AS sop_approver_name, CONCAT(u4.name, ' ',u4.surname) AS sop_author_name, organization.name AS organization_name ,  r1.name AS sop_approver_role , r2.name AS sop_owner_role , r3.name AS sop_author_role , r3.name AS sop_reviewer_role,    sop_category.name AS sop_category_name, repository.url as file, document_creation.document_name AS title
      FROM sops 
      ${joinsRecord} LEFT JOIN roles r1 ON u3.role = r1.id LEFT JOIN roles r2 ON u3.role = r2.id LEFT JOIN roles r3 ON u4.role = r3.id where sops.deleted = 0 AND ${approvalCondition} (sops.sop_status = 'Under Review')  ${searchCondition} ${condition}`;
  let [SOP] = await db.query(SOPQuery);
  SOP = await decodeAndParseFields(SOP);
  for (let item of SOP) {
    delete item.updated_at;
    delete item.created_at;
    let sop_reviewer = [];
    if (Array.isArray(item.sop_reviewer)) {
      for (const i of item.sop_reviewer) {
        if (i?.employee) sop_reviewer.push(i?.employee);
        else break;
      }
    }
    // sop_reviewer = item.sop_reviewer;
    if (Array.isArray(sop_reviewer) && sop_reviewer.length) {
      const [usersList] = await db.query(
        `SELECT id, CONCAT(name , ' ' , surname) as name, profile  FROM users WHERE id IN (${sop_reviewer})`
      );
      item.sop_reviewer_details = usersList;
    }

    if (Array.isArray(item.with_access_items) && item.with_access_items.length && item.with_access_items[0].employee) {
      for (let i of item.with_access_items) {
        const [employee] = await db.query(
          `SELECT CONCAT(users.name , ' ' , users.surname) as name FROM users WHERE id = ${i.employee}`
        );
        i.employee_name = employee[0].name;
      }
    }
    if (
      Array.isArray(item.without_access_items) &&
      item.without_access_items.length &&
      item.without_access_items[0].employee
    ) {
      for (let i of item.without_access_items) {
        const [employee] = await db.query(
          `SELECT CONCAT(users.name , ' ' , users.surname) as name FROM users WHERE id = ${i.employee}`
        );
        i.employee_name = employee[0].name;
        // i.email = employee[0].email;
      }
    }
    if (
      Array.isArray(item.without_access_and_without_email) &&
      item.without_access_and_without_email.length &&
      item.without_access_and_without_email[0].employee
    ) {
      for (let i of item.without_access_and_without_email) {
        const [employee] = await db.query(
          `SELECT CONCAT(users.name , ' ' , users.surname) as name FROM users WHERE id = ${i.employee}`
        );
        i.employee_name = employee[0].name;
      }
    }
  }
  let sopComments = [];
  if (id) sopComments = await getSopComment(id);

  /**Count of all sop */
  const totalRecord = await countQueryCondition(SOPQuery);

  const dataWithComments = SOP.map((item) => {
    let isCommented = false;
    return {
      ...item,
      comments: sopComments.filter((comment) => {
        if (req.user.sessionid === comment.created_by) {
          isCommented = true;
        }
        if (
          req.user.sessionid === item.sop_owner ||
          req.user.sessionid === item.sop_approver ||
          req.user.sessionid === item.created_by ||
          req.user.isSuperAdmin == 1
        ) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        } else if (req.user.sessionid === comment.created_by) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            status: comment.status,
          };
        }
      }),
      isCommented,
    };
  });

  return sendResponse(res, 200, dataWithComments, totalRecord);
};
export const getSopHistory = async (req, res) => {
  const tableName = "sops";
  const { id } = req.params;
  if (!id) return sendResponse(res, 400, "Id is required");
  const page = parseInt(req.query.page) || 1;
  const pageSize = parseInt(req.query.pageSize) || 10;
  const offset = (page - 1) * pageSize;
  const orderBy = req.query.orderBy || "DESC";

  const query = `SELECT * FROM log_history WHERE table_name = ? AND record_id = ? ORDER BY changed_at ${orderBy} LIMIT ? OFFSET ?`;
  const [result] = await db.query(query, [tableName, id, pageSize, offset]);

  const countQuery = `SELECT COUNT(*) as total FROM log_history WHERE table_name = ? AND record_id = ?`;
  const [countResult] = await db.query(countQuery, [tableName, id]);
  const totalRecord = countResult[0]?.total || 0;

  const filteredResult = result.map((entry) => {
    if (entry.changed_fields) {
      let changedFields = entry.changed_fields;

      if (typeof changedFields === "string") {
        try {
          changedFields = JSON.parse(changedFields);
        } catch (e) {
          console.error("Error parsing JSON in changed_fields:", e);
        }
      }

      Object.keys(changedFields).forEach((key) => {
        let value = changedFields[key];

        if (typeof value === "string") {
          try {
            value = JSON.parse(value);
          } catch (e) {}
        }

        changedFields[key] = value;
      });

      const filteredChangedFields = Object.entries(changedFields)
        .filter(([key, value]) => value !== null) // Keep only non-null values
        .reduce((acc, [key, value]) => {
          acc[key] = value; // Rebuild the object with non-null values
          return acc;
        }, {});

      entry = {
        ...entry,
        changed_fields: filteredChangedFields,
      };
    }

    entry.changed_at = moment(entry.changed_at).format("MMMM Do YYYY, h:mm A"); // Example: January 22, 2025 10:30 AM

    return entry;
  });

  return sendResponse(res, 200, filteredResult, totalRecord);
};
