import db from "../db-config.js";
import {
  countQueryCondition,
  decodeSingle_statement,
  deleteSettingRecord,
  encodeSingle_statement,
  insertActivityLog,
  makeJoins,
  whereCondition,
  settingsUpdater,
  searchConditionRecord,
  createQueryBuilder,
  uniqueIdGenerator,
} from "../helper/general.js";
import TemplateCategory from "../sequelize/TemplateCategorySchema.js";
import { sendResponse } from "../helper/wrapper.js";

/**Function to create new Incident Category for employee  */
export const createUpdateTemplateCategory = async (req, res) => {
  let { id, category, description, organization } = req.body;
  req.body.description = await encodeSingle_statement(description);
  req.body.name = category;

  req.body[req.body.id ? "updated_by" : "created_by"] = req.user.sessionid;

  /**If id comes in body then it will update the query */
  if (id) {
    if (organization.length === 0) {
      return sendResponse(
        res,
        404,
        "Please Select at least one business Structure"
      );
    }
    const result = await settingsUpdater(
      TemplateCategory,
      organization,
      req.body,
      req.user
    );
    if (!result) {
      return sendResponse(res, 404, "No data found");
    }

    await insertActivityLog(
      req.user.sessionid,
      "update",
      "Template Category",
      `This user Update Template Category which id is ${id}`
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const insertPromises = organization.map(async (element) => {
      // Check if the category type already exists for the organization
      const [checkNameWithOrganization] = await db.query(
        `SELECT name FROM template_category WHERE name = ? AND deleted = "0" AND organization = ?`,
        [category, element]
      );
      if (req.body.parent_category) {
        const [checkName] = await db.query(
          `SELECT name FROM template_category WHERE id = ${req.body.parent_category}`
        );
        const [newParent] = await db.query(
          `SELECT id FROM template_category WHERE name  = '${checkName[0].name}' AND organization = ${element}`
        );
        req.body.parent_category = newParent[0].id;
      }

      req.body.organization = element;

      const unique_id = await uniqueIdGenerator(
        req.body.organization,
        req?.body?.department || null,
        "TemplateCategory",
        "template_category",
        "unique_id",
        "unique_id"
      );
      req.body.unique_id = unique_id;

      if (checkNameWithOrganization.length === 0) {
        const { query, values } = createQueryBuilder(TemplateCategory, {
          ...req.body,
          organization: element,
        });

        await db.query(query, values);

        // Insert record for activity log
        await insertActivityLog(
          req.user.sessionid,
          "create",
          "Template Category",
          `This user created a new category '${category}' for this organization ${element}`
        );
      }
    });

    await Promise.all(insertPromises);

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single Incident Category */
export const viewTemplateCategory = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "template_category",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "template_category.name",
    "template_category.description",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = template_category.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = template_category.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const templateCategoryQuery = `SELECT template_category.unique_id,template_category.id,template_category.created_by as created_by_id,template_category.name AS category,template_category.description,users.name as created_by,users.surname as created_by_surname ,users.profile as created_by_profile,organization.name as organization_name ,template_category.organization ,template_category.parent_category  FROM template_category ${joinsRecord} where template_category.deleted = 0 ${searchCondition} ${condition}`;
  const [templateCategory] = await db.query(templateCategoryQuery);

  for (const item of templateCategory) {
    item.description = await decodeSingle_statement(item.description);
    if (req.query.grouped == "true") {
      const [organizations] = await db.query(
        `SELECT organization FROM template_category WHERE deleted = 0 AND name = ?`,
        [item.category]
      );
      // console.log(organizations);
      const arr = organizations.map((item) => item.organization);
      item.organizations = arr;
    }
    const [fetchChild] = await db.query(
      `SELECT * FROM template_category WHERE deleted = 0 AND parent_category = ?`,
      [item.id]
    );
    item.sub_category = fetchChild;
  }

  /**Count all organization */
  const totalRecord = await countQueryCondition(templateCategoryQuery);

  return sendResponse(res, 200, templateCategory, totalRecord);
};

/**Function to delete a specific Incident Category */
export const deleteTemplateCategory = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteSettingRecord("template_category", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "template_category",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

export const getParentTemplateCategoryList = async (req, res) => {
  let { organizations } = req.query;
  if (!organizations) {
    const [arrOrg] = await db.query(
      `SELECT id FROM organization WHERE deleted = 0`
    );
    organizations = arrOrg.map((item) => item.id);
    // return sendResponse(res, 400, "organizations is required");
  }
  // console.log(organizations);
  organizations =
    typeof organizations == "string"
      ? JSON.parse(organizations)
      : organizations;
  const parentTemplateCategoryQuery = `SELECT r1.*
FROM template_category r1
JOIN (
   SELECT name
   FROM template_category
   WHERE organization IN (${organizations})
   GROUP BY name
   HAVING COUNT(DISTINCT organization) = ${organizations.length}
) r2 ON r1.name = r2.name
WHERE r1.organization IN (${organizations}) AND r1.deleted = 0 AND r1.parent_category IS NULL;
`;
  // console.log(parentTemplateCategoryQuery, "parentTemplateCategoryQuery")
  const [parentTemplateCategoryList] = await db.query(
    parentTemplateCategoryQuery
  );
  const uniqueObj = {};
  parentTemplateCategoryList.forEach((item) => {
    if (!uniqueObj[item.name]) {
      uniqueObj[item.name] = item;
    }
  });
  const uniqueArr = Object.values(uniqueObj);
  // console.log(uniqueArr);
  // return uniqueArr;

  return sendResponse(res, 200, uniqueArr);
};
