import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  getRecord,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import TrainingSchedule from "../sequelize/TrainingScheduleSchema.js";

import fs from "fs/promises";
import path from "path";
import ical from "ical-generator";

export const createUpdateTrainingSchedule = async (req, res) => {
  let { schedule_training, id } = req.body;
  let status = id ? "Updated" : "Created";
  for (let training of schedule_training) {
    training[id ? "updated_by" : "created_by"] = req.user.sessionid;
    const [trainingProgramRecord] = await getRecord(
      "training_need_analysis",
      "id",
      training?.training_program
    );
    training.department =
      training?.department || trainingProgramRecord?.department;
    training.organization =
      training.organization || trainingProgramRecord?.organization;

    // console.log(" training.organization ", training.organization )

    if (!id) {
      const unique_id = await uniqueIdGenerator(
        training.organization,
        training.department,
        "TrainingSchedule",
        "training_schedule",
        "unique_id",
        "unique_id"
      );
      training.unique_id = unique_id;
    }

    training = await encodeAndStringifyFields(training);
    const { query, values } = id
      ? updateQueryBuilder(TrainingSchedule, { ...training, id })
      : createQueryBuilder(TrainingSchedule, training);
    const result = await db.query(query, values);

    await insertActivityLog(
      req.user.sessionid,
      status,
      "Training Schedule",
      id ? id : result[0].insertId
    );
  }

  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getTrainingSchedule = async (req, res) => {
  const { id } = req.params;
  const { training_type } = req.query;
  const trainingFilter = training_type
    ? `AND training_need_analysis.training_type = '${training_type}'`
    : "";
  const condition = await whereCondition({
    table: "training_schedule",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  const searchTableName = [
    "createdUser.name",
    "createdUser.surname",
    // "createdUser.employee_id",
    "createdUser.gender",
    "createdUser.disability",
    "createdUser.race",
    "createdUser.employee_type",
    "internal_dep.name",
    // "roles.name",
    "training_schedule.financial_year",
    " purpose_of_training.name",
    "training_need_analysis.training_name",
    "training_need_analysis.training_type",
    "training_schedule.training_provider",
    "training_schedule.external_provider",
    "CONCAT(createdUser.name , ' ' , createdUser.surname)",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joinsRecord = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = training_schedule.created_by",
    },
    {
      type: "left",
      targetTable: "department AS d",
      onCondition: "d.id = training_schedule.provider_department",
    },
    {
      type: "left",
      targetTable: "training_need_analysis",
      onCondition:
        "training_need_analysis.id = training_schedule.training_program",
    },
    {
      type: "left",
      targetTable: "purpose_of_training",
      onCondition:
        "purpose_of_training.id = training_need_analysis.training_purpose",
    },
    {
      type: "left",
      targetTable: "users as internal_employee",
      onCondition: "internal_employee.id = training_schedule.internal_emp_name",
    },
    {
      type: "left",
      targetTable: "department as internal_dep",
      onCondition: "internal_dep.id = internal_employee.department",
    },
    {
      type: "left",
      targetTable: "roles as internal_role",
      onCondition: "internal_role.id = internal_employee.role",
    },
    {
      type: "left",
      targetTable: "contractor_registration",
      onCondition:
        "contractor_registration.id = training_schedule.company_name",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = training_schedule.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = training_schedule.organization",
    },
  ];

  const joinCondition = await makeJoins(joinsRecord);
  const viewTrainingScheduleQuery = `SELECT training_schedule.* , 
  CONCAT(internal_employee.name , ' ' , internal_employee.surname) as internal_emp,
  internal_dep.name as internal_dep_name, internal_role.name as internal_role_name,
  purpose_of_training.name as training_purpose_name,
  contractor_registration.contractor_name as company_contractor_name,
  department.name as department_name, organization.name as organization_name,
  createdUser.name as created_by_name, createdUser.surname as created_by_surname, createdUser.profile as created_by_profile , d.name AS provider_department_name , training_need_analysis.training_type , 
  training_need_analysis.training_name,  training_need_analysis.training_purpose , training_need_analysis.training_name AS training_program_name FROM training_schedule  ${joinCondition} WHERE training_schedule.deleted = 0 ${trainingFilter} ${searchCondition} ${condition}`;

  let [viewTrainingSchedule] = await db.query(viewTrainingScheduleQuery);
  viewTrainingSchedule = await decodeAndParseFields(viewTrainingSchedule);
  for (const schedule of viewTrainingSchedule) {
    let employees = schedule?.employees || [];
    const provider_department = schedule.provider_department;
    schedule.provider_department_name = "";
    if (provider_department) {
      if (schedule.training_provider === "Internal") {
        const [departmentRecord] = await getRecord(
          "department",
          "id",
          provider_department
        );
        schedule.provider_department_name = departmentRecord?.name || "";
      } else if (schedule.training_provider === "External") {
        schedule.provider_department_name = Number(schedule.provider_department)
          ? ""
          : schedule.provider_department;
      }
    }
    let arr = [];
    for (let i = 0; i < employees.length; i++) {
      const employee = employees[i]?.user_id;
      if (employee) {
        const [empDetails] = await db.query(
          `SELECT users.id , users.name , users.surname , users.unique_id , users.gender , users.disability , users.race , users.employee_type , users.role , users.profile , organization.name AS organization_name FROM users LEFT JOIN organization ON organization.id = users.my_organization WHERE users.id = ${employee}`
        );
        const [roleName] = await db.query(
          `SELECT name FROM roles WHERE id = ${empDetails[0]?.role}`
        );
        employees[i].name = empDetails[0].name;
        employees[i].surname = empDetails[0].surname;
        let obj = {};
        obj.user_id = empDetails[0].id;
        obj.name = empDetails[0].name;
        obj.profile = empDetails[0].profile;
        obj.surname = empDetails[0].surname;
        obj.employee_id = empDetails[0].unique_id;
        obj.gender = empDetails[0].gender;
        obj.disability = empDetails[0].disability;
        obj.race = empDetails[0].race;
        obj.employee_type = empDetails[0].employee_type;
        obj.role_name = roleName[0]?.name;
        obj.training_purpose = schedule.training_purpose;
        obj.training_type = schedule.training_type;
        obj.training_provider = schedule.training_provider;
        obj.training_program_name = schedule.training_program_name;
        obj.training_date = schedule.single_day
          ? schedule.single_day
          : schedule.multiple_day;
        obj.organization = empDetails[0].organization_name;
        arr.push(obj);
      }
    }
    schedule.training_schedule = arr;
  }

  const totalRecord = await countQueryCondition(viewTrainingScheduleQuery);

  return sendResponse(res, 200, viewTrainingSchedule, totalRecord);
};
export const deleteTrainingSchedule = async (req, res) => {
  const { id } = req.params;
  await deleteRecord("training_schedule", id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Training Schedule",
    `This user deleted a Training Schedule Record with id ${id}`
  );
  return sendResponse(res, 200, "Record Deleted Successfully");
};

export const fetchTrainingScheduleData = async (req, res) => {
  const [trainingSchedule] = await db.query(
    `SELECT * FROM training_schedule WHERE deleted = 0`
  );

  // Create a new calendar
  const calendar = ical({
    name: "Meeting Calendar",
    prodId: { company: "Harmony and Help", product: "SSHEQ" },
    timezone: "Asia/Kolkata",
    url: "https://ssheq.thewingshield.com/",
  });

  trainingSchedule.forEach((schedule) => {
    let eventOptions = {
      id: schedule.id,
      organizer: { name: "Harmony & Help", email: "info@harmonyandhelp.com" },
    };

    if (schedule.schedule_type === 0) {
      eventOptions.start = new Date(
        `${schedule.single_day}T${schedule.single_start_time}`
      );
      eventOptions.end = new Date(
        `${schedule.single_day}T${schedule.single_end_time}`
      );
    } else if (schedule.schedule_type === 1) {
      eventOptions.start = new Date(
        `${schedule.multiple_day}T${schedule.multiple_start_time}`
      );
      eventOptions.end = new Date(
        `${schedule.multiple_end_day}T${schedule.multiple_end_time}`
      );
    }

    const event = calendar.createEvent(eventOptions);

    // If the event is recurring, set the recurrence rule
    if (schedule?.repeat_event) {
      const rule = {
        freq: schedule?.repeat_every_cycle?.toUpperCase(), // FREQ
        interval: schedule.repeat_every, // INTERVAL
      };

      // If weekly, add by day
      if (
        schedule.repeat_every_cycle.toLowerCase() === "weekly" &&
        schedule.repeat_every_dwy
      ) {
        rrule.byDay = schedule.repeat_every_dwy
          .split(",")
          .map((day) => day.toUpperCase());
      }

      event.repeating(rrule);
    }
  });

  // Ensure the directory exists
  const calendarDir = path.join(process.cwd(), "public", "calendar");
  await fs.mkdir(calendarDir, { recursive: true });

  // Save the calendar to an .ics file
  const icsContent = calendar.toString();
  const writeFilePath = path.join(calendarDir, "training-schedule-data.ics");

  await fs.writeFile(writeFilePath, icsContent);

  res.status(200).json({
    status: true,
    message: "Training Schedule details saved to .ICS file",
    data: {
      filePath: "public/calendar/training-schedule-data.ics",
    },
  });
};
