import bcrypt from "bcryptjs";
import axios from "axios";
import jwt from "jsonwebtoken";
import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeSingle_statement,
  encodeSingle_statement,
  getOrganizationAccordingToDepartment,
  inWhereFormat,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  tableDataFetch,
  tableRecord,
  updateQueryBuilder,
  uploadFile,
  whereCondition,
  getUserDetails,
  calculateAgeFromLicense,
  makeLoopAndGetMultipleUsersDetails,
  recordIdExist,
  deleteRecord,
  getRecord,
  processesSingleDDRMDocument,
  processUploadDocuments,
  decodeAndParseFields,
  pre_employment_check,
  post_employment_check,
  employeeCheck,
  updateArchiveRepoDoc,
  uniqueIdGenerator,
  getDateRangeForSQL,
  updateTeamMembers,
  getSupervisor,
  getUserListByIds,
  getUserById,
  getManagerAndSuperAdmin,
  getCompanyName,
  getPlatform,
  sendDynamicEmail,
  addUserLimitCheck,
  checkUnifiedUserApi,
  createUnifiedUserApi,
} from "../helper/general.js";
import sendEmail from "../helper/sendEmail.js";
import User from "../sequelize/UserSchema.js";
import { ApiResponse, sendResponse } from "../helper/wrapper.js";
import { ipMapper } from "../middleware/rateLimiting.js";
import UserDocs from "../sequelize/UserDocsSchema.js";
import UserCheckDocs from "../sequelize/UserCheckDocsSchema.js";
import UserPermitAndLicenseDocs from "../sequelize/UserPermitAndLicenseDocsSchema.js";
import { uploadToDDRM } from "../helper/ddrmUploader.js";
import { handleExcessiveFailedLogins } from "../util/clientAppAuthFailureHandler.js";

// Map to store failed login attempts, key is field (email or unique_id), value is count
const failedLoginAttempts = new Map();

// Map to track if email has already been sent for excessive failed logins for a user
const emailSentStatus = new Map(); // Key: field, Value: true (if email sent)

/**Function to login employee with checking email id or unique id*/
export const login = async (req, res) => {
  const { field, password } = req.body;

  /**Function to get users data by email or Employee ID */
  const userDataFetchQuery = `SELECT * FROM users WHERE deleted = "0" AND (email = ? OR unique_id = ?)`;

  const trimmedField = field.trim(); // Trim spaces from input

  const [userDataFetch] = await db.query(userDataFetchQuery, [trimmedField, trimmedField]);
  // console.log("user",userDataFetch)
  /**If users data found properly */
  if (userDataFetch.length > 0) {
    if (userDataFetch[0].password) {
      /**Check that given password is correct or not*/
      const result = await bcrypt.compare(password, userDataFetch[0].password);

      if (result) {
        // Password is correct - Reset failed login attempts count
        failedLoginAttempts.delete(field);

        /**Insert record for activity log */
        await insertActivityLog(
          userDataFetch[0].id,
          "login",
          `${userDataFetch[0].name} ${userDataFetch[0].surname}`,
          userDataFetch[0].id
        );

        /**If Password is correct then create session or token */
        const token = jwt.sign(
          {
            sessionEmail: userDataFetch[0].email,
            sessionid: userDataFetch[0].id,
            sessionidRole: userDataFetch[0].role,
            sessionOrganization: userDataFetch[0].organization,
            sessionMyOrganization: userDataFetch[0].my_organization,
            isSuperAdmin: userDataFetch[0]?.is_super_admin || 0,
          },
          process.env.JWT_KEY,
          { expiresIn: "21600s" }
        );

        /**Get Users Details according to given id*/
        const userDetails = await getUserDetails(userDataFetch[0].id);
        ipMapper.delete(
          req.headers["x-forwarded-for"] || req.connection.remoteAddress
        );
        return res.status(200).json({
          status: true,
          token: token,
          data: userDetails,
        });
      } else {
        // Invalid credentials - Increment failed login attempts count
        const currentAttempts = failedLoginAttempts.get(field) || 0;
        failedLoginAttempts.set(field, currentAttempts + 1);
        console.log(
          `Failed login attempt ${currentAttempts + 1} for field: ${field}`
        );

        if (currentAttempts + 1 > 3) {
          if (!emailSentStatus.has(field)) {
            const mailSentStatus = await handleExcessiveFailedLogins(field);
            if (mailSentStatus) {
              emailSentStatus.set(field, true); // Set the flag after sending email
            }
            return sendResponse(res, 404, "Invalid credentials");
          } else {
            console.log(
              `Email already sent for excessive failed logins for field: ${field}. Skipping email.`
            ); // Log that email is skipped
            return sendResponse(res, 404, "Invalid credentials"); // Still return "Invalid credentials" response even if email is skipped to not reveal email sending status.
          }
        } else {
          return sendResponse(res, 404, "Invalid credentials"); // Standard "Invalid credentials" response for attempts < 3
        }
      }
    } else {
      return sendResponse(res, 404, "You don't have access to login this");
    }
  } else {
    return sendResponse(res, 404, "Email or Unique Id not found");
  }
};

/**Function for forgot password */
export const forgotPassword = async (req, res) => {
  const { field } = req.body;

  /**Function to get users data by email or Employee ID */
  const userDataFetchWithEmail = await tableDataFetch("email", field, "users");
  const userDataFetchWithUniqueNo = await tableDataFetch(
    "unique_id",
    field,
    "users"
  );

  if (userDataFetchWithEmail || userDataFetchWithUniqueNo) {
    const userDataFetch = userDataFetchWithEmail
      ? userDataFetchWithEmail
      : userDataFetchWithUniqueNo;

    const randomOTP = Math.floor(1e5 + Math.random() * 9e5);

    /**Update forgot password otp on users table */
    const [updateOtp] = await db.query(
      `UPDATE users SET otp='${randomOTP}' WHERE id = '${userDataFetch.id}'`
    );

    if (updateOtp) {
      /**Send array in mail function */
      const sendRecordArray = {
        templateFileUrl: "forgot_password_template.html",
        otp: randomOTP,
        name: userDataFetch.name,
        templateName: "Forgot Password",
      };
      const sendMailResponse = await sendEmail(
        "info@harmonyandhelp.com",
        userDataFetch.email,
        "Reset Your Password - One-Time Password (OTP)",
        sendRecordArray
      );

      if (sendMailResponse) {
        return sendResponse(res, 200, "OTP Send Successfully");
      }
    }
  } else {
    return sendResponse(res, 404, "Email address or employee ID not found");
  }
};

/**Forgot Password OTP Verify */
export const forgotPasswordOTPVerify = async (req, res) => {
  const { field, otp } = req.body;

  /**Function to get users data by email or Employee ID */
  const userDataFetchWithEmail = await tableDataFetch("email", field, "users");
  const userDataFetchWithUniqueNo = await tableDataFetch(
    "unique_id",
    field,
    "users"
  );

  if (userDataFetchWithEmail || userDataFetchWithUniqueNo) {
    const userDataFetch = userDataFetchWithEmail
      ? userDataFetchWithEmail
      : userDataFetchWithUniqueNo;

    if (userDataFetch.otp === otp) {
      return sendResponse(res, 200, "OTP Match successfully");
    } else {
      return sendResponse(res, 404, "Invalid OTP");
    }
  } else {
    return sendResponse(res, 404, "Email address not found");
  }
};

/**Function for changing password */
export const passwordChange = async (req, res) => {
  const { field, password } = req.body;
  let hashedPassword = null;
  if (password) {
    /**Make password hash with */
    const salt = bcrypt.genSaltSync(10);
    hashedPassword = bcrypt.hashSync(password, salt);
  }

  /**Update password in user panel*/
  const updateUserPasswordQuery = `UPDATE users SET password = ? WHERE email = ?`;
  await db.query(updateUserPasswordQuery, [hashedPassword, field]);

  return sendResponse(res, 200, "Password updated successfully");
};

export const updateCredentials = async (req, res) => {
  const { id, password } = req.body;
  let hashedPassword = null;
  if (password) {
    /**Make password hash with */
    const salt = bcrypt.genSaltSync(10);
    hashedPassword = bcrypt.hashSync(password, salt);
  }

  /**Update password in user panel*/
  const updateUserPasswordQuery = `UPDATE users SET password = ? WHERE id = ?`;
  await db.query(updateUserPasswordQuery, [hashedPassword, id]);

  return sendResponse(res, 200, "Password updated successfully");
};

export const GetEmployeeData = async (req, res) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "users",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const adminFilter = id ? "" : `AND  users.id != 1`;

  const searchTableName = [
    "CONCAT(users.name, ' ', users.surname)",
    "users.email",
    "users.phone",
    "users.disability",
    "roles.name",
    "cities.name",
    "users.current_address",
    "gender.name",
    "department.name",
    "organization.name",
    "country.name",
    "CONCAT(createdUser.name , ' ' , createdUser.surname)",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "roles",
      onCondition: "roles.id = users.role",
    },
    {
      type: "left",
      targetTable: "role_hierarchy",
      onCondition: "role_hierarchy.id = users.hierarchy_level",
    },
    {
      type: "left",
      targetTable: "gender",
      onCondition: "gender.id = users.gender",
    },
    {
      type: "left",
      targetTable: "users as manager",
      onCondition: "manager.id = users.manager",
    },
    {
      type: "left",
      targetTable: "highest_qualification",
      onCondition: "highest_qualification.id = users.highest_qualification",
    },
    {
      type: "left",
      targetTable: "reason_for_inactive",
      onCondition:
        "reason_for_inactive.id = users.reason_for_employee_becoming_inactive",
    },
    // {
    //   type: "left",
    //   targetTable: "location",
    //   onCondition: "location.id = users.work_location",
    // },
    {
      type: "left",
      targetTable: "language",
      onCondition: "language.id = users.language",
    },
    {
      type: "left",
      targetTable: "cities",
      onCondition: "cities.id = users.city",
    },
    {
      type: "left",
      targetTable: "states",
      onCondition: "states.id = users.state",
    },
    {
      type: "left",
      targetTable: "country",
      onCondition: "country.id = users.country",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = users.my_organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = users.department",
    },
    {
      type: "left",
      targetTable: "license",
      onCondition: "license.id = users.license",
    },
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = users.created_by",
    },
    {
      type: "left",
      targetTable: "users as updatedUser",
      onCondition: "updatedUser.id = users.updated_by",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = users.ddrm_id",
    },
    {
      type: "left",
      targetTable: "banks",
      onCondition: "banks.id = users.bank_name",
    },
    {
      type: "left",
      targetTable: "race",
      onCondition: "race.id = users.race",
    },
    {
      type: "left",
      targetTable: "employee_type",
      onCondition: "employee_type.id = users.employee_type",
    },
  ];

  let createdByCheck = ``;
  // if id is not provided to get all employees data then add this check
  if(!id) {
    createdByCheck = `AND users.created_by = ${req.user.sessionid}`;
  }

  const joinsRecord = await makeJoins(joins);
  const userDataFetchQuery = `SELECT users.*,CONCAT(manager.name , ' ' , manager.surname) as manager_name, language.name as language_name, gender.name as gender_name, highest_qualification.name as highest_qualification_name, users.employment_type as employment_type, employee_type.name as employee_type_name, reason_for_inactive.name as reason_for_employee_becoming_inactive_name,users.my_organization as organization, department.name as department_name,createdUser.id as created_by_id ,roles.description as role_description, roles.year_of_experience , roles.level as role_level,organization.name as organization_name, organization.business_address as work_location, roles.name as role_name, roles.pre_employment, race.name as race_name, roles.post_employment, cities.name as city_name,states.name as state_name ,country.name as country_name, createdUser.name as created_by , createdUser.surname as created_by_surname , createdUser.profile as created_by_profile,repository.url as profile, role_hierarchy.name as hierarchy_level_name, banks.name as bank FROM users ${joinsRecord} WHERE users.deleted = "0" ${createdByCheck} ${adminFilter}  ${searchCondition} ${condition}`;
  // console.log(userDataFetchQuery);
  const [userDataFetch] = await db.query(userDataFetchQuery);

  /**Check that member is already assigned to any team leader or not */

  /**Getting license record */

  for (const licenseRecordData of userDataFetch) {
    if (
      licenseRecordData.license != 0 &&
      licenseRecordData.license != null &&
      JSON.parse(licenseRecordData?.license || "[]").length > 0
    ) {
      const usersLicenseId = await inWhereFormat(licenseRecordData.license);
      const [licenseRecord] = await db.query(
        `SELECT id,name FROM license WHERE id in ${usersLicenseId}`
      );
      licenseRecordData.licenseName = licenseRecord;
    } else {
      licenseRecordData.licenseName = []; // Assigning an empty array if license is 0 or null
    }
  }

  for (const user of userDataFetch) {
    if (user.profile) {
      user.profile = `${process.env.HARMONY_BACKEND_URL}/${user.profile.replace(/\\/g, "/")}`;
    }
    if (user.created_by_profile) {
      user.created_by_profile = `${process.env.HARMONY_BACKEND_URL}/${user.created_by_profile.replace(/\\/g, "/")}`;
    }
    
    const disability = JSON.parse(user?.disability || "[]");
    user.disability_name = [];
    user.disability = [];
    user.disability = disability;
    if (disability.length > 0) {
      const [disabilityList] = await db.query(
        `SELECT name FROM disability WHERE deleted = 0 AND id In (${disability})`
      );
      const disabilityNameList = disabilityList.map((e) => e.name);
      user.disability_name = disabilityNameList;
      user.disability = disability;
    }
    user.emergency_contact_details = JSON.parse(
      user?.emergency_contact_details || "[]"
    );
    for (let i = 0; i < user?.emergency_contact_details?.length; i++) {
      const [item] = await getRecord(
        "relation",
        "id",
        user.emergency_contact_details[i].employee
      );
      user.emergency_contact_details[i].relation_name = item?.name;
    }
    user.pre_employment = JSON.parse(user?.pre_employment || "[]");
    user.post_employment = JSON.parse(user?.post_employment || "[]");
    user.pre_employment_check = JSON.parse(user?.pre_employment_check || "[]");
    let vaccination_records = JSON.parse(user?.vaccination_records || "[]");
    user.completed_step = user.completed_step
      ? JSON.parse(user.completed_step)
      : [];
    if (vaccination_records.length === 0) {
      vaccination_records = [
        {
          title: "",
          file: "",
          ddrm_id: "",
        },
      ];
    }
    user.vaccination_records = vaccination_records;
    user.post_employment_check = JSON.parse(
      user?.post_employment_check || "[]"
    );
    user.role_desc = await decodeSingle_statement(user.role_desc);
    user.permit_license = JSON.parse(user?.permit_license || "[]");
    // console.log(
    //   user?.emergency_contact_details,
    //   "user?.emergency_contact_details"
    // );
    user.role_description = await decodeSingle_statement(user.role_description);
    const id = user.id;
    if (id) {
      const [total] = await db.query(
        `SELECT COUNT(*) AS total, responsible_person
            FROM custom_action_creation 
            WHERE responsible_person = '${id}' AND deleted = '0' 
            GROUP BY responsible_person, custom_action_creation.task_title`
      );
      user.task = total[0]?.total ? total[0]?.total : 0;
    }

    const [files] = await db.query(
      `SELECT user_docs.*, repository.url as file, document_name as title FROM user_docs LEFT JOIN repository ON repository.id = user_docs.ddrm_id LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE user_docs.deleted = 0 AND user_id = ${user.id}`
    );
    user.vaccination_records = files.length
      ? files
      : [{ title: "", ddrm_id: null, file: null }];
    const [permit_license_files] = await db.query(
      `SELECT user_permit_license_docs.*, permit_license_compliance.permit_license_number as permit_license_no, 
      permit_license_compliance.end_date as license_expire_date,
      repository.url as file FROM user_permit_license_docs 
      LEFT JOIN repository ON repository.id = user_permit_license_docs.ddrm_id 
      LEFT JOIN permit_license_compliance ON permit_license_compliance.id = user_permit_license_docs.permit_license_compliance
      WHERE user_permit_license_docs.deleted = 0 AND user_id = ${user.id}`
    );
    user.permit_license_files = permit_license_files.map((doc) => {
      return {
        ...doc,
        checked: Boolean(doc.checked),
      };
    });
    // user.permit_docs = permit_docs.length ? permit_docs : [{ title: "", ddrm_id: null, file: null }];

    const [UserCheckDocs] = await db.query(
      `SELECT user_checks_docs.*, repository.url as file FROM user_checks_docs LEFT JOIN repository ON repository.id = user_checks_docs.ddrm_id WHERE user_checks_docs.deleted = 0 AND user_id = ${user.id}`
    );

    function mergeUserChecks(staticData, userChecks) {
      const mergedData = {};

      for (const [key, value] of Object.entries(staticData)) {
        mergedData[key] = value.map((staticItem) => {
          const matchingDoc = userChecks.find(
            (doc) => doc.name === staticItem.name && doc.type === key
          );

          return matchingDoc
            ? {
              doc_no: matchingDoc.doc_no,
              name: matchingDoc.name,
              ddrm_id: matchingDoc.ddrm_id,
              completed: Boolean(matchingDoc.completed),
              file: matchingDoc.file,
              id: matchingDoc.id,
            }
            : staticItem;
        });
      }

      return mergedData;
    }

    // Merge pre-employment and post-employment data

    user.pre_employment_check = mergeUserChecks(
      pre_employment_check,
      UserCheckDocs
    );
    user.post_employment_check = mergeUserChecks(
      post_employment_check,
      UserCheckDocs
    );
  }

  for (const team of userDataFetch) {
    /** skills parse and get id and name */

    let skills = JSON.parse(team?.skills || "[]");
    let skillsData = [];
    if (skills.length > 0) {
      [skillsData] = await db.query(
        `SELECT id,name FROM skills WHERE id IN (${skills})`
      );
    }
    team.skills = skills;
    team.skillsDetail = skillsData.map((ele) => {
      return {
        id: ele.id,
        name: ele.name,
      };
    });

    const findTeamLeaderQuery = `SELECT users.id as team_leader_id , users.name as team_leader_name , users.surname as team_leader_surname , users.profile as team_leader_profile FROM teams
        LEFT JOIN users ON users.id = teams.team_leader WHERE JSON_CONTAINS(team_members, '${team.id}') AND teams.deleted = 0`;

    const [findTeamLeader] = await db.query(findTeamLeaderQuery);

    findTeamLeader.length > 0 && findTeamLeader.map(team => {
      if (team.team_leader_profile) {
        team.team_leader_profile = `${process.env.HARMONY_BACKEND_URL}/${team.team_leader_profile.replace(/\\/g, '/')}`
      }
    })

    /**Change status according to that members of filter*/

    const userTeamMemberStatus =
      findTeamLeader.length > 0 ? "Assigned" : "Free";
    team.userTeamMemberStatus = userTeamMemberStatus;
    /** Team leader Details */
    team.team_leader = findTeamLeader[0];

    /** add team members details if Employee is team leader */
    const [teamLeaderCheck] = await db.query(
      `SELECT * FROM teams WHERE team_leader = ${team.id} AND deleted = 0`
    );

    if (teamLeaderCheck.length > 0) {
      let members = teamLeaderCheck[0].team_members?.length
        ? teamLeaderCheck[0]?.team_members
        : [-1];
      const [teamMembers] = await db.query(
        `SELECT id,name, surname, profile FROM users WHERE id IN (${members}) AND deleted = "0"`
      );
      team.team_members = teamMembers;
    }
  }
  const result = userDataFetch;
  const totalRecord = await countQueryCondition(userDataFetchQuery);
  return sendResponse(res, 200, result, totalRecord);
};

export const AuthOnlyUsersData = async (req, res) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "users",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const adminFilter = id ? "" : `AND  users.id != 1`;

  const searchTableName = [
    "CONCAT(users.name, ' ', users.surname)",
    "users.email",
    "users.phone",
    "users.disability",
    "roles.name",
    "cities.name",
    "users.current_address",
    "gender.name",
    "department.name",
    "organization.name",
    "country.name",
    "CONCAT(createdUser.name , ' ' , createdUser.surname)",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "roles",
      onCondition: "roles.id = users.role",
    },
    {
      type: "left",
      targetTable: "role_hierarchy",
      onCondition: "role_hierarchy.id = users.hierarchy_level",
    },
    {
      type: "left",
      targetTable: "gender",
      onCondition: "gender.id = users.gender",
    },
    {
      type: "left",
      targetTable: "users as manager",
      onCondition: "manager.id = users.manager",
    },
    {
      type: "left",
      targetTable: "highest_qualification",
      onCondition: "highest_qualification.id = users.highest_qualification",
    },
    {
      type: "left",
      targetTable: "reason_for_inactive",
      onCondition:
        "reason_for_inactive.id = users.reason_for_employee_becoming_inactive",
    },
    // {
    //   type: "left",
    //   targetTable: "location",
    //   onCondition: "location.id = users.work_location",
    // },
    {
      type: "left",
      targetTable: "language",
      onCondition: "language.id = users.language",
    },
    {
      type: "left",
      targetTable: "cities",
      onCondition: "cities.id = users.city",
    },
    {
      type: "left",
      targetTable: "states",
      onCondition: "states.id = users.state",
    },
    {
      type: "left",
      targetTable: "country",
      onCondition: "country.id = users.country",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = users.my_organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = users.department",
    },
    {
      type: "left",
      targetTable: "license",
      onCondition: "license.id = users.license",
    },
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = users.created_by",
    },
    {
      type: "left",
      targetTable: "users as updatedUser",
      onCondition: "updatedUser.id = users.updated_by",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = users.ddrm_id",
    },
    {
      type: "left",
      targetTable: "banks",
      onCondition: "banks.id = users.bank_name",
    },
    {
      type: "left",
      targetTable: "race",
      onCondition: "race.id = users.race",
    },
    {
      type: "left",
      targetTable: "employee_type",
      onCondition: "employee_type.id = users.employee_type",
    },
  ];

  const joinsRecord = await makeJoins(joins);
  const userDataFetchQuery = `SELECT users.*,CONCAT(manager.name , ' ' , manager.surname) as manager_name, language.name as language_name, gender.name as gender_name, highest_qualification.name as highest_qualification_name, users.employment_type as employment_type, employee_type.name as employee_type_name, reason_for_inactive.name as reason_for_employee_becoming_inactive_name,users.my_organization as organization, department.name as department_name,createdUser.id as created_by_id ,roles.description as role_description, roles.year_of_experience , roles.level as role_level,organization.name as organization_name, organization.business_address as work_location, roles.name as role_name, roles.pre_employment, race.name as race_name, roles.post_employment, cities.name as city_name,states.name as state_name ,country.name as country_name, createdUser.name as created_by , createdUser.surname as created_by_surname , createdUser.profile as created_by_profile,repository.url as profile, role_hierarchy.name as hierarchy_level_name, banks.name as bank FROM users ${joinsRecord} WHERE users.deleted = "0" ${adminFilter} AND users.password IS NOT NULL  ${searchCondition} ${condition}`;
  // return console.log(userDataFetchQuery);
  const [userDataFetch] = await db.query(userDataFetchQuery);

  /**Check that member is already assigned to any team leader or not */

  /**Getting license record */

  for (const licenseRecordData of userDataFetch) {
    if (
      licenseRecordData.license != 0 &&
      licenseRecordData.license != null &&
      JSON.parse(licenseRecordData?.license || "[]").length > 0
    ) {
      const usersLicenseId = await inWhereFormat(licenseRecordData.license);
      const [licenseRecord] = await db.query(
        `SELECT id,name FROM license WHERE id in ${usersLicenseId}`
      );
      licenseRecordData.licenseName = licenseRecord;
    } else {
      licenseRecordData.licenseName = []; // Assigning an empty array if license is 0 or null
    }
  }

  for (const user of userDataFetch) {
    const disability = JSON.parse(user?.disability || "[]");
    if (disability.length > 0) {
      const [disabilityList] = await db.query(
        `SELECT name FROM disability WHERE deleted = 0 AND id In (${disability})`
      );
      const disabilityNameList = disabilityList.map((e) => e.name);
      user.disability_name = disabilityNameList;
      user.disability = disability;
    }
    user.emergency_contact_details = JSON.parse(
      user?.emergency_contact_details || "[]"
    );
    for (let i = 0; i < user?.emergency_contact_details?.length; i++) {
      const [item] = await getRecord(
        "relation",
        "id",
        user.emergency_contact_details[i].employee
      );
      user.emergency_contact_details[i].relation_name = item?.name;
    }
    user.pre_employment = JSON.parse(user?.pre_employment || "[]");
    user.post_employment = JSON.parse(user?.post_employment || "[]");
    user.pre_employment_check = JSON.parse(user?.pre_employment_check || "[]");
    let vaccination_records = JSON.parse(user?.vaccination_records || "[]");
    user.completed_step = user.completed_step
      ? JSON.parse(user.completed_step)
      : [];
    if (vaccination_records.length === 0) {
      vaccination_records = [
        {
          title: "",
          file: "",
          ddrm_id: "",
        },
      ];
    }
    user.vaccination_records = vaccination_records;
    user.post_employment_check = JSON.parse(
      user?.post_employment_check || "[]"
    );
    user.role_desc = await decodeSingle_statement(user.role_desc);
    user.permit_license = JSON.parse(user?.permit_license || "[]");
    // console.log(
    //   user?.emergency_contact_details,
    //   "user?.emergency_contact_details"
    // );
    user.role_description = await decodeSingle_statement(user.role_description);
    const id = user.id;
    if (id) {
      const [total] = await db.query(
        `SELECT COUNT(*) AS total, responsible_person
            FROM custom_action_creation 
            WHERE responsible_person = '${id}' AND deleted = '0' 
            GROUP BY responsible_person, custom_action_creation.task_title`
      );
      user.task = total[0]?.total ? total[0]?.total : 0;
    }

    const [files] = await db.query(
      `SELECT user_docs.*, repository.url as file, document_name as title FROM user_docs LEFT JOIN repository ON repository.id = user_docs.ddrm_id LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE user_docs.deleted = 0 AND user_id = ${user.id}`
    );
    user.vaccination_records = files.length
      ? files
      : [{ title: "", ddrm_id: null, file: null }];
    const [permit_license_files] = await db.query(
      `SELECT user_permit_license_docs.*, permit_license_compliance.permit_license_number as permit_license_no, 
      permit_license_compliance.end_date as license_expire_date,
      repository.url as file FROM user_permit_license_docs 
      LEFT JOIN repository ON repository.id = user_permit_license_docs.ddrm_id 
      LEFT JOIN permit_license_compliance ON permit_license_compliance.id = user_permit_license_docs.permit_license_compliance
      WHERE user_permit_license_docs.deleted = 0 AND user_id = ${user.id}`
    );
    user.permit_license_files = permit_license_files.map((doc) => {
      return {
        ...doc,
        checked: Boolean(doc.checked),
      };
    });
    // user.permit_docs = permit_docs.length ? permit_docs : [{ title: "", ddrm_id: null, file: null }];

    const [UserCheckDocs] = await db.query(
      `SELECT user_checks_docs.*, repository.url as file FROM user_checks_docs LEFT JOIN repository ON repository.id = user_checks_docs.ddrm_id WHERE user_checks_docs.deleted = 0 AND user_id = ${user.id}`
    );

    function mergeUserChecks(staticData, userChecks) {
      const mergedData = {};

      for (const [key, value] of Object.entries(staticData)) {
        mergedData[key] = value.map((staticItem) => {
          const matchingDoc = userChecks.find(
            (doc) => doc.name === staticItem.name && doc.type === key
          );

          return matchingDoc
            ? {
              doc_no: matchingDoc.doc_no,
              name: matchingDoc.name,
              ddrm_id: matchingDoc.ddrm_id,
              completed: Boolean(matchingDoc.completed),
              file: matchingDoc.file,
              id: matchingDoc.id,
            }
            : staticItem;
        });
      }

      return mergedData;
    }

    // Merge pre-employment and post-employment data

    user.pre_employment_check = mergeUserChecks(
      pre_employment_check,
      UserCheckDocs
    );
    user.post_employment_check = mergeUserChecks(
      post_employment_check,
      UserCheckDocs
    );
  }

  for (const team of userDataFetch) {
    /** skills parse and get id and name */

    let skills = JSON.parse(team?.skills || "[]");
    let skillsData = [];
    if (skills.length > 0) {
      [skillsData] = await db.query(
        `SELECT id,name FROM skills WHERE id IN (${skills})`
      );
    }
    team.skills = skills;
    team.skillsDetail = skillsData.map((ele) => {
      return {
        id: ele.id,
        name: ele.name,
      };
    });

    const findTeamLeaderQuery = `SELECT users.id as team_leader_id , users.name as team_leader_name , users.surname as team_leader_surname , users.profile as team_leader_profile FROM teams
       LEFT JOIN users ON users.id = teams.team_leader WHERE JSON_CONTAINS(team_members, '${team.id}') AND teams.deleted = 0`;

    const [findTeamLeader] = await db.query(findTeamLeaderQuery);

    /**Change status according to that members of filter*/

    const userTeamMemberStatus =
      findTeamLeader.length > 0 ? "Assigned" : "Free";
    team.userTeamMemberStatus = userTeamMemberStatus;
    /** Team leader Details */
    team.team_leader = findTeamLeader[0];

    /** add team members details if Employee is team leader */
    const [teamLeaderCheck] = await db.query(
      `SELECT * FROM teams WHERE team_leader = ${team.id} AND deleted = 0`
    );

    if (teamLeaderCheck.length > 0) {
      let members = teamLeaderCheck[0].team_members?.length
        ? teamLeaderCheck[0]?.team_members
        : [-1];
      const [teamMembers] = await db.query(
        `SELECT id,name, surname, profile FROM users WHERE id IN (${members}) AND deleted = "0"`
      );
      team.team_members = teamMembers;
    }
  }
  const result = userDataFetch;
  const totalRecord = await countQueryCondition(userDataFetchQuery);
  return sendResponse(res, 200, result, totalRecord);
};
/**function to update the employee details */
export const updateEmployee = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];

  let {
    id,
    organization,
    email,
    end_date,
    role,
    role_description,
    id_number,
    employee_department,
    sidebar_id = 16,
    pre_employment_check,
    post_employment_check,
    permit_license_files = [],
    team_id,
    completed_step,
  } = req.body;
  // console.log("req.body: ", JSON.stringify(req.body));

  if (req.body.active_step == 3 && !req.body.role) {
    return sendResponse(res, 400, "Role is required for step 3.");
  }

  for (let key in pre_employment_check) {
    req.body[key] = pre_employment_check[key];
  }
  for (let key in post_employment_check) {
    req.body[key] = post_employment_check[key];
  }
  let department = req.body?.department || employee_department;
  let age = null;
  id_number = id_number?.toString();
  // const { valid, error } = validateSouthAfricanID(id_number);
  // if (!valid) {
  //   return sendResponse(res, 400, error);
  // }
  // const { valid: v1, error: e1 } = validatePassEMAIL_PORT_NUMBER(passport_number);
  // if (!v1) {
  //   return sendResponse(res, 400, e1);
  // }
  if (id_number) {
    age = await calculateAgeFromLicense(id_number);
  }

  const documents = await processUploadDocuments(
    req,
    sidebar_id,
    "vaccination_records"
  );

  if (id && documents.length) {
    const deleteQuery = `UPDATE user_docs SET deleted = 1 where deleted = 0 AND user_id = ?`;
    await db.query(deleteQuery, [id]);
  }

  req.body.age = age;
  const profile = req.files && req.files.profile;
  req.body.role_description = await decodeSingle_statement(role_description);

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  // console.log('organization: ', organization);

  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId =
      recordAccordingToOrganization[0]?.organization || organizationId;
  }
  req.body.organization = [organizationId];
  // console.log('req.body.organization: ', req.body.organization);

  let userDataFetch = [];
  if (email) {
    [userDataFetch] = await db.query(
      `SELECT * FROM users WHERE deleted = "0" AND email = ? AND id != ?`,
      [email, id]
    );

    if (userDataFetch.length > 0) {
      return sendResponse(res, 404, "These details already exists");
    }

    let [userData] = await db.query(
      `SELECT unified_user_uique_id FROM users WHERE deleted = "0" AND id = ?`,
      [id]
    );
    const uniqueId = userData[0]?.unified_user_uique_id;
    const bodyData = {
      email,
      name: req.body.name,
      surname: req.body.surname,
      phone: req.body.phone,
      profile: req.body.profile,
      uniqueId: uniqueId,
    };
    await checkUnifiedUserApi(bodyData);
  }

  /**Check that profile has been uploaded or not */
  if (profile) {
    // const updatePath = await uploadFile("users", profile);
    const ddrm_id = await processesSingleDDRMDocument(
      "users",
      sidebar_id,
      profile,
      req
    );
    req.body.ddrm_id = ddrm_id;
    const [repository] = await db.query(
      `SELECT url FROM repository WHERE id = ${req.body.ddrm_id}`
    );
    req.body.profile = repository[0].url;
    const oldFilePath = userDataFetch[0]?.profile;

    /**unlink the old image */
    if (
      oldFilePath &&
      fs.existsSync(process.cwd() + "/public/" + oldFilePath)
    ) {
      await fs.promises.unlink(process.cwd() + "/public/" + oldFilePath);
    }
  }

  /**Function to edit employee details according to that field */
  const { query, values } = updateQueryBuilder(User, req.body);
  const [updateUser] = await db.query(query, values);
  if (end_date) {
    await db.query(`UPDATE users SET end_date='${end_date}' WHERE id=${id}`);
  }

  for (let item of permit_license_files) {
    item.user_id = id;
    if (item?.id) {
      const { query: docQuery, values: docValues } = updateQueryBuilder(
        UserPermitAndLicenseDocs,
        item
      );
      await db.query(docQuery, docValues);
    } else {
      delete item?.id;
      const { query: docQuery, values: docValues } = createQueryBuilder(
        UserPermitAndLicenseDocs,
        item
      );
      await db.query(docQuery, docValues);
    }
  }

  if (role && email != "superadmin@sartiaglobal.com") {
    const [permissionsRecord] = await db.query(
      `SELECT permission FROM roles WHERE id= ${role}`
    );

    if (permissionsRecord[0]?.permission) {
      const updatePermissionQuery = `UPDATE users SET permission = ? WHERE id = ?`;
      const updatePermissionQueryValues = [permissionsRecord[0].permission, id];
      await db.query(updatePermissionQuery, updatePermissionQueryValues);
    }
  }

  if (organizationId && organizationId != "undefined") {
    await db.query(
      `UPDATE users SET my_organization='${organizationId}' WHERE id=${id}`
    );
  }
  if (department) {
    await db.query(
      `UPDATE users SET department='${department}' WHERE id=${id}`
    );
  }
  // get Role details
  const [roleRecord] = await getRecord("roles", "id", role);

  // return console.log("documents: ", documents);
  if (documents || Array.isArray(documents)) {
    // Iterate through uploaded files
    for (const file of documents) {
      // console.log("file", file);
      const title = file.title; // Use file name as title
      const ddrm_id = file?.ddrm_id;
      const id_doc = file.id;
      // Move the file to the public folder
      // Save file details for database
      const { query: docQuery, values: docValues } = createQueryBuilder(
        UserDocs,
        {
          doc_no: id_doc,
          user_id: id,
          title,
          ddrm_id,
        }
      );

      await db.query(docQuery, docValues);
    }
  }

  for (let empCheck of employeeCheck) {
    // console.log("empCheck: ", empCheck);
    const items = req.body[empCheck];
    if (items && items.length > 0) {
      const documents = await processUploadDocuments(req, sidebar_id, empCheck);
      // return console.log("documents: ", documents);
      // console.log("documents: ", documents);
      // if (documents.length) {
      //   const deleteQuery = `UPDATE user_checks_docs SET deleted = 1 where deleted = 0 AND user_id = ? AND type = ?`;
      //   await db.query(deleteQuery, [id, empCheck]);
      // }
      if (documents || Array.isArray(documents)) {
        // Iterate through uploaded files
        for (let i = 0; i < documents.length; i++) {
          const file = documents[i];
          const check_doc_id = file.id;
          // console.log("file;+++++++++>>>>>>", file);
          const name = file.name;
          const ddrm_id = file?.ddrm_id;
          // console.log("name: ", name);
          // console.log("ddrm_id: +++++++++>>>>>>", ddrm_id);
          const id_doc = file.id || i + 1;
          const type = file.key;
          // console.log("type: ", type);
          const completed = file.completed;
          // Move the file to the public folder
          // Save file details for database
          const { query: docQuery, values: docValues } = check_doc_id
            ? updateQueryBuilder(UserCheckDocs, {
              id: check_doc_id,
              doc_no: id_doc,
              user_id: id,
              name,
              ddrm_id,
              type,
              completed,
            })
            : createQueryBuilder(UserCheckDocs, {
              doc_no: id_doc,
              user_id: id,
              name,
              ddrm_id,
              type,
              completed,
            });

          const resp = await db.query(docQuery, docValues);
          // console.log("resp: ", resp);
        }
      }
    }
  }

  if (team_id && id) {
    await updateTeamMembers(team_id, id);
  }

  const [user] = await db.query(
    "SELECT completed_step FROM users WHERE id = ?",
    [id]
  );
  const completedStep = JSON.parse(user[0]?.completed_step);
  if (completed_step[5] && !completedStep[5]) {
    const usersToSend = await getManagerAndSuperAdmin(req.body.manager);
    const companyWebsite = process.env.HARMONY_FRONTEND_URL;
    const companyName = await getCompanyName(req.body.organization);
    const employeeName = req.body?.name + " " + req.body?.surname;
    const platform = await getPlatform();

    for (let user of usersToSend) {
      await sendDynamicEmail({
        to: user.email,
        subject: `Successful Creation of Employee Record for ${platform.platform_name}`,
        data: {
          templateFileUrl: "mail_for_employee_creation_template.html",
          employee_name: employeeName,
          name: user.name,
          company_name: companyName,
          company_website: companyWebsite,
          templateName: "Employee Creation",
        },
      });
    }
  }
  /**Insert Activity  */
  await insertActivityLog(req.user.sessionid, "update", "Employee", id);
  return res.status(200).json({
    status: true,
    message: "Employee details updated Successfully",
    id,
    pre_employment: JSON.parse(roleRecord?.pre_employment || "[]"),
    post_employment: JSON.parse(roleRecord?.post_employment || "[]"),
    pre_employment_check: JSON.parse(roleRecord?.pre_employment_check || "[]"),
    post_employment_check: JSON.parse(
      roleRecord?.post_employment_check || "[]"
    ),
  });
  // return sendResponse(res, 200, "Employee details updated Successfully");
};

/**Function to create employee new function*/
export const createEmployee = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];

  let {
    organization,
    employeeId,
    role,
    email,
    end_date,
    role_description,
    employee_department,
    id_number,
    passport_number,
    sidebar_id = 16,
    permit_license_files = [],
    team_id,
  } = req.body;
  let department = req.body?.department || employee_department;
  let age = null;
  id_number = id_number?.toString();
  // a check for max user limit
  const result = await addUserLimitCheck();
  if (result.status === false) {
    return sendResponse(res, 400, result.message);
  }
  if (req.body.active_step == 3 && !req.body.role) {
    return sendResponse(res, 400, "Role is required for step 3.");
  }

  // const { valid, error } = validateSouthAfricanID(id_number);
  // if (!valid) {
  //   return sendResponse(res, 400, error);
  // }
  // const { valid: v1, error: e1 } = validatePassEMAIL_PORT_NUMBER(passport_number);
  // if (!v1) {
  //   return sendResponse(res, 400, e1);
  // }
  if (id_number) {
    age = await calculateAgeFromLicense(id_number);
  }
  req.body.age = age;

  const profile = req.files && req.files.profile;

  const documents = await processUploadDocuments(
    req,
    sidebar_id,
    "vaccination_records"
  );

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId =
      recordAccordingToOrganization[0]?.organization || organizationId;
  }

  /**Check that this user name or email id is already exist or not */
  if (email) {
    try {
      const response = await axios.get(
        `${process.env.SUPER_ADMIN_BACKEND_URL}/api/auth/check-email?email=${email}`
      );
    } catch (err) {
      if (err?.response?.data?.status === false) {
        return sendResponse(
          res,
          404,
          "Employee with this email already exists"
        );
      }
    }
    const userDataFetch = await tableRecord("email", email, "users");

    if (userDataFetch.length > 0) {
      return sendResponse(res, 404, "Employee with this email already exists");
    }
  }
  /**Find last automatic inserted unique id */
  const uniqueIdFetchQuery = `SELECT unique_id FROM users WHERE unique_id_status = "automatic" ORDER BY id DESC LIMIT 1`;
  const [uniqueIdFetch] = await db.query(uniqueIdFetchQuery);

  let newSeries = "";
  let newSeriesStatus = "automatic";
  newSeries = await uniqueIdGenerator(
    organizationId,
    "",
    "",
    "users",
    "unique_id"
  );
  req.body.role_description = await encodeSingle_statement(role_description);
  req.body.unique_id = newSeries;
  req.body.unique_id_status = newSeriesStatus;

  req.body.role_description = await encodeSingle_statement(role_description);
  req.body.unique_id = newSeries;
  req.body.unique_id_status = newSeriesStatus;

  const orrArr = [parseInt(organizationId)];
  req.body.my_organization = organizationId;
  req.body.created_by = req.user.sessionid;
  req.body.organization = orrArr;
  /**Insert new users */
  const userOrganization = [organizationId];

  /**Check that profile has been uploaded or not */
  if (profile) {
    // const updatePath = await uploadFile("users", profile);
    // const updateUserQuery = `UPDATE users SET profile='${updatePath}' WHERE id=${insertUsers.insertId}`;
    // await db.query(updateUserQuery);
    const ddrm_id = await processesSingleDDRMDocument(
      "users",
      sidebar_id,
      profile,
      req
    );
    req.body.ddrm_id = ddrm_id;
    const [repository] = await db.query(
      `SELECT url FROM repository WHERE id = ${req.body.ddrm_id}`
    );
    req.body.profile = repository[0].url;
  }
  const { query, values } = createQueryBuilder(User, req.body);
  const [insertUsers] = await db.query(query, values);

  if (end_date) {
    await db.query(
      `UPDATE users SET end_date='${end_date}' WHERE id=${insertUsers.insertId}`
    );
  }

  if (role) {
    const [permissionsRecord] = await db.query(
      `SELECT permission FROM roles WHERE id= ${role}`
    );
    const updatePermissionQuery = `UPDATE users SET permission = ? WHERE id = ?`;
    const updatePermissionQueryValues = [
      permissionsRecord[0].permission,
      insertUsers.insertId,
    ];
    const [result] = await db.query(
      updatePermissionQuery,
      updatePermissionQueryValues
    );
  }

  if (team_id && insertUsers?.insertId) {
    await updateTeamMembers(team_id, insertUsers.insertId);
  }
  // console.log("permit_license_files: ", permit_license_files);
  for (let item of permit_license_files) {
    // const name = item.name;

    // const fileKey = `permit_license_files[${name.replace(/\s+/g, "_")}]`;
    // const file = req.files?.[fileKey];
    // const ddrm_id = await processesSingleDDRMDocument("user_permit_license_docs", sidebar_id, file, req);
    // item.ddrm_id = ddrm_id;
    item.user_id = insertUsers.insertId;
    const { query: docQuery, values: docValues } = createQueryBuilder(
      UserPermitAndLicenseDocs,
      item
    );
    await db.query(docQuery, docValues);
  }

  if (documents || Array.isArray(documents)) {
    // Iterate through uploaded files
    for (const file of documents) {
      const title = file.title; // Use file name as title
      const ddrm_id = file?.ddrm_id;
      const id_doc = file.id;
      // Move the file to the public folder
      // Save file details for database
      const { query: docQuery, values: docValues } = createQueryBuilder(
        UserDocs,
        {
          doc_no: id_doc,
          user_id: insertUsers.insertId,
          title,
          ddrm_id,
        }
      );

      await db.query(docQuery, docValues);
    }
  }
  for (let empCheck of employeeCheck) {
    const items = req.body[empCheck];
    if (items && items.length > 0) {
      const documents = await processUploadDocuments(req, sidebar_id, empCheck);
      if (documents || Array.isArray(documents)) {
        // Iterate through uploaded files
        for (let i = 0; i < documents.length; i++) {
          const file = documents[i];
          const name = file.name;
          const ddrm_id = file?.ddrm_id;
          const id_doc = file.id || i + 1;
          const type = file.key;
          const completed = file.completed;
          // Move the file to the public folder
          // Save file details for database
          const { query: docQuery, values: docValues } = createQueryBuilder(
            UserCheckDocs,
            {
              doc_no: id_doc,
              user_id: insertUsers.insertId,
              name,
              ddrm_id,
              type,
              completed,
            }
          );

          await db.query(docQuery, docValues);
        }
      }
    }
  }

  /**Insert Activity  */
  await insertActivityLog(
    req.user.sessionid,
    "create",
    "Employee",
    insertUsers.insertId
  );
  //  add record in super admin unified user table for unified user and update that record in users table unified_user_uique_id
  const unifiedUserResponse = await createUnifiedUserApi(insertUsers.insertId);
  if (unifiedUserResponse) {
    console.log("Unified user created successfully");
  } else {
    console.log("Unified user not created");
  }
  return res.status(200).json({
    status: true,
    message: "Record created successfully",
    id: insertUsers.insertId,
  });
};

/**Function to create employee old function*/
// export const createEmployee = async (req, res) => {
//   const {
//     organization,
//     employeeId,
//     role,
//     email,
//     end_date,
//     role_description,
//     department,
//     id_number,
//   } = req.body;

//   let age = null;
//   const {valid , error} = validateSouthAfricanIDOrPassport(id_number);
//   if(!valid){
//     return sendResponse(res, 400, error);
//   }
//   if (id_number) {
//     age = await calculateAgeFromLicense(id_number);
//   }
//   req.body.age = age;

//   const profile = req.files && req.files.profile;

//   /**Check record if organization is not coming then fetch organization according to department */
//   let organizationId = organization;
//   if (department) {
//     const recordAccordingToOrganization =
//       await getOrganizationAccordingToDepartment(department);
//     organizationId = recordAccordingToOrganization[0].organization;
//   }

//   /**Check that this user name or email id is already exist or not */
//   if (email) {
//     const userDataFetch = await tableRecord("email", email, "users");

//     if (userDataFetch.length > 0) {
//       return sendResponse(res, 404, "These details already exists");
//     }
//   }
//   /**Find last automatic inserted unique id */
//   const uniqueIdFetchQuery = `SELECT unique_id FROM users WHERE unique_id_status = "automatic" ORDER BY id DESC LIMIT 1`;
//   const [uniqueIdFetch] = await db.query(uniqueIdFetchQuery);

//   let newSeries = "";
//   let newSeriesStatus = "";
//   if (employeeId) {
//     /**Check that enter unique number is different */
//     const checkUniqueIdQuery = `SELECT unique_id FROM users WHERE unique_id = ?`;
//     const [checkUniqueId] = await db.query(checkUniqueIdQuery, [employeeId]);

//     if (checkUniqueId.length > 0) {
//       return sendResponse(
//         res,
//         400,
//         "An employee with this employee number already exists , please try another one"
//       );
//     }
//     newSeries = employeeId;
//     newSeriesStatus = "manual";
//   } else {
//     /**Check the last unique id store in db */
//     const checkLastUniqueIdQuery = `SELECT id,unique_id FROM users ORDER by id desc LIMIT 1;`;
//     const [checkLastUniqueId] = await db.query(checkLastUniqueIdQuery);

//     const originalSeries = uniqueIdFetch[0]?.unique_id;
//     newSeries = originalSeries.match(/^([A-Za-z]+)(\d+)$/)
//       ? originalSeries.replace(/(\d+)$/, (_, match) =>
//           String(parseInt(match, 10) + 1).padStart(match.length, "0")
//         )
//       : null;

//     /**if last inserted unique id match with created unique id then replace it  */
//     if (checkLastUniqueId[0].unique_id == newSeries) {
//       newSeries = originalSeries.match(/^([A-Za-z]+)(\d+)$/)
//         ? originalSeries.replace(/(\d+)$/, (_, match) =>
//             String(parseInt(match, 10) + 2).padStart(match.length, "0")
//           )
//         : null;
//     }
//     newSeriesStatus = "automatic";
//   }

//   req.body.role_description = await encodeSingle_statement(role_description);
//   req.body.unique_id = newSeries;
//   req.body.unique_id_status = newSeriesStatus;

//   const orrArr = [parseInt(organizationId)];
//   req.body.my_organization = organizationId;
//   req.body.created_by = req.user.sessionid;
//   req.body.organization = orrArr;
//   /**Insert new users */
//   const userOrganization = [organizationId];
//   const { query, values } = createQueryBuilder(User, req.body);

//   const [insertUsers] = await db.query(query, values);

//   /**Check that profile has been uploaded or not */
//   if (profile) {
//     const updatePath = await uploadFile("profile", profile);
//     const updateUserQuery = `UPDATE users SET profile='${updatePath}' WHERE id=${insertUsers.insertId}`;
//     await db.query(updateUserQuery);
//   }
//   if (end_date) {
//     await db.query(
//       `UPDATE users SET end_date='${end_date}' WHERE id=${insertUsers.insertId}`
//     );
//   }

//   if (role) {
//     const [permissionsRecord] = await db.query(
//       `SELECT permission FROM roles WHERE id= ${role}`
//     );
//     const updatePermissionQuery = `UPDATE users SET permission = ? WHERE id = ?`;
//     const updatePermissionQueryValues = [
//       permissionsRecord[0].permission,
//       insertUsers.insertId,
//     ];
//     const [result] = await db.query(
//       updatePermissionQuery,
//       updatePermissionQueryValues
//     );
//   }

//   /**Insert Activity  */
//   await insertActivityLog(
//     req.user.sessionid,
//     "create",
//     "Employee",
//     insertUsers.insertId
//   );

//   return sendResponse(res, 200, "Record created successfully");
// };

/**Function to change user password */
export const userPasswordChange = async (req, res) => {
  const { id, new_password, old_password } = req.body;

  const [userDataFetch] = await db.query(
    `SELECT password FROM users where id = ${id}`
  );

  /**Check that given password is correct or not*/
  const result = await bcrypt.compare(old_password, userDataFetch[0].password);
  if (result) {
    /**Make password hash with */
    const salt = bcrypt.genSaltSync(10);
    const hashedPassword = bcrypt.hashSync(new_password, salt);
    /**Update password in user panel*/
    const updateUserPasswordQuery = `UPDATE users SET password = ? WHERE id = ?`;
    const [updateUserPassword] = await db.query(updateUserPasswordQuery, [
      hashedPassword,
      id,
    ]);
    return sendResponse(res, 200, "password Updated successfully");
  } else {
    return sendResponse(res, 400, "Old Password is not correct");
  }
};

/**Function to get organization employee chart */
export const getOrganizationChart = async (req, res) => {
  /**Check that this user name or email id is already exist or not */
  const recordFirstLevelRecordQuery = `
      SELECT users.id, users.name, roles.name AS role_name
      FROM users
      LEFT JOIN roles ON roles.id = users.role
      WHERE users.deleted = "0" AND roles.deleted = "0" AND roles.level = "1"
    `;
  const [recordFirstLevelRecord] = await db.query(recordFirstLevelRecordQuery);

  const teamMembersByLeader = [];

  // Iterate over each first level user
  for (const user of recordFirstLevelRecord) {
    // Query to fetch team members assigned as team leaders for the current user
    const teamMembersQuery = `
        SELECT created_by,team_leader,team_members as participants
        FROM teams
        WHERE teams.deleted = "0" AND team_leader = ${user.id}
      `;
    const [teamMembers] = await db.query(teamMembersQuery);
    // console.log(teamMembers.length > 0 ? teamMembers[0].team_members : []);
    let teamMember = [];
    if (teamMembers.length > 0) {
      teamMember = await makeLoopAndGetMultipleUsersDetails(
        teamMembers,
        "team_member"
      );
    }

    // Store the user along with their team members in an object
    const userWithTeamMembers = {
      user: user,
      teamMembers: teamMember,
    };

    // Push the object to the array
    teamMembersByLeader.push(userWithTeamMembers);
  }

  return sendResponse(res, 200, teamMembersByLeader);
};

/**Function for get active login employee data */
export const GetActiveLoginEmployeeData = async (req, res) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "users",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  /** If value come with any search condition then search that word */
  let searchCondition = req.query.search
    ? `AND (CONCAT(users.name, ' ', users.surname) LIKE '%${req.query.search}%' OR users.email LIKE '%${req.query.search}%' OR users.phone LIKE '%${req.query.search}%' OR roles.name LIKE '%${req.query.search}%' OR cities.name LIKE '%${req.query.search}%' OR users.current_add LIKE '%${req.query.search}%' OR users.gender LIKE '%${req.query.search}%' OR department.name LIKE '%${req.query.search}%' OR country.name LIKE '%${req.query.search}%' OR organization.name LIKE '%${req.query.search}%' AND CONCAT(created.name , ' ' , created.surname) LIKE '%${req.query.search}%' ) `
    : "";

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "roles",
      onCondition: "roles.id = users.role",
    },

    {
      type: "left",
      targetTable: "cities",
      onCondition: "cities.id = users.city",
    },
    {
      type: "left",
      targetTable: "states",
      onCondition: "states.id = users.state",
    },
    {
      type: "left",
      targetTable: "country",
      onCondition: "country.id = users.country",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = users.my_organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = users.department",
    },
    {
      type: "left",
      targetTable: "license",
      onCondition: "license.id = users.license",
    },
    {
      type: "left",
      targetTable: "users as created",
      onCondition: "created.id = users.created_by",
    },
  ];

  const joinsRecord = await makeJoins(joins);
  const userDataFetchQuery = `SELECT users.license,users.skills, department.name as department_name, users.id,users.unique_id,users.unique_id_status,users.name,users.surname,users.email,users.password,users.profile,users.phone,users.id_number,users.joining_date,users.end_date,users.gender,users.disability,users.race,users.employee_type,users.highest_qualification,users.name_of_qualification,users.country,users.state,users.city,users.current_add,users.role,users.role_desc,users.hierarchy,users.department,users.accountability,users.otp,users.my_organization as organization,users.permission,users.special_permission,users.notification_sender,users.created_by as created_by_id ,roles.description as role_description ,roles.hierarchy as role_hierarchy , roles.level as role_level,organization.name as organization_name,roles.name as role_name,cities.name as city_name,states.name as state_name ,country.name as country_name, created.name as created_by , created.surname as created_by_surname , created.profile as created_by_profile FROM users ${joinsRecord} WHERE users.deleted = "0" AND users.password IS NOT NULL AND users.is_super_admin = 0 ${searchCondition} ${condition}`;
  const [userDataFetch] = await db.query(userDataFetchQuery);

  /**Check that member is already assigned to any team leader or not */

  /**Getting license record */

  for (const licenseRecordData of userDataFetch) {
    if (licenseRecordData.license != 0 && licenseRecordData.license != null) {
      const usersLicenseId = await inWhereFormat(licenseRecordData.license);
      const [licenseRecord] = await db.query(
        `SELECT id,name FROM license WHERE id in ${usersLicenseId}`
      );
      licenseRecordData.licenseName = licenseRecord;
    } else {
      licenseRecordData.licenseName = []; // Assigning an empty array if license is 0 or null
    }
  }

  for (const user of userDataFetch) {
    user.role_desc = await decodeSingle_statement(user.role_desc);
  }

  for (const team of userDataFetch) {
    /** skills parse and get id and name */

    let skills = JSON.parse(team?.skills || "[]");
    let skillsData = [];
    if (skills.length > 0) {
      [skillsData] = await db.query(
        `SELECT id,name FROM skills WHERE id IN (${skills})`
      );
    }
    team.skills = skills;
    team.skillsDetail = skillsData.map((ele) => {
      return {
        id: ele.id,
        name: ele.name,
      };
    });

    const findTeamLeaderQuery = `SELECT users.id as team_leader_id , users.name as team_leader_name , users.surname as team_leader_surname , users.profile as team_leader_profile FROM teams
     LEFT JOIN users ON users.id = teams.team_leader WHERE JSON_CONTAINS(team_members, '${team.id}') AND teams.deleted = 0`;
    const [findTeamLeader] = await db.query(findTeamLeaderQuery);

    /**Change status according to that members of filter*/

    const userTeamMemberStatus =
      findTeamLeader.length > 0 ? "Assigned" : "Free";
    team.userTeamMemberStatus = userTeamMemberStatus;
    /** Team leader Details */
    team.team_leader = findTeamLeader[0];

    /** add team members details if Employee is team leader */
    const [teamLeaderCheck] = await db.query(
      `SELECT * FROM teams WHERE team_leader = ${team.id} AND deleted = 0`
    );

    if (teamLeaderCheck.length > 0) {
      let members = teamLeaderCheck[0].team_members.length
        ? teamLeaderCheck[0].team_members
        : [-1];

      const [teamMembers] = await db.query(
        `SELECT name, surname, profile FROM users WHERE id IN (${members}) AND deleted = "0"`
      );
      team.teamMembers = teamMembers;
    }
  }
  const result = userDataFetch;
  const totalRecord = await countQueryCondition(userDataFetchQuery);
  return sendResponse(res, 200, result, totalRecord);
};

/**function to import Employee data*/
export const importEmployeeData = async (req, res) => {
  const { organization, data } = req.body;
  const alreadyExistData = [];

  for (let i = 0; i < data.length; i++) {
    let employee = data[i];
    const {
      employeeId,
      name,
      surname,
      email,
      phone,
      id_number,
      joining_date,
      end_date,
      gender,
      disability,
      race,
      employee_type,
      highest_qualification,
      name_of_qualification,
      country,
      state,
      city,
      current_add,
      role,
      accountability,
      department,
      license,
    } = employee;
    const emailCheck =
      (await tableRecord("email", email, "users")).length > 0 ? true : false;
    /** Check if role dependent data exists or not */
    const organizationCheck = await recordIdExist("organization", organization);

    const roleCheck = await recordIdExist("roles", role);

    const licenseCheck = true || (await recordIdExist("license", license));

    let departmentCheck = true;
    if (department) {
      departmentCheck = await recordIdExist("department", department);
    }

    /**Find last automatic inserted unique id */
    const uniqueIdFetchQuery = `SELECT unique_id FROM users WHERE unique_id_status = "automatic" ORDER BY id DESC LIMIT 1`;
    const [uniqueIdFetch] = await db.query(uniqueIdFetchQuery);
    let uniqueIdCheck = true;
    let newSeries = "";
    let newSeriesStatus = "";
    if (employeeId) {
      /**Check that enter unique number is different */
      const checkUniqueIdQuery = `SELECT unique_id FROM users WHERE unique_id = ?`;
      const [checkUniqueId] = await db.query(checkUniqueIdQuery, [employeeId]);

      if (checkUniqueId.length > 0) {
        uniqueIdCheck = false;
      }

      newSeries = employeeId;
      newSeriesStatus = "manual";
    } else {
      /**Check the last unique id store in db */
      const checkLastUniqueIdQuery = `SELECT id,unique_id FROM users ORDER by id desc LIMIT 1;`;
      const [checkLastUniqueId] = await db.query(checkLastUniqueIdQuery);

      const originalSeries = uniqueIdFetch[0].unique_id;
      newSeries = originalSeries.match(/^([A-Za-z]+)(\d+)$/)
        ? originalSeries.replace(/(\d+)$/, (_, match) =>
          String(parseInt(match, 10) + 1).padStart(match.length, "0")
        )
        : null;

      /**if last inserted unique id match with created unique id then replace it  */
      if (checkLastUniqueId[0].unique_id == newSeries) {
        newSeries = originalSeries.match(/^([A-Za-z]+)(\d+)$/)
          ? originalSeries.replace(/(\d+)$/, (_, match) =>
            String(parseInt(match, 10) + 2).padStart(match.length, "0")
          )
          : null;
      }
      newSeriesStatus = "automatic";
    }

    if (emailCheck) {
      alreadyExistData.push({
        message: `Employee With email already exists at index ${i + 1}`,
      });
    } else if (!departmentCheck) {
      alreadyExistData.push({
        message: `Department not found at index ${i + 1}`,
      });
    } else if (!licenseCheck) {
      alreadyExistData.push({
        message: `License not found at index ${i + 1}`,
      });
    } else if (!roleCheck) {
      alreadyExistData.push({
        message: `Role not found at index ${i + 1}`,
      });
    } else if (!uniqueIdCheck) {
      alreadyExistData.push({
        message: `Employee With unique id already exists at index ${i + 1}`,
      });
    } else if (!organizationCheck) {
      alreadyExistData.push({
        message: `Organization not found at index ${i + 1}`,
      });
    } else {
      const insertUsersQuery = `
      INSERT INTO users(unique_id, unique_id_status, name, surname, email, phone, id_number,joining_date, end_date, gender, disability, race, employee_type, highest_qualification,name_of_qualification, country, state, city, current_add, role, license, accountability, my_organization, organization, department, created_by)VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;

      const insertUsersValues = [
        newSeries,
        newSeriesStatus,
        name,
        surname,
        email,
        phone,
        id_number,
        joining_date,
        end_date ?? null,
        gender,
        disability,
        race,
        employee_type,
        highest_qualification,
        name_of_qualification,
        country,
        state,
        city,
        current_add,
        role,
        license,
        JSON.stringify(accountability),
        organization,
        JSON.stringify([organization]),
        department ?? null,
        req.user.sessionid,
      ];

      const [insertUsers] = await db.query(insertUsersQuery, insertUsersValues);

      /**Insert Activity  */
      insertActivityLog(
        req.user.sessionid,
        "create",
        "Employee",
        insertUsers.insertId
      );
    }
  }
  if (alreadyExistData.length > 0) {
    return res.status(500).json({
      status: false,
      message: "Few rows not inserted due to invalid data",
      errorMessage: alreadyExistData,
    });
  } else {
    return res.status(200).json({
      status: true,
      message: "Record created successfully",
      errorMessage: alreadyExistData,
    });
  }
};

/**Function to delete employee */
export const deleteEmployee = async (req, res) => {
  const { id } = req.params;
  const [getUser] = await db.query(
    `SELECT * FROM users WHERE deleted = "0" AND id = ${id}`
  );
  if (!getUser[0]) {
    return sendResponse(res, 404, "User Not Found");
  }
  if (getUser[0].is_super_admin == 1) {
    return sendResponse(res, 400, "You Can't Delete Super Admin");
  }
  const deleteRecordQuery = await deleteRecord("users", id);
  if (deleteRecordQuery?.status) {
    /**Insert record for activity log */
    insertActivityLog(req.user.sessionid, "delete", "Users", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

function validateSouthAfricanIDOrPassport(id) {
  // First, check if the input is an alphanumeric passport number or numeric ID number
  if (isNaN(id)) {
    // Passport Number (Alphanumeric): If it's not numeric, assume it's a passport number.
    return validatePassEMAIL_PORT_NUMBER(id);
  } else {
    // South African ID Number (Numeric): Apply the ID validation
    return validateSouthAfricanID(id);
  }
}

// Function to validate South African ID (13 digits)
function validateSouthAfricanID(id) {
  // Ensure ID is a string and contains exactly 13 digits
  if (id.length !== 13 || !/^\d+$/.test(id)) {
    return { valid: false, error: "ID must contain exactly 13 digits" };
  }

  // Extract parts of the ID
  const birthDate = id.substring(0, 6);
  const sequenceNumber = id.substring(6, 10);
  const citizenship = id[10];
  const racialClassification = id[11];
  const checksum = id[12];

  // Validate birth date (YYMMDD)
  if (!isValidDate(birthDate)) {
    return { valid: false, error: "Invalid date of birth" };
  }

  // Validate sequence number (SSSS)
  if (parseInt(sequenceNumber) < 0 || parseInt(sequenceNumber) > 9999) {
    return { valid: false, error: "Invalid sequence number" };
  }

  // Validate citizenship (C)
  if (citizenship !== "0" && citizenship !== "1") {
    return { valid: false, error: "Invalid citizenship status" };
  }

  // Validate racial classification (A)
  if (racialClassification !== "8") {
    return { valid: false, error: "Invalid racial classification" };
  }

  // Validate checksum (Z) using Luhn algorithm
  // if (!isValidLuhn(id.substring(0, 12), checksum)) {
  //   return { valid: false, error: "Invalid checksum" };
  // }

  // If all validations pass
  return { valid: true };
}

// Function to validate Passport Number (Alphanumeric)
function validatePassEMAIL_PORT_NUMBER(passport) {
  // Passport numbers can be alphanumeric, and their format can vary.
  // For this example, we assume a passport number must be between 6 and 9 characters long.
  // Modify this logic as needed based on specific passport number rules.

  const validPassportFormat = /^[A-Z0-9]{6,9}$/i; // Alphanumeric with 6 to 9 characters

  if (validPassportFormat.test(passport)) {
    return { valid: true };
  } else {
    return { valid: false, error: "Invalid passport number format" };
  }
}

// Function to validate birth date (YYMMDD)
function isValidDate(birthDate) {
  const year = parseInt(birthDate.substring(0, 2), 10);
  const month = parseInt(birthDate.substring(2, 4), 10);
  const day = parseInt(birthDate.substring(4, 6), 10);

  const currentYear = new Date().getFullYear() % 100;
  const fullYear = year <= currentYear ? 2000 + year : 1900 + year;

  // Check if month is valid (01-12)
  if (month < 1 || month > 12) {
    return false;
  }

  // Check if day is valid for the month
  const daysInMonth = new Date(fullYear, month, 0).getDate();
  if (day < 1 || day > daysInMonth) {
    return false;
  }

  return true;
}

// Luhn algorithm to validate checksum (Z)
function isValidLuhn(number, checksum) {
  let sum = 0;
  let isSecond = false;

  for (let i = number.length - 1; i >= 0; i--) {
    let digit = parseInt(number[i], 10);

    if (isSecond) {
      digit *= 2;
      if (digit > 9) {
        digit -= 9;
      }
    }

    sum += digit;
    isSecond = !isSecond;
  }

  // Calculate the checksum (Z)
  const calculatedChecksum = (10 - (sum % 10)) % 10;

  return calculatedChecksum === parseInt(checksum, 10);
}

// console.log(validateSouthAfricanID("0301015898184"))

// const usersToSend = await getManagerAndSuperAdmin(null)
//   const companyWebsite = process.env.HARMONY_FRONTEND_URL;
//   const companyName = await getCompanyName(14)
//   const employeeName = "John" + " " + "Doe";

//   for(let user of usersToSend){
//     const sendRecordArray = {
//       templateFileUrl: "sheq_mail.html",
//       employee_name: employeeName,
//       name: "Roshan Gupta",
//       company_name: companyName,
//       company_website: companyWebsite,
//       templateName: "Employee Creation",
//     };

//     const info = await sendEmail(
//       "info@harmonyandhelp.com",
//       "mohammad.tariq.sartia@gmail.com",
//       `Successful Creation of Employee Record for {{Platform_Name}}`,
//       sendRecordArray
//     );
//   }
