import Excel from "exceljs";
import path from "path";
import fs from "fs";
import db from "../db-config.js";
import {
  decodeSingle_statement,
  makeJoins,
  whereCondition,
  storeError,
  textExtractor,
  decodeAndParseFields,
  isHtml,
} from "./general.js";
import asyncHandler from "express-async-handler";

const romanRankObj = {
  I: "red",
  II: "yellow",
  III: "blue",
  IV: "green",
  V: "grey",
};

const colorObj = {
  white: "FFFFFF",
  black: "000000",
  blue: "90CAF9",
  yellow: "FFF59D",
  darkBlue: "000080",
  violet: "9400D3",
  red: "FF0000",
  gray: "808080",
  lightGray: "EEEEEE",
  green: "A5D6A7",
};

export const getContractorAssessmentRegisterExcel = asyncHandler(
  async (req, res) => {
    try {
      const workbook = new Excel.Workbook();
      const worksheet = workbook.addWorksheet("Contractor Risk Register");
      const white = "FFFFFF";
      const black = "000000";
      const blue = "90CAF9";
      const darkBlue = "000080";
      const violet = "9400D3";
      const red = "FF0000";
      const yellow = "FFF59D";
      const gray = "808080";
      const lightGray = "D3D3D3";
      const green = "A5D6A7";
      const headers = [
        { header: "Risk ID", key: "riskId", width: 20 },
        { header: "Risk Name", key: "riskName", width: 25 },
        { header: "Risk Description", key: "riskDescription", width: 25 },
        { header: "Impact", key: "impact", width: 25 },
        {
          header: "Causes",
          key: "causes",
          width: 25,
        },
        { header: "Contractor", key: "contractorName", width: 25 },
        {
          header: "Likelihood",
          key: "likelihood",
          width: 25,
        },
        { header: "Consequence", key: "consequence", width: 25 },
        { header: "Risk Ranking", key: "riskRanking", width: 25 },
        { header: "Control Name", key: "controlName", width: 25 },
        { header: "Control Description", key: "controlDescription", width: 25 },
        {
          header: "Control Effectiveness",
          key: "controlEffectiveness",
          width: 25,
        },
        {
          header:
            "Control Design Intent: /(enter) Preventative (P)/Corrective( C)/Reactive (R )",
          key: "controlDesignIntent",
          width: 25,
        },
        { header: "Control Owner", key: "controlOwner", width: 25 },
        { header: "Risk Rating", key: "residualRiskRating", width: 25 },
        {
          header: "Risk Ranking",
          key: "residualRiskRanking",
          width: 25,
        },
      ];

      headers.forEach((header) => {
        const columnIndex = headers.indexOf(header) + 1;
        worksheet.getColumn(columnIndex).height = 20;
        worksheet.getColumn(columnIndex).width = header.width;
      });

      worksheet.mergeCells("A1:Q1");
      worksheet.getCell("A1").value = "Contractor Risk Register";
      worksheet.getCell("A1").alignment = { horizontal: "center" };

      worksheet.mergeCells("A2:F2");
      worksheet.mergeCells("G2:I2");
      worksheet.getCell("G2").value = "INHERENT RISK";
      worksheet.getCell("G2").alignment = { vertical: "middle" };
      worksheet.getCell("G2").alignment = { horizontal: "center" };

      worksheet.mergeCells("P2:Q2");
      worksheet.getCell("P2").value = "RESIDUAL RISK";
      worksheet.getCell("P2").alignment = { vertical: "middle" };
      worksheet.getCell("P2").alignment = { horizontal: "center" };
      worksheet.getCell("P2").fill = {
        type: "pattern",
        pattern: "solid",
        fgColor: { argb: "FF0000FF" }, // Blue color
      };

      worksheet.addRow(headers.map((header) => header.header));

      const { id } = req.params;
      const condition = await whereCondition({
        table: "contractor_risk_assessment",
        page: req.query.page,
        all: req.query.all,
        pageSize: req.query.pageSize,
        filter: req.query.filter,
        id,
        grouped: req.query.grouped,
        user: req.user,
      });

      // console.log("condition", condition)

      /**Make Joins according to tables */

      const joins = [
        {
          type: "left",
          targetTable: "contractor_registration as cr",
          onCondition: "cr.id = contractor_risk_assessment.contractor",
        },
      ];

      const joinRecord = await makeJoins(joins);

      let [dataFetch] = await db.query(
        `SELECT 
         contractor_risk_assessment.id as riskId , contractor_risk_assessment.risk_name as riskName , 
         contractor_risk_assessment.risk_description as riskDescription , 
         contractor_risk_assessment.impact_description as impact , 
         contractor_risk_assessment.causes_description as causes, 
         cr.contractor_name as contractorName ,
         contractor_risk_assessment.inherent_likelihood as likelihood , 
         contractor_risk_assessment.inherent_consequence as consequence , 
         contractor_risk_assessment.inherent_risk_ranking as riskRanking , 
         contractor_risk_assessment.control_name as controlName , 
         contractor_risk_assessment.control_description as controlDescription , 
         contractor_risk_assessment.control_effectiveness as controlEffectiveness , 
         contractor_risk_assessment.control_design_intent as controlDesignIntent , 
         contractor_risk_assessment.residual_risk_rating as residualRiskRating , 
         contractor_risk_assessment.residual_risk_ranking as residualRiskRanking
         FROM contractor_risk_assessment
         ${joinRecord}
     WhERE contractor_risk_assessment.deleted = 0 ${condition}`
      );

      dataFetch = await decodeAndParseFields(dataFetch);

      /** Check for Html content if present, then extract text from it and replace it */
      dataFetch = dataFetch.map((item) => {
        const updatedItem = { ...item };
        for (const key in updatedItem) {
          const value = updatedItem[key];
          if (typeof value === 'string' && isHtml(value)) {
            updatedItem[key] = textExtractor(value);
          }
        }
        return updatedItem;
      });

      /**Decode content */

      dataFetch.forEach((row, index) => {
        const rowIndex = index + 4; // Adjust the row index based on the starting row
        Object.keys(row).forEach((key, colIndex) => {
          worksheet.getCell(rowIndex, colIndex + 1).value = row[key]; // Set cell value
          if (key === "riskRanking") {
            worksheet.getCell(rowIndex, colIndex + 1).fill = {
              type: "pattern",
              pattern: "solid",
              fgColor: { argb: colorObj[romanRankObj[row[key]]] },
            };
          }
          if (key === "residualRiskRanking") {
            worksheet.getCell(rowIndex, colIndex + 1).fill = {
              type: "pattern",
              pattern: "solid",
              fgColor: { argb: colorObj[romanRankObj[row[key]]] },
            };
          }
        });
      });

      const second = worksheet.getRow(2);
      second.height = 20;

      const cwd = process.cwd();

      worksheet.eachRow({ includeEmpty: false }, function (row, rowNumber) {
        // Iterate over each cell in the row
        row.eachCell({ includeEmpty: false }, function (cell, colNumber) {
          // Set the border for each cell
          cell.border = {
            top: { style: "medium" }, // Thin top border
            left: { style: "medium" }, // Thin left border
            bottom: { style: "medium" }, // Thin bottom border
            right: { style: "medium" }, // Thin right border
            // You can adjust the style to 'medium' or 'thick' for darker or bolder borders
          };
          cell.alignment = { vertical: "middle", horizontal: "center" };
        });
      });

      const heading = worksheet.getRow(1);

      heading.height = 40;

      heading.eachCell((cell) => {
        cell.fill = {
          type: "pattern",
          pattern: "solid",
          fgColor: { argb: gray },
        };
        cell.font = {
          size: 16,
        };
      });

      const subHeading = worksheet.getRow(2);
      subHeading.eachCell((cell) => {
        // console.log("cell", cell)
        cell.font = {
          size: 14,
        };
        cell.fill = {
          type: "pattern",
          pattern: "solid",
          fgColor: { argb: "FF808080" }, // Gray color
        };
      });

      const startCol = 10; // Column J
      const endCol = 15; // Column O

      applyCss(worksheet, 2, 10, 15, gray, black);

      applyCss(worksheet, 3, 1, 1, blue, white);
      applyCss(worksheet, 3, 2, 4, darkBlue, white);
      applyCss(worksheet, 3, 5, 6, red, white);
      applyCss(worksheet, 3, 7, 10, violet, white);
      applyCss(worksheet, 3, 11, 15, lightGray, black);
      applyCss(worksheet, 3, 16, 17, green, white);
      const headerRow = worksheet.getRow(2);
      headerRow.height = 20;
      // Define the file path where you want to save the Excel file
      const filePath = path.join(
        cwd,
        "public",
        "excel",
        "Contractor_Risk_Assessment_Register.xlsx"
      );

      const folderPath = path.join(cwd, "public", "excel");

      if (!fs.existsSync(folderPath)) {
        fs.mkdirSync(folderPath, { recursive: true });
      }

      // Save the workbook
      await workbook.xlsx.writeFile(filePath);

      return res.status(200).json({
        status: true,
        message: "Excel file created successfully",
        data: `${process.env.HARMONY_BACKEND_URL}/excel/Contractor_Risk_Assessment_Register.xlsx`,
      });
    } catch (error) {
      storeError(error);
      return res.status(500).json({
        status: false,
        message: error.message,
      });
    }
  }
);

function applyCss(worksheet, row, startCol, endCol, BgColor, textColor) {
  for (let col = startCol; col <= endCol; col++) {
    const cell = worksheet.getCell(row, col);
    cell.fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: BgColor },
    };
    cell.font = {
      color: { argb: textColor },
    };
    cell.alignment = {
      vertical: "middle",
      horizontal: "center",
      wrapText: true,
    };
  }
}
