import { DataTypes } from "sequelize";

/** New Updated Function */
export const createQueryBuilder = (model, requestBody) => {
  if (!model) {
    throw new Error("Model is undefined or null.");
  }
  requestBody = JSON.parse(JSON.stringify(requestBody));
  const columns = Object.keys(model.rawAttributes);

  const filteredColumns = columns.filter((column) => {
    return (
      requestBody.hasOwnProperty(column) &&
      requestBody[column] !== undefined &&
      requestBody[column] !== "undefined" &&
      requestBody[column] !== null &&
      requestBody[column] !== "null" &&
      // requestBody[column] !== "" && // Exclude empty strings
      !(
        model.rawAttributes[column].allowNull === false && !requestBody[column]
      ) && // Check if required field is missing
      !model.rawAttributes[column].primaryKey &&
      !model.rawAttributes[column].autoIncrement
    );
  });

  const placeholders = Array.from(
    { length: filteredColumns.length },
    () => `?`
  ).join(", ");
  const columnNames = filteredColumns.join(", ");
  const query = `INSERT INTO ${model.tableName}(${columnNames}) VALUES (${placeholders})`;

  const values = filteredColumns.map((column) => {
    const attributeType = model.rawAttributes[column].type;
    if (
      (attributeType instanceof DataTypes.JSON ||
        attributeType instanceof DataTypes.TEXT) &&
      (typeof requestBody[column] === "object" ||
        Array.isArray(requestBody[column]))
    ) {
      return JSON.stringify(requestBody[column]);
    }
    // set null if frontend is sending key value "" empty string
    if (requestBody[column] === "") {
      return null;
    }
    return requestBody[column];
  });

  return {
    query: query,
    values: values,
  };
};

export const updateQueryBuilder = (model, requestBody, customWhere = null) => {
  if (!model) {
    throw new Error("Model is undefined or null.");
  }
  requestBody = JSON.parse(JSON.stringify(requestBody));

  const columns = Object.keys(model.rawAttributes);

  // Identify primary keys and auto increment keys
  const primaryKeyColumns = columns.filter(
    (column) =>
      model.rawAttributes[column].primaryKey ||
      model.rawAttributes[column].autoIncrement
  );

  // Filter columns to update, excluding primary keys, auto increment keys, and the 'created_by' column
  const filteredColumns = columns.filter((column) => {
    return (
      column !== "created_by" && // Exclude the 'created_by' column
      requestBody.hasOwnProperty(column) &&
      requestBody[column] !== undefined &&
      requestBody[column] !== "undefined" &&
      requestBody[column] !== null &&
      requestBody[column] !== "null" &&
      // requestBody[column] !== "" &&
      !model.rawAttributes[column].primaryKey &&
      !model.rawAttributes[column].autoIncrement
    );
  });

  // Generate SET part of the update query
  const setStatements = filteredColumns
    .map((column) => `${column} = ?`)
    .join(", ");

  // Determine columns for the WHERE part of the update query
  const whereColumns =
    customWhere && Object.keys(customWhere).length > 0
      ? Object.keys(customWhere)
      : primaryKeyColumns;

  const whereClauses = whereColumns
    .filter((column) => requestBody.hasOwnProperty(column))
    .map((column) => `${column} = ?`);

  if (whereClauses.length === 0) {
    throw new Error("No columns provided for the WHERE clause.");
  }

  const whereClause = whereClauses.join(" AND ");
  const query = `UPDATE ${model.tableName} SET ${setStatements} WHERE ${whereClause}`;

  const values = filteredColumns.map((column) => {
    const attributeType = model.rawAttributes[column].type;
    if (
      (attributeType instanceof DataTypes.JSON ||
        attributeType instanceof DataTypes.TEXT) &&
      (typeof requestBody[column] === "object" ||
        Array.isArray(requestBody[column]))
    ) {
      return JSON.stringify(requestBody[column]);
    }
    // set null if frontend is sending key value "" empty string
    if (requestBody[column] === "") {
      return null;
    }
    return requestBody[column];
  });

  // Append WHERE clause values to the values array
  if (customWhere) {
    whereColumns.forEach((column) => {
      //   if (customWhere.hasOwnProperty(column)) {
      values.push(customWhere[column]);
      //   }
    });
  } else {
    primaryKeyColumns.forEach((column) => {
      if (requestBody.hasOwnProperty(column)) {
        values.push(requestBody[column]);
      }
    });
  }

  return {
    query: query,
    values: values,
  };
};

/** Optimized Function */

// export const createQueryBuilder = (model, requestBody) => {
//   if (!model) {
//     throw new Error("Model is undefined or null.");
//   }

//   // Copy the requestBody to avoid mutation
//   requestBody = { ...requestBody };

//   const columns = Object.keys(model.rawAttributes);

//   const { filteredColumns, values } = columns.reduce(
//     (acc, column) => {
//       const attribute = model.rawAttributes[column];
//       const value = requestBody[column];

//       if (
//         requestBody.hasOwnProperty(column) &&
//         value !== undefined &&
//         value !== null &&
//         !(attribute.allowNull === false && !value) &&
//         !attribute.primaryKey &&
//         !attribute.autoIncrement
//       ) {
//         acc.filteredColumns.push(column);

//         let finalValue = value;
//         if (
//           (attribute.type instanceof DataTypes.JSON ||
//             attribute.type instanceof DataTypes.TEXT) &&
//           (typeof value === "object" || Array.isArray(value))
//         ) {
//           finalValue = JSON.stringify(value);
//         }
//         if (value === "") {
//           finalValue = null;
//         }

//         acc.values.push(finalValue);
//       }
//       return acc;
//     },
//     { filteredColumns: [], values: [] }
//   );

//   const placeholders = filteredColumns.map(() => `?`).join(", ");
//   const columnNames = filteredColumns.join(", ");
//   const query = `INSERT INTO ${model.tableName} (${columnNames}) VALUES (${placeholders})`;

//   return {
//     query,
//     values,
//   };
// };

// export const updateQueryBuilder = (model, requestBody, customWhere = null) => {
//   if (!model) {
//     throw new Error("Model is undefined or null.");
//   }

//   // Copy the requestBody to avoid mutation
//   requestBody = { ...requestBody };

//   const columns = Object.keys(model.rawAttributes);

//   // Identify primary keys and auto increment keys
//   const primaryKeyColumns = columns.filter(
//     (column) => model.rawAttributes[column].primaryKey || model.rawAttributes[column].autoIncrement
//   );

//   // Filter columns to update, excluding primary keys, auto increment keys, and the 'created_by' column
//   const filteredColumns = columns.filter((column) => {
//     const attribute = model.rawAttributes[column];
//     const value = requestBody[column];
//     return (
//       column !== "created_by" &&
//       requestBody.hasOwnProperty(column) &&
//       value !== undefined &&
//       value !== null &&
//       !(attribute.allowNull === false && !value) &&
//       !attribute.primaryKey &&
//       !attribute.autoIncrement
//     );
//   });

//   if (filteredColumns.length === 0) {
//     throw new Error("No columns provided for the SET clause.");
//   }

//   // Generate SET part of the update query
//   const setStatements = filteredColumns.map((column) => `${column} = ?`).join(", ");

//   // Determine columns for the WHERE part of the update query
//   const whereColumns = customWhere && Object.keys(customWhere).length > 0
//     ? Object.keys(customWhere)
//     : primaryKeyColumns;

//   const whereClauses = whereColumns.map((column) => `${column} = ?`);

//   if (whereClauses.length === 0) {
//     throw new Error("No columns provided for the WHERE clause.");
//   }

//   const whereClause = whereClauses.join(" AND ");
//   const query = `UPDATE ${model.tableName} SET ${setStatements} WHERE ${whereClause}`;

//   const values = filteredColumns.map((column) => {
//     const attributeType = model.rawAttributes[column].type;
//     const value = requestBody[column];
//     if (
//       (attributeType instanceof DataTypes.JSON || attributeType instanceof DataTypes.TEXT) &&
//       (typeof value === "object" || Array.isArray(value))
//     ) {
//       return JSON.stringify(value);
//     }
//     // set null if frontend is sending key value "" empty string
//     if (value === "") {
//       return null;
//     }
//     return value;
//   });

//   // Append WHERE clause values to the values array
//   const whereValues = whereColumns.map((column) => {
//     return customWhere ? customWhere[column] : requestBody[column];
//   });

//   return {
//     query,
//     values: [...values, ...whereValues],
//   };
// };
