import { Router } from "express";
import {
  createUpdateActionTemplate,
  getActionTemplateList,
  deleteActionTemplateList,
  approveRejectTemplateList,
} from "../controller/actionTemplateListController.js";

import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

/**
 * @swagger
 * tags:
 *   name: Action Template List
 *   description: API operations for managing action template list
 */

const actionTemplateListRoutes = Router();

/**
 * @swagger
 * /api/actionTemplateList/create-update-action-template:
 *   post:
 *     tags: [Action Template List]
 *     summary: Create or update an action template
 *     description: Creates a new action template or updates an existing one.
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *                 description: The ID of the action template (for updates).
 *               name:
 *                 type: string
 *                 description: The name of the action template.
 *               visibility:
 *                 type: string
 *                 description: The visibility level of the action template.
 *               category:
 *                 type: string
 *                 description: The category of the action template.
 *               description:
 *                 type: string
 *                 description: A detailed description of the action template.
 *               subcategory:
 *                 type: string
 *                 description: The subcategory of the action template.
 *               attachment:
 *                 type: string
 *                 description: Any attachments related to the action template.
 *               assigned_to:
 *                 type: integer
 *                 description: The ID of the user assigned to the action template.
 *               dependencies:
 *                 type: string
 *                 description: Dependencies for the action template.
 *               task_type:
 *                 type: string
 *                 description: The type of the task related to the action template.
 *               module_name:
 *                 type: string
 *                 description: The module name related to the action template.
 *               sub_module_name:
 *                 type: string
 *                 description: The sub-module name related to the action template.
 *               record_name:
 *                 type: string
 *                 description: The record name related to the action template.
 *               task_title:
 *                 type: string
 *                 description: The title of the task related to the action template.
 *               task_description:
 *                 type: string
 *                 description: A description of the task related to the action template.
 *               priority:
 *                 type: string
 *                 description: The priority level of the action template.
 *               task_duration:
 *                 type: string
 *                 description: The duration of the task related to the action template.
 *               organization:
 *                 type: integer
 *                 description: The ID of the organization related to the action template.
 *               department:
 *                 type: integer
 *                 description: The ID of the department related to the action template.
 *             required:
 *               - name
 *               - category
 *           example:
 *             id: 1
 *             name: "Template Name"
 *             visibility: "public"
 *             category: "Category Name"
 *             description: "Detailed description of the action template."
 *             subcategory: "Subcategory Name"
 *             attachment: "attachment_file.pdf"
 *             assigned_to: 2
 *             dependencies: "None"
 *             task_type: "type1"
 *             module_name: "Module Name"
 *             sub_module_name: "Sub Module Name"
 *             record_name: "Record Name"
 *             task_title: "Task Title"
 *             task_description: "Task Description"
 *             priority: "High"
 *             task_duration: "2h"
 *             organization: 3
 *             department: 4
 *
 *     responses:
 *       200:
 *         description: Action template created or updated successfully.
 *       400:
 *         description: Bad request.
 *       401:
 *         description: Unauthorized.
 */

actionTemplateListRoutes.post(
  "/create-update-action-template",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      233,
      "action_template_list"
    )(req, res, next);
  },
  controllerWrapper(createUpdateActionTemplate)
);

/**
 * @swagger
 * /api/actionTemplateList/get-action-template:
 *   get:
 *     tags: [Action Template List]
 *     summary: Get a list of action templates
 *     description: Retrieves a list of all action templates.
 *     responses:
 *       200:
 *         description: A list of action templates.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                   name:
 *                     type: string
 *                   visibility:
 *                     type: string
 *                   category:
 *                     type: string
 *                   description:
 *                     type: string
 *                   subcategory:
 *                     type: string
 *                   attachment:
 *                     type: string
 *                   assigned_to:
 *                     type: integer
 *                   dependencies:
 *                     type: string
 *                   task_type:
 *                     type: string
 *                   module_name:
 *                     type: string
 *                   sub_module_name:
 *                     type: string
 *                   record_name:
 *                     type: string
 *                   task_title:
 *                     type: string
 *                   task_description:
 *                     type: string
 *                   priority:
 *                     type: string
 *                   task_duration:
 *                     type: string
 *                   organization:
 *                     type: integer
 *                   department:
 *                     type: integer
 *             example:
 *               - id: 1
 *                 name: "Template Name 1"
 *                 visibility: "public"
 *                 category: "Category 1"
 *                 description: "Description for template 1."
 *                 subcategory: "Subcategory 1"
 *                 attachment: "attachment1.pdf"
 *                 assigned_to: 2
 *                 dependencies: "None"
 *                 task_type: "type1"
 *                 module_name: "Module 1"
 *                 sub_module_name: "Sub Module 1"
 *                 record_name: "Record 1"
 *                 task_title: "Task Title 1"
 *                 task_description: "Task Description 1"
 *                 priority: "High"
 *                 task_duration: "1h"
 *                 organization: 3
 *                 department: 4
 *               - id: 2
 *                 name: "Template Name 2"
 *                 visibility: "private"
 *                 category: "Category 2"
 *                 description: "Description for template 2."
 *                 subcategory: "Subcategory 2"
 *                 attachment: "attachment2.pdf"
 *                 assigned_to: 3
 *                 dependencies: "Dependency 1"
 *                 task_type: "type2"
 *                 module_name: "Module 2"
 *                 sub_module_name: "Sub Module 2"
 *                 record_name: "Record 2"
 *                 task_title: "Task Title 2"
 *                 task_description: "Task Description 2"
 *                 priority: "Medium"
 *                 task_duration: "3h"
 *                 organization: 4
 *                 department: 5
 *       401:
 *         description: Unauthorized.
 */

actionTemplateListRoutes.get(
  "/get-action-template",
  verifyToken,
  sideBarPermissionCheck("view", 233, "action_template_list"),
  controllerWrapper(getActionTemplateList)
);

/**
 * @swagger
 * /api/actionTemplateList/get-action-template/{id}:
 *   get:
 *     tags: [Action Template List]
 *     summary: Get an action template by ID
 *     description: Retrieves an action template by its ID.
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: The ID of the action template to retrieve.
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: The action template details.
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                 name:
 *                   type: string
 *                 visibility:
 *                   type: string
 *                 category:
 *                   type: string
 *                 description:
 *                   type: string
 *                 subcategory:
 *                   type: string
 *                 attachment:
 *                   type: string
 *                 assigned_to:
 *                   type: integer
 *                 dependencies:
 *                   type: string
 *                 task_type:
 *                   type: string
 *                 module_name:
 *                   type: string
 *                 sub_module_name:
 *                   type: string
 *                 record_name:
 *                   type: string
 *                 task_title:
 *                   type: string
 *                 task_description:
 *                   type: string
 *                 priority:
 *                   type: string
 *                 task_duration:
 *                   type: string
 *                 organization:
 *                   type: integer
 *                 department:
 *                   type: integer
 *             example:
 *               id: 1
 *               name: "Template Name"
 *               visibility: "public"
 *               category: "Category Name"
 *               description: "Detailed description of the action template."
 *               subcategory: "Subcategory Name"
 *               attachment: "attachment_file.pdf"
 *               assigned_to: 2
 *               dependencies: "None"
 *               task_type: "type1"
 *               module_name: "Module Name"
 *               sub_module_name: "Sub Module Name"
 *               record_name: "Record Name"
 *               task_title: "Task Title"
 *               task_description: "Task Description"
 *               priority: "High"
 *               task_duration: "2h"
 *               organization: 3
 *               department: 4
 *       401:
 *         description: Unauthorized.
 *       404:
 *         description: Action template not found.
 */

actionTemplateListRoutes.get(
  "/get-action-template/:id",
  verifyToken,
  sideBarPermissionCheck("view", 233, "action_template_list"),
  controllerWrapper(getActionTemplateList)
);

/**
 * @swagger
 * /api/actionTemplateList/delete-action-template/{id}:
 *   delete:
 *     tags: [Action Template List]
 *     summary: Delete an action template by ID
 *     description: Deletes an action template by its ID.
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: The ID of the action template to delete.
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Action template deleted successfully.
 *       401:
 *         description: Unauthorized.
 *       404:
 *         description: Action template not found.
 */

actionTemplateListRoutes.delete(
  "/delete-action-template/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 233, "action_template_list"),
  controllerWrapper(deleteActionTemplateList)
);

/**
 * @swagger
 * /api/actionTemplateList/approve-reject-action-template:
 *   post:
 *     tags: [Action Template List]
 *     summary: Approve or reject a action Template
 *     description: Approves or rejects a action template.
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               status:
 *                 type: string
 *                 description: The status of the action template (for updates) Approved , Rejected.
 *           example:
 *             status: "status"
 *     responses:
 *       200:
 *         description: Action template created or updated successfully.
 *       400:
 *         description: Bad request.
 *       401:
 *         description: Unauthorized.
 */

actionTemplateListRoutes.post(
  "/approve-reject-action-template",
  verifyToken,
  sideBarPermissionCheck("edit", 233, "action_template_list"),
  controllerWrapper(approveRejectTemplateList)
);

export default actionTemplateListRoutes;
