import { Router } from 'express';
import { verifyToken } from '../helper/tokenVerify.js';
import { sideBarPermissionCheck } from '../helper/checkPermission.js';
import {
  createUpdateAsset,
  getAsset,
  deleteAsset,
} from '../controller/assetController.js';
import { controllerWrapper } from '../helper/wrapper.js';

/**
 * @swagger
 * tags:
 *   name: Asset
 *   description: API operations for managing Asset
 */
/**
 * @swagger
 * components:
 *   schemas:
 *     Asset:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated ID of the asset
 *         asset_name:
 *           type: string
 *           description: The name of the asset
 *         asset_id:
 *           type: string
 *           description: A unique identifier for the asset
 *         asset_type:
 *           type: integer
 *           description: ID of the asset type (e.g., machinery, equipment)
 *         asset_description:
 *           type: string
 *           description: Detailed description of the asset
 *         model_make:
 *           type: string
 *           description: Model or make of the asset
 *         acquisition_type:
 *           type: string
 *           enum: ["Owned", "Leased"]
 *           description: How the asset was acquired (owned or leased)
 *         purchase_date:
 *           type: string
 *           format: date
 *           description: Purchase date of the asset (if owned)
 *         purchase_price:
 *           type: integer
 *           description: Purchase price of the asset (if owned)
 *         vendor_supplier:
 *           type: string
 *           description: Vendor or supplier of the asset
 *         leased_date:
 *           type: string
 *           format: date
 *           description: Lease start date of the asset (if leased)
 *         lease_price:
 *           type: integer
 *           description: Lease price of the asset (if leased)
 *         lease_provider:
 *           type: string
 *           description: Provider of the lease
 *         lease_duration_value:
 *           type: integer
 *           description: Duration of the lease
 *         lease_duration_unit:
 *           type: string
 *           enum: ["Days", "Months", "Years"]
 *           description: Unit of time for the lease duration
 *         location:
 *           type: integer
 *           description: ID of the asset's location
 *         warranty_terms:
 *           type: string
 *           description: Warranty terms for the asset
 *         warranty_expiration_date:
 *           type: string
 *           format: date
 *           description: Expiration date of the asset's warranty
 *         maintenance_regime:
 *           type: string
 *           enum:
 *             - "Preventive Maintenance"
 *             - "Predictive Maintenance"
 *             - "Corrective Maintenance"
 *             - "Reactive Maintenance"
 *             - "Condition-Based Maintenance"
 *             - "Total Productive Maintenance"
 *             - "Scheduled Maintenance"
 *             - "Reliability-Centered Maintenance"
 *           description: Maintenance regime for the asset
 *         conditions:
 *           type: string
 *           enum: ["New", "Used", "Damaged"]
 *           description: Condition of the asset
 *         status:
 *           type: string
 *           enum: ["Active", "Inactive", "Under repair"]
 *           description: Current status of the asset
 *         department:
 *           type: integer
 *           description: ID of the associated department
 *         organization:
 *           type: integer
 *           description: ID of the associated organization
 */

const assetRoutes = Router();

/**
 * @swagger
 * /api/asset/create-update-asset:
 *   post:
 *     summary: Create or update a Asset
 *     tags: [Asset]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Asset'
 *     responses:
 *       200:
 *         description: The Asset was successfully created/updated
 */

assetRoutes.post(
  '/create-update-asset',
  verifyToken,
  controllerWrapper(createUpdateAsset)
);

/**
 * @swagger
 * /api/asset/get-asset:
 *   get:
 *     summary: Get all Asset
 *     tags: [Asset]
 *     responses:
 *       200:
 *         description: List of all Asset
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Asset'
 */

assetRoutes.get('/get-asset', verifyToken, controllerWrapper(getAsset));

/**
 * @swagger
 * /api/asset/get-asset/{id}:
 *   get:
 *     summary: Get a Asset by ID
 *     tags: [Asset]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the Asset
 *     responses:
 *       200:
 *         description: The Asset
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Asset'
 */

assetRoutes.get('/get-asset/:id', verifyToken, controllerWrapper(getAsset));

/**
 * @swagger
 * /api/asset/delete-asset/{id}:
 *   delete:
 *     summary: Delete a Asset by ID
 *     tags: [Asset]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the Asset to delete
 *     responses:
 *       200:
 *         description:Final Handover was deleted
 */

assetRoutes.delete(
  '/delete-asset/:id',
  verifyToken,
  controllerWrapper(deleteAsset)
);

export default assetRoutes;
