import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateFinalHandover,
  getFinalHandover,
  deleteFinalHandover,
  fillForm,
  sendFinalHandover,
} from "../controller/finalHandoverController.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Contractor FinalHandover
 *   description: API operations for managing contractor final handover
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     FinalHandover:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated ID of the contractor final handover
 *         contractor:
 *           type: integer
 *           description: ID of the contractor
 *         status:
 *           type: string
 *           description: The current status of the site access
 *         form:
 *           type: string
 *           description: The form data in long text format
 *         organization:
 *           type: integer
 *           description: ID of the associated organization
 *         department:
 *           type: integer
 *           description: ID of the associated department
 *         deleted:
 *           type: integer
 *           description: Whether the site access is deleted (0 or 1)
 *         created_by:
 *           type: integer
 *           description: ID of the user who created the site access
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the site access was created
 *         updated_by:
 *           type: integer
 *           description: ID of the user who last updated the site access
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the site access was last updated
 */

const finalHandoverRoutes = Router();

/**
 * @swagger
 * /api/finalHandover/create-update-final-handover:
 *   post:
 *     summary: Create or update a contractor final handover
 *     tags: [Contractor FinalHandover]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/FinalHandover'
 *     responses:
 *       200:
 *         description: The contractor final handover was successfully created/updated
 */

finalHandoverRoutes.post(
  "/create-update-final-handover",
  verifyToken,
  controllerWrapper(createUpdateFinalHandover)
);

/**
 * @swagger
 * /api/finalHandover/get-final-handover:
 *   get:
 *     summary: Get all contractor final handover
 *     tags: [Contractor FinalHandover]
 *     responses:
 *       200:
 *         description: List of all contractor final handover
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/FinalHandover'
 */

finalHandoverRoutes.get(
  "/get-final-handover",
  verifyToken,
  controllerWrapper(getFinalHandover)
);

/**
 * @swagger
 * /api/finalHandover/get-final-handover/{id}:
 *   get:
 *     summary: Get a contractor final handover by ID
 *     tags: [Contractor FinalHandover]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor final handover
 *     responses:
 *       200:
 *         description: The final handover data
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/FinalHandover'
 */

finalHandoverRoutes.get(
  "/get-final-handover/:id",
  verifyToken,
  controllerWrapper(getFinalHandover)
);

/**
 * @swagger
 * /api/finalHandover/delete-final-handover/{id}:
 *   delete:
 *     summary: Delete a contractor final handover by ID
 *     tags: [Contractor FinalHandover]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor final handover to delete
 *     responses:
 *       200:
 *         description:Final Handover was deleted
 */

finalHandoverRoutes.delete(
  "/delete-final-handover/:id",
  verifyToken,
  controllerWrapper(deleteFinalHandover)
);


/**
 * @swagger
 * /api/finalHandover/send-final-handover":
 *   get:
 *     summary: Send a contractor final-handover",
 *     tags: [Contractor FinalHandover]
 *     responses:
 *       200:
 *         description: The final-handover was successfully sent
 */

finalHandoverRoutes.post( "/send-final-handover",verifyToken,sideBarPermissionCheck("view", 277, "final_handover"),controllerWrapper(sendFinalHandover));

/**
 * @swagger
 * /api/finalHandover/fill-final-handover:
 *   post:
 *     summary: Fill out a contractor final-handover form
 *     tags: [Contractor FinalHandover]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/finalHandover'
 *     responses:
 *       200:
 *         description: The form was successfully submitted
 */

finalHandoverRoutes.post("/fill-final-handover", controllerWrapper(fillForm));

export default finalHandoverRoutes;
