// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  deleteFirstAidBox,
  FirstAidBoxCreateUpdate,
  viewFirstAidBox,
} from "../controller/firstAidBoxController.js";
import { controllerWrapper } from "../helper/wrapper.js";


/**
 * @swagger
 * tags:
 *   name: FirstAidBox
 *   description: API Operations For Managing First Aid Box
 */
const FirstAidBoxRoutes = Router();

/**
 * @swagger
 * /api/firstAidBox/create-update-first-aid-box:
 *   post:
 *     summary: Create or Update a First Aid Box
 *     tags: [FirstAidBox]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               content_list:
 *                 type: integer
 *                 example: 5
 *               location:
 *                 type: integer
 *                 description: ID of the location
 *                 example: 9
 *               last_inspection_date:
 *                 type: string
 *                 format: date
 *                 example: 2023-09-10
 *               next_inspection_date:
 *                 type: string
 *                 format: date
 *                 example: 2024-09-10
 *               status:
 *                 type: string
 *                 enum: ["Compliant", "Non-compliant"]
 *                 example: Non-compliant
 *               department:
 *                 type: integer
 *                 description: ID of the department
 *                 example: 3
 *               organization:
 *                 type: integer
 *                 description: ID of the organization
 *                 example: 14
 *               created_by:
 *                 type: integer
 *                 description: ID of the user who created this record
 *                 example: 12
 *               updated_by:
 *                 type: integer
 *                 description: ID of the user who last updated this record
 *                 example: 15
 *               deleted:
 *                 type: boolean
 *                 example: false
 *     responses:
 *       200:
 *         description: Success
 *       400:
 *         description: Bad Request
 *       500:
 *         description: Internal Server Error
 */

FirstAidBoxRoutes.post(
  "/create-update-first-aid-box",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      315,
      "first_aid_box"
    )(req, res, next);
  },
  controllerWrapper(FirstAidBoxCreateUpdate)
);

/**
 * @swagger
 * /api/firstAidBox/get-first-aid-box:
 *   get:
 *     summary: Get list of all First Aid Box
 *     tags: [FirstAidBox]
 */
FirstAidBoxRoutes.get(
  "/get-first-aid-box",
  verifyToken,
  sideBarPermissionCheck("view", 315, "first_aid_box"),
  controllerWrapper(viewFirstAidBox)
);

/**
 * @swagger
 * /api/firstAidBox/get-first-aid-box/{id}:
 *   get:
 *     summary: Get A Specific First Aid Box record
 *     tags: [FirstAidBox]
 */
FirstAidBoxRoutes.get(
  "/get-first-aid-box/:id",
  verifyToken,
  sideBarPermissionCheck("view", 315, "first_aid_box"),
  controllerWrapper(viewFirstAidBox)
);

/**
 * @swagger
 * /api/firstAidBox/delete-first-aid-box/{id}:
 *   delete:
 *     summary: Delete a specific record of First Aid Box
 *     tags: [FirstAidBox]
 */
FirstAidBoxRoutes.delete(
  "/delete-first-aid-box/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 315, "first_aid_box"),
  controllerWrapper(deleteFirstAidBox)
);

export default FirstAidBoxRoutes;
