import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateIncidentRole,
  deleteIncidentRole,
  viewIncidentRole,
} from "../controller/incidentRoleController.js";

/**
 * @swagger
 * tags:
 *   name:  incident role
 *   description: API operations for managing  incident role
 */
const incidentRoleRoutes = Router();

/**
 * @swagger
 * /api/incidentRole/create-update-incident-role:
 *   post:
 *     summary: Create or update a  incident role
 *     tags: [ incident role]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { id: "id", title: "title", description: "description" }
 */
incidentRoleRoutes.post(
  "/create-update-incident-role",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 412, "incident_role")(req, res, next);
  },
  controllerWrapper(createUpdateIncidentRole)
);

/**
 * @swagger
 * /api/incidentRole/get-incident-role:
 *   get:
 *     summary: Get all records of  incident role
 *     tags: [ incident role]
 */
incidentRoleRoutes.get(
  "/get-incident-role",
  verifyToken,
  sideBarPermissionCheck("view", 412, "incident_role"),
  controllerWrapper(viewIncidentRole)
);

/**
 * @swagger
 * /api/incidentRole/get-incident-role/{id}:
 *   get:
 *     summary: Get a specific record of  incident role
 *     tags: [ incident role]
 */
incidentRoleRoutes.get(
  "/get-incident-role/:id",
  verifyToken,
  sideBarPermissionCheck("view", 412, "incident_role"),
  controllerWrapper(viewIncidentRole)
);

/**
 * @swagger
 * /api/incidentRole/delete-incident-role/{id}:
 *   delete:
 *     summary: Delete a specific record of  incident role
 *     tags: [ incident role]
 */
incidentRoleRoutes.delete(
  "/delete-incident-role/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 412, "incident_role"),
  controllerWrapper(deleteIncidentRole)
);

export default incidentRoleRoutes;
