import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateInsuranceClaimLogs,
  deleteInsuranceClaimLogs,
  getInsuranceClaimLogs,
} from "../controller/insuranceClaimLogsController.js";

const insuranceClaimLogsRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Insurance Claim logs
 *   description: API operations for managing insurance claims and approved payout status
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     InsuranceClaimLogs:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated ID of the insurance claim log
 *         insurance_claim_id:
 *           type: integer
 *           description: ID of the related insurance claim
 *         comments:
 *           type: string
 *           description: Additional comments regarding the insurance claim
 *         status:
 *           type: string
 *           description: Status of the insurance claim (e.g., pending, approved)
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the insurance claim log was created
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the insurance claim log was last updated
 *         created_by:
 *           type: integer
 *           description: ID of the user who created the insurance claim log
 *         updated_by:
 *           type: integer
 *           description: ID of the user who last updated the insurance claim log
 */

/**
 * @swagger
 * /api/insuranceClaimLogs/create-update-insurance-claim-logs:
 *   post:
 *     summary: Create or update an insurance claim log
 *     tags: [Insurance Claim logs]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/InsuranceClaimLogs'
 *           example:
 *             insurance_claim_id: 1
 *             comments: "Insurance claim approved"
 *             status: "Approved"
 *     responses:
 *       200:
 *         description: Insurance claim log created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
insuranceClaimLogsRoutes.post(
  "/create-update-insurance-claim-logs",
  verifyToken,
  controllerWrapper(createUpdateInsuranceClaimLogs)
);

/**
 * @swagger
 * /api/insuranceClaimLogs/get-insurance-claim-logs/{id}:
 *   get:
 *     summary: Get a specific insurance claim log by ID
 *     tags: [Insurance Claim logs]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the insurance claim log
 *     responses:
 *       200:
 *         description: Insurance claim log data
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/InsuranceClaimLogs'
 *       404:
 *         description: Insurance claim log not found
 *       500:
 *         description: Internal server error
 */
insuranceClaimLogsRoutes.get(
  "/get-insurance-claim-logs/:id",
  verifyToken,
  controllerWrapper(getInsuranceClaimLogs)
);

/**
 * @swagger
 * /api/insuranceClaimLogs/delete-insurance-claim-logs/{id}:
 *   delete:
 *     summary: Delete a specific insurance claim log by ID
 *     tags: [Insurance Claim logs]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the insurance claim log to delete
 *     responses:
 *       200:
 *         description: Insurance claim log successfully deleted
 *       404:
 *         description: Insurance claim log not found
 *       500:
 *         description: Internal server error
 */
insuranceClaimLogsRoutes.delete(
  "/delete-insurance-claim-logs/:id",
  verifyToken,
  controllerWrapper(deleteInsuranceClaimLogs)
);

export default insuranceClaimLogsRoutes;
