// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateParentObjective,
  deleteParentObjective,
  getParentObjective,
} from "../controller/parentObjectiveController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { parentObjectiveValidationSchema } from "../validationsRecord/validationSchema.js";
import { validateRequest } from "../helper/general.js";

/**
 * @swagger
 * tags:
 *   name: Parent Objective
 *   description: API Operations For Managing All parent objectives
 */
const parentObjectiveRoutes = Router();

/**
 * @swagger
 * /api/parentObjective/create-update-parent-objective:
 *   post:
 *     summary: Create or update a parent objective
 *     tags: [Parent Objective]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: string
 *           example:
 *             id: 1
 *             name: "Parent Objective Name"
 *             description: "Description of the parent objective"
 *             organization: "Organization Name"
 */
parentObjectiveRoutes.post(
  "/create-update-parent-objective",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      138,
      "parent_objective"
    )(req, res, next);
  },
  validateRequest(parentObjectiveValidationSchema),
  controllerWrapper(createUpdateParentObjective)
);

/**
 * @swagger
 * /api/parentObjective/get-parent-objective:
 *   get:
 *     summary: Get all parent objectives
 *     tags: [Parent Objective]
 */
parentObjectiveRoutes.get(
  "/get-parent-objective",
  verifyToken,
  sideBarPermissionCheck("view", 138, "parent_objective"),
  controllerWrapper(getParentObjective)
);

/**
 * @swagger
 * /api/parentObjective/get-parent-objective/{id}:
 *   get:
 *     summary: Get a specific parent objective by ID
 *     tags: [Parent Objective]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the parent objective
 */
parentObjectiveRoutes.get(
  "/get-parent-objective/:id",
  verifyToken,
  sideBarPermissionCheck("view", 138, "parent_objective"),
  controllerWrapper(getParentObjective)
);

/**
 * @swagger
 * /api/parentObjective/delete-parent-objective/{id}:
 *   delete:
 *     summary: Delete a specific parent objective by ID
 *     tags: [Parent Objective]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the parent objective
 */
parentObjectiveRoutes.delete(
  "/delete-parent-objective/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 138, "parent_objective"),
  controllerWrapper(deleteParentObjective)
);

export default parentObjectiveRoutes;
