// Import necessary modules
import { Router } from "express";
import { publicUrlAccess, verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  changePolicyStatus,
  createUpdatePolicy,
  deletePolicy,
  policyComment,
  viewAllPolicy,
  viewAllPolicyWorkflow,
  viewPolicyForApproval,
} from "../controller/policiesControllerOld.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Policy
 *   description: API operations for managing all Policies
 */
const policyRoutes = Router();
/**
 * @swagger
 * /api/policy/create-update-policy:
 *   post:
 *     summary: Create or update a Policy
 *     tags: [Policy]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               organization:
 *                 type: integer
 *               new_sop:
 *                 type: string
 *               policy_title:
 *                 type: string
 *               document_classification:
 *                 type: string
 *               effective_date:
 *                 type: string
 *                 format: date
 *               next_review_date:
 *                 type: string
 *                 format: date
 *               policy_status:
 *                 type: string
 *               policy_author:
 *                 type: integer
 *               policy_reviewer:
 *                 type: array
 *                 items:
 *                   type: integer
 *               policy_owner:
 *                 type: integer
 *               policy_approver:
 *                 type: integer
 *               policy_purpose:
 *                 type: string
 *               policy_scope:
 *                 type: string
 *               policy_terms:
 *                 type: array
 *                 items:
 *                   type: object
 *                   properties:
 *                     term:
 *                       type: string
 *                     description:
 *                       type: string
 *               policy_abbreviations:
 *                 type: array
 *                 items:
 *                   type: object
 *                   properties:
 *                     abbreviation:
 *                       type: string
 *                     description:
 *                       type: string
 *               roles_responsibilities:
 *                 type: array
 *                 items:
 *                   type: object
 *                   properties:
 *                     role_name:
 *                       type: integer
 *                     responsibility:
 *                       type: string
 *               policy_statement:
 *                 type: string
 *               policy_description:
 *                 type: string
 *               document_name:
 *                 type: string
 *               document_reference_number:
 *                 type: string
 *               version_number:
 *                 type: string
 *               revision_effective_date:
 *                 type: string
 *                 format: date
 *               details_of_addition:
 *                 type: string
 *               department:
 *                 type: string
 *               status:
 *                 type: string
 *           example:
 *             organization: 14
 *             new_sop: "1"
 *             policy_title: "new policy31"
 *             document_classification: "Unclassified"
 *             effective_date: "2024-10-16"
 *             next_review_date: "2024-10-11"
 *             policy_status: "Complete"
 *             policy_author: 34
 *             policy_reviewer: [34, 33]
 *             policy_owner: 34
 *             policy_approver: 34
 *             policy_purpose: "<p>q</p>"
 *             policy_scope: "<p>q</p>"
 *             policy_terms:
 *               - term: "q"
 *                 description: "<p>q</p>"
 *             policy_abbreviations:
 *               - abbreviation: "q"
 *                 description: "<p>q</p>"
 *             roles_responsibilities:
 *               - role_name: 16
 *                 responsibility: "<p>q</p>"
 *             policy_statement: "<p>q</p>"
 *             policy_description: "<p>q</p>"
 *             document_name: "q"
 *             document_reference_number: "q"
 *             version_number: ""
 *             revision_effective_date: "2024-10-28"
 *             details_of_addition: "<p>q</p>"
 *             status: "draft"
 */

policyRoutes.post(
  "/create-update-policy",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 69, "policy")(
      req,
      res,
      next
    );
  },
  controllerWrapper(createUpdatePolicy)
);

/**
 * @swagger
 * /api/policy/get-policy:
 *   get:
 *     summary: Get all list of Policies
 *     tags: [Policy]
 */
policyRoutes.get(
  "/get-policy",
  verifyToken,
  sideBarPermissionCheck("view", 69, "policy"),
  controllerWrapper(viewAllPolicyWorkflow)
);

/**
 * @swagger
 * /api/policy/get-policy-workflow:
 *   get:
 *     summary: Get all list of Policy workflows
 *     tags: [Policy]
 */
policyRoutes.get(
  "/get-policy-workflow",
  verifyToken,
  sideBarPermissionCheck("view", 69, "policy"),
  controllerWrapper(viewAllPolicyWorkflow)
);

/**
 * @swagger
 * /api/policy/get-policy/{id}:
 *   get:
 *     summary: Get a specific Policy by ID
 *     tags: [Policy]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: The ID of the Policy
 */
policyRoutes.get(
  "/get-policy/:id",
  verifyToken,
  sideBarPermissionCheck("view", 69, "policy"),
  controllerWrapper(viewAllPolicyWorkflow)
);

/**
 * @swagger
 * /api/policy/policy-comment:
 *   post:
 *     summary: Comment on Policy
 *     tags: [Policy]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               policy_id:
 *                 type: string
 *               comments:
 *                 type: string
 *           example:
 *             policy_id: "1"
 *             comments: "This is a comment."
 */
policyRoutes.post(
  "/policy-comment",
  verifyToken,
  sideBarPermissionCheck("view", 69, "policy"),
  controllerWrapper(policyComment)
);

/**
 * @swagger
 * /api/policy/policy-status-changes:
 *   post:
 *     summary: Change Policy Status
 *     tags: [Policy]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               status:
 *                 type: string
 *               policy_id:
 *                 type: string
 *               comments:
 *                 type: string
 *           example:
 *             status: "Approved"
 *             policy_id: "1"
 *             comments: "Approved for implementation."
 */
policyRoutes.post(
  "/policy-status-changes",
  verifyToken,
  // sideBarPermissionCheck("view", 69, "policy"),
  controllerWrapper(changePolicyStatus)
);

/**
 * @swagger
 * /api/policy/get-policy-public-url/{id}:
 *   get:
 *     summary: Get public URL for a specific Policy
 *     tags: [Policy]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: The ID of the Policy
 */
policyRoutes.get(
  "/get-policy-public-url/:id",
  verifyToken,
  publicUrlAccess,
  controllerWrapper(viewAllPolicy)
);

/**
 * @swagger
 * /api/policy/get-policy-for-approval:
 *   get:
 *     summary: Get all Policies for Approval
 *     tags: [Policy]
 */
policyRoutes.get(
  "/get-policy-for-approval",
  verifyToken,
  sideBarPermissionCheck("view", 69, "policy"),
  controllerWrapper(viewPolicyForApproval)
);

/**
 * @swagger
 * /api/policy/delete-policy/{id}:
 *   delete:
 *     summary: Delete a specific Policy
 *     tags: [Policy]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: The ID of the Policy
 */
policyRoutes.delete(
  "/delete-policy/:id",
  verifyToken,
  sideBarPermissionCheck("view", 69, "policy"),
  controllerWrapper(deletePolicy)
);

export default policyRoutes;
