import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import {
  createUpdateActivityScheduling,
  getStakeholderActivityScheduling,
  deleteActivityScheduling,
} from "../controller/stakeholderActivitySchedulingController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

/**
 * @swagger
 * tags:
 *   name: Stakeholder Activity Scheduling
 *   description: API operations for managing stakeholder activity scheduling
 */

const stakeholderActivitySchedulingRoutes = Router();

/**
 * @swagger
 * /api/stakeholderActivityScheduling/create-update-activity-scheduling:
 *   post:
 *     tags: [Stakeholder Activity Scheduling]
 *     summary: Create or update an activity scheduling
 *     description: Creates a new activity scheduling or updates an existing one.
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *                 description: The ID of the activity scheduling (for updates).
 *               activity_title:
 *                 type: string
 *                 description: The title of the activity.
 *               hierarchy:
 *                 type: string
 *                 description: The hierarchy level of the activity.
 *               activity_participants:
 *                 type: string
 *                 description: Participants of the activity.
 *               activity_owner:
 *                 type: integer
 *                 description: The ID of the activity owner.
 *               planned_activity_date_from:
 *                 type: string
 *                 format: date-time
 *                 description: Start date of the planned activity.
 *               planned_activity_date_to:
 *                 type: string
 *                 format: date-time
 *                 description: End date of the planned activity.
 *               venue:
 *                 type: string
 *                 description: Venue of the activity.
 *               agenda:
 *                 type: string
 *                 description: Agenda of the activity.
 *               organization:
 *                 type: integer
 *                 description: The ID of the organization related to the activity.
 *             required:
 *               - activity_title
 *               - activity_owner
 *               - planned_activity_date_from
 *               - planned_activity_date_to
 *               - venue
 *           example:
 *             {
 *               "id": 1,
 *               "activity_title": "Annual Meeting",
 *               "hierarchy": "Executive",
 *               "activity_participants": "John Doe, Jane Smith",
 *               "activity_owner": 2,
 *               "planned_activity_date_from": "2024-08-01T09:00:00Z",
 *               "planned_activity_date_to": "2024-08-01T17:00:00Z",
 *               "venue": "Conference Room A",
 *               "agenda": "Discuss annual targets and review performance",
 *               "organization": 3
 *             }
 *     responses:
 *       200:
 *         description: Activity scheduling created or updated successfully.
 *       400:
 *         description: Bad request.
 *       401:
 *         description: Unauthorized.
 */

stakeholderActivitySchedulingRoutes.post(
  "/create-update-activity-scheduling",
  verifyToken,
  //   (req, res, next) => {
  //     sideBarPermissionCheck(
  //       req.body.id ? "edit" : "create",
  //       -,
  //       "stakeholder_activity_scheduling"
  //     )(req, res, next);
  //   },
  controllerWrapper(createUpdateActivityScheduling)
);

/**
 * @swagger
 * /api/stakeholderActivityScheduling/get-stakeholder-activity-scheduling:
 *   get:
 *     tags: [Stakeholder Activity Scheduling]
 *     summary: Get a list of stakeholder activity scheduling
 *     description: Retrieves a list of all stakeholder activity scheduling.
 *     responses:
 *       200:
 *         description: A list of stakeholder activity scheduling.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                   activity_title:
 *                     type: string
 *                   hierarchy:
 *                     type: string
 *                   activity_participants:
 *                     type: string
 *                   activity_owner:
 *                     type: integer
 *                   planned_activity_date_from:
 *                     type: string
 *                     format: date-time
 *                   planned_activity_date_to:
 *                     type: string
 *                     format: date-time
 *                   venue:
 *                     type: string
 *                   agenda:
 *                     type: string
 *                   organization:
 *                     type: integer
 *             example:
 *               - id: 1
 *                 activity_title: "Annual Meeting"
 *                 hierarchy: "Executive"
 *                 activity_participants: "John Doe, Jane Smith"
 *                 activity_owner: 2
 *                 planned_activity_date_from: "2024-08-01T09:00:00Z"
 *                 planned_activity_date_to: "2024-08-01T17:00:00Z"
 *                 venue: "Conference Room A"
 *                 agenda: "Discuss annual targets and review performance"
 *                 organization: 3
 *               - id: 2
 *                 activity_title: "Team Building"
 *                 hierarchy: "Managerial"
 *                 activity_participants: "Alice Brown, Bob White"
 *                 activity_owner: 3
 *                 planned_activity_date_from: "2024-08-10T10:00:00Z"
 *                 planned_activity_date_to: "2024-08-10T15:00:00Z"
 *                 venue: "Outdoor Park"
 *                 agenda: "Team-building exercises and activities"
 *                 organization: 4
 *       401:
 *         description: Unauthorized.
 */

stakeholderActivitySchedulingRoutes.get(
  "/get-stakeholder-activity-scheduling",
  verifyToken,
  controllerWrapper(getStakeholderActivityScheduling)
);

/**
 * @swagger
 * /api/stakeholderActivityScheduling/get-stakeholder-activity-scheduling/{id}:
 *   get:
 *     tags: [Stakeholder Activity Scheduling]
 *     summary: Get an activity scheduling by ID
 *     description: Retrieves an activity scheduling by its ID.
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: The ID of the activity scheduling to retrieve.
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: The activity scheduling details.
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                 activity_title:
 *                   type: string
 *                 hierarchy:
 *                   type: string
 *                 activity_participants:
 *                   type: string
 *                 activity_owner:
 *                   type: integer
 *                 planned_activity_date_from:
 *                   type: string
 *                   format: date-time
 *                 planned_activity_date_to:
 *                   type: string
 *                   format: date-time
 *                 venue:
 *                   type: string
 *                 agenda:
 *                   type: string
 *                 organization:
 *                   type: integer
 *             example:
 *               id: 1
 *               activity_title: "Annual Meeting"
 *               hierarchy: "Executive"
 *               activity_participants: "John Doe, Jane Smith"
 *               activity_owner: 2
 *               planned_activity_date_from: "2024-08-01T09:00:00Z"
 *               planned_activity_date_to: "2024-08-01T17:00:00Z"
 *               venue: "Conference Room A"
 *               agenda: "Discuss annual targets and review performance"
 *               organization: 3
 *       401:
 *         description: Unauthorized.
 *       404:
 *         description: Activity scheduling not found.
 */

stakeholderActivitySchedulingRoutes.get(
  "/get-stakeholder-activity-scheduling/:id",
  verifyToken,
  controllerWrapper(getStakeholderActivityScheduling)
);

/**
 * @swagger
 * /api/stakeholderActivityScheduling/delete-activity-scheduling/{id}:
 *   delete:
 *     tags: [Stakeholder Activity Scheduling]
 *     summary: Delete an activity scheduling by ID
 *     description: Deletes an activity scheduling by its ID.
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: The ID of the activity scheduling to delete.
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Activity scheduling deleted successfully.
 *       401:
 *         description: Unauthorized.
 *       404:
 *         description: Activity scheduling not found.
 */

stakeholderActivitySchedulingRoutes.delete(
  "/delete-activity-scheduling/:id",
  verifyToken,
  controllerWrapper(deleteActivityScheduling)
);

export default stakeholderActivitySchedulingRoutes;
