import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

import {
  createUpdateSurveyDesign,
  getSurveyDesign,
  deleteSurveyDesign,
  sendOtpToExternalUser,
  verifySurveyOtp,
  submitSurveyResponse,
  shareToInternal,
  sendToMail,
  urlShorten,
  enableDisableSurvey,
} from "../controller/surveyDesignController.js";

const surveyDesignRoutes = Router();

/**
 * @swagger
 * components:
 *   schemas:
 *     SurveyDesign:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated id of the survey design
 *         name:
 *           type: string
 *           description: Name of the survey design
 *         description:
 *           type: string
 *           description: Description of the survey design
 *         quiz:
 *           type: integer
 *           description: Quiz associated with the survey design
 *         survey_analysis:
 *           type: string
 *           description: Survey analysis
 *         organization:
 *           type: integer
 *           description: Organization id
 *         department:
 *           type: integer
 *           description: Department id
 *       example:
 *         imd: 1
 *         nae: Survey 1
 *         description: This is a survey
 *         quiz: 1
 *         survey_analysis: Some analysis
 *         organization: 1
 *         department: 1
 *         questions : [{} , {}]
 */

/**
 * @swagger
 * /api/surveyDesign/create-update-survey-design:
 *   post:
 *     summary: Create or update a survey design
 *     tags: [SurveyDesign]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             imd: 1
 *             nae: Survey 1
 *             description: This is a survey
 *             quiz: 1
 *             survey_analysis: Some analysis
 *             organization: 1
 *             department: 1
 *             questions : [{} , {}]
 *           schema:
 *             $ref: '#/components/schemas/SurveyDesign'
 *     responses:
 *       200:
 *         description: The created/updated survey design
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/SurveyDesign'
 */
surveyDesignRoutes.post(
  "/create-update-survey-design",
  verifyToken,
  controllerWrapper(createUpdateSurveyDesign)
);

/**
 * @swagger
 * /api/surveyDesign/get-survey-design:
 *   get:
 *     summary: Get all survey designs
 *     tags: [SurveyDesign]
 *     responses:
 *       200:
 *         description: List of all survey designs
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/SurveyDesign'
 */
surveyDesignRoutes.get(
  "/get-survey-design",
  verifyToken,
  controllerWrapper(getSurveyDesign)
);

/**
 * @swagger
 * /api/surveyDesign/get-public-survey-design/{id}:
 *   get:
 *     summary: Get all survey designs
 *     tags: [SurveyDesign]
 *     responses:
 *       200:
 *         description: List of all survey designs
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/SurveyDesign'
 */
surveyDesignRoutes.get(
  "/get-public-survey-design/:id",
  controllerWrapper(getSurveyDesign)
);

/**
 * @swagger
 * /api/surveyDesign/get-survey-design/{id}:
 *   get:
 *     summary: Get a survey design by id
 *     tags: [SurveyDesign]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: Id of the survey design
 *     responses:
 *       200:
 *         description: Survey design details
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/SurveyDesign'
 */
surveyDesignRoutes.get(
  "/get-survey-design/:id",
  verifyToken,
  controllerWrapper(getSurveyDesign)
);

/**
 * @swagger
 * /api/surveyDesign/delete-survey-design/{id}:
 *   delete:
 *     summary: Delete a survey design by id
 *     tags: [SurveyDesign]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: Id of the survey design
 *     responses:
 *       200:
 *         description: Survey design deleted
 */
surveyDesignRoutes.delete(
  "/delete-survey-design/:id",
  verifyToken,
  controllerWrapper(deleteSurveyDesign)
);

/**
 * @swagger
 * /api/surveyDesign/send-otp-to-external:
 *   post:
 *     summary: Send otp to external user
 *     tags: [SurveyDesign]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             name : "name"
 *             email : "email"
 *             survey_id : 1
 *     responses:
 *       200:
 *         description: Otp sent to email successfully
 *
 */

surveyDesignRoutes.post(
  "/send-otp-to-external",
  controllerWrapper(sendOtpToExternalUser)
);

/**
 * @swagger
 * /api/surveyDesign/verify-external-otp:
 *   post:
 *     summary: Verify external user's otp
 *     tags: [SurveyDesign]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             email : "email"
 *             otp : "otp"
 *             survey_id : 1
 *     responses:
 *       200:
 *         description: OTP verified successfully
 *         id : response ID
 */

surveyDesignRoutes.post(
  "/verify-external-otp",
  controllerWrapper(verifySurveyOtp)
);

/**
 * @swagger
 * /api/surveyDesign/submit-survey-response:
 *   post:
 *     summary: Submit survey response
 *     tags: [SurveyDesign]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             values : [{} , {}]
 *             id : "response id"
 *     responses:
 *       200:
 *         description: Record updated successfully
 */

surveyDesignRoutes.post(
  "/submit-survey-response",
  controllerWrapper(submitSurveyResponse)
);

/**
 * @swagger
 * /api/surveyDesign/share-to-internal:
 *   post:
 *     summary: Share to internal person
 *     tags: [SurveyDesign]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             user_id : "userId"
 *             id : "survey design id"
 *     responses:
 *       200:
 *         description: Survey sent successfully
 */

surveyDesignRoutes.post(
  "/share-to-internal",
  verifyToken,
  controllerWrapper(shareToInternal)
);

/**
 * @swagger
 * /api/surveyDesign/share-to-mail:
 *   post:
 *     summary: Share to external person via mail
 *     tags: [SurveyDesign]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             message : "message"
 *             to : "to email"
 *             subject : "subject"
 *     responses:
 *       200:
 *         description: Email sent successfully
 */

surveyDesignRoutes.post("/share-to-mail", controllerWrapper(sendToMail));

/**
 * @swagger
 * /api/surveyDesign/shorten-url:
 *   post:
 *     summary: Shorten Url for survey
 *     tags: [SurveyDesign]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             url : "url"
 *     responses:
 *       200:
 *         description: shortUrl
 */

surveyDesignRoutes.post("/shorten-url", controllerWrapper(urlShorten));

/**
 * @swagger
 * /api/surveyDesign/enable-disable-survey:
 *   post:
 *     summary: Enable disable a survey
 *     tags: [SurveyDesign]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             status : 1/0
 *             id : "survey design id"
 *     responses:
 *       200:
 *         description: shortUrl
 */

surveyDesignRoutes.post(
  "/enable-disable-survey",
  verifyToken,
  controllerWrapper(enableDisableSurvey)
);

export default surveyDesignRoutes;

