import { Router } from "express";
import {
  createUpdateAttendanceRecording,
  getAttendanceRecording,
  deleteAttendanceRecording,
  getTrainingReport,
} from "../controller/trainingAttendanceRecordingController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const attendanceRecordingRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Training Attendance Recording
 *   description: API operations for managing training attendance recordings
 */

/**
 * @swagger
 * /api/attendanceRecording/create-update-attendance-recording:
 *   post:
 *     summary: Create or update an attendance recording
 *     tags: [Training Attendance Recording]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/AttendanceRecording'
 *           example:
 *             organization: 1
 *             department: 2
 *             training_awareness_program: 3
 *             training_planned_start_date: "2024-07-17"
 *             training_actual_date: "2024-07-18"
 *             actual_training_service_provider: "Provider Name"
 *             training_venue: "Conference Room 1"
 *             training_total_cost: "1000"
 *             training_participation: "50 participants"
 *             document: "document_link"
 *     responses:
 *       200:
 *         description: The created or updated attendance recording
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/AttendanceRecording'
 */
attendanceRecordingRoutes.post(
  "/create-update-attendance-recording",
  verifyToken,
  controllerWrapper(createUpdateAttendanceRecording)
);

/**
 * @swagger
 * /api/attendanceRecording/get-attendance-recording:
 *   get:
 *     summary: Get all attendance recordings
 *     tags: [Training Attendance Recording]
 *     responses:
 *       200:
 *         description: A list of attendance recordings
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/AttendanceRecording'
 */
attendanceRecordingRoutes.get(
  "/get-attendance-recording",
  verifyToken,
  sideBarPermissionCheck("view", 210, "attendance_recording"),
  controllerWrapper(getAttendanceRecording)
);

/**
 * @swagger
 * /api/attendanceRecording/get-attendance-recording/{id}:
 *   get:
 *     summary: Get an attendance recording by ID
 *     tags: [Training Attendance Recording]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the attendance recording
 *     responses:
 *       200:
 *         description: The requested attendance recording
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/AttendanceRecording'
 */
attendanceRecordingRoutes.get(
  "/get-attendance-recording/:id",
  verifyToken,
  sideBarPermissionCheck("view", 210, "attendance_recording"),
  controllerWrapper(getAttendanceRecording)
);

/**
 * @swagger
 * /api/attendanceRecording/delete-attendance-recording/{id}:
 *   delete:
 *     summary: Delete an attendance recording by ID
 *     tags: [Training Attendance Recording]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the attendance recording
 *     responses:
 *       200:
 *         description: The attendance recording was deleted
 *         content:
 *           application/json:
 *             example:
 *               message: "Attendance recording deleted successfully"
 */
attendanceRecordingRoutes.delete(
  "/delete-attendance-recording/:id",
  verifyToken,
  sideBarPermissionCheck("view", 210, "attendance_recording"),
  controllerWrapper(deleteAttendanceRecording)
);

/**
 * @swagger
 * /api/attendanceRecording/get-training-report:
 *   get:
 *     summary: Get all Training Reports
 *     tags: [Training Attendance Recording]
 *     responses:
 *       200:
 *         description: A list of Training Reports (List of Employees vs Training Name ,  Who are scheduled for the training , Who have completed the training and who have not completed the training)
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/AttendanceRecording'
 */
attendanceRecordingRoutes.get(
  "/get-training-report",
  verifyToken,
  sideBarPermissionCheck("view", 210, "attendance_recording"),
  controllerWrapper(getTrainingReport)
);

export default attendanceRecordingRoutes;
