import { Router } from "express";
import {
  createUpdateTrainingCertificate,
  getTrainingCertificate,
  deleteTrainingCertificate,
  getTrainingCertificateEmployee,
} from "../controller/trainingCertificateController.js";

import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

/**
 * @swagger
 * tags:
 *   name: Training Certificate
 *   description: API operations for managing training certificates
 */

const trainingCertificateRoutes = Router();

/**
 * @swagger
 * /api/trainingCertificate/create-update-training-certificate:
 *   post:
 *     summary: Create or update a training certificate
 *     tags: [Training Certificate]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *                 example: 1
 *               attendees:
 *                 type: string
 *                 example: "John Doe, Jane Smith"
 *               certificate_issued:
 *                 type: string
 *                 example: "Yes"
 *               training_program:
 *                 type: integer
 *                 example: 5
 *               deleted:
 *                 type: integer
 *                 example: 0
 *               created_at:
 *                 type: string
 *                 format: date-time
 *               created_by:
 *                 type: integer
 *                 example: 1
 *               updated_at:
 *                 type: string
 *                 format: date-time
 *               updated_by:
 *                 type: integer
 *                 example: 1
 *     responses:
 *       200:
 *         description: Training certificate created or updated successfully
 *       400:
 *         description: Bad request
 */
trainingCertificateRoutes.post(
  "/create-update-training-certificate",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      211,
      "training_certificate"
    )(req, res, next);
  },
  controllerWrapper(createUpdateTrainingCertificate)
);

/**
 * @swagger
 * /api/trainingCertificate/get-training-certificate:
 *   get:
 *     summary: Get all training certificates
 *     tags: [Training Certificate]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: A list of training certificates
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                     example: 1
 *                   attendees:
 *                     type: string
 *                     example: "John Doe, Jane Smith"
 *                   certificate_issued:
 *                     type: string
 *                     example: "Yes"
 *                   training_program:
 *                     type: integer
 *                     example: 5
 *                   deleted:
 *                     type: integer
 *                     example: 0
 *                   created_at:
 *                     type: string
 *                     format: date-time
 *                   created_by:
 *                     type: integer
 *                     example: 1
 *                   updated_at:
 *                     type: string
 *                     format: date-time
 *                   updated_by:
 *                     type: integer
 *                     example: 1
 *       400:
 *         description: Bad request
 */
trainingCertificateRoutes.get(
  "/get-training-certificate",
  verifyToken,
  sideBarPermissionCheck("view", 211, "training_certificate"),
  controllerWrapper(getTrainingCertificate)
);

/**
 * @swagger
 * /api/trainingCertificate/get-training-certificate/{id}:
 *   get:
 *     summary: Get a training certificate by ID
 *     tags: [Training Certificate]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the training certificate
 *     responses:
 *       200:
 *         description: A training certificate
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                   example: 1
 *                 attendees:
 *                   type: string
 *                   example: "John Doe, Jane Smith"
 *                 certificate_issued:
 *                   type: string
 *                   example: "Yes"
 *                 training_program:
 *                   type: integer
 *                   example: 5
 *                 deleted:
 *                   type: integer
 *                   example: 0
 *                 created_at:
 *                   type: string
 *                   format: date-time
 *                 created_by:
 *                   type: integer
 *                   example: 1
 *                 updated_at:
 *                   type: string
 *                   format: date-time
 *                 updated_by:
 *                   type: integer
 *                   example: 1
 *       400:
 *         description: Bad request
 *       404:
 *         description: Training certificate not found
 */
trainingCertificateRoutes.get(
  "/get-training-certificate/:id",
  verifyToken,
  sideBarPermissionCheck("view", 211, "training_certificate"),
  controllerWrapper(getTrainingCertificate)
);

/**
 * @swagger
 * /api/trainingCertificate/delete-training-certificate/{id}:
 *   delete:
 *     summary: Delete a training certificate by ID
 *     tags: [Training Certificate]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the training certificate to delete
 *     responses:
 *       200:
 *         description: Training certificate deleted successfully
 *       400:
 *         description: Bad request
 *       404:
 *         description: Training certificate not found
 */
trainingCertificateRoutes.delete(
  "/delete-training-certificate/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 211, "training_certificate"),
  controllerWrapper(deleteTrainingCertificate)
);

trainingCertificateRoutes.get(
  "/get-training-certificate-employee/:user_id",
  verifyToken,
  controllerWrapper(getTrainingCertificateEmployee)
);

export default trainingCertificateRoutes;
