import { Router } from "express";
import {
  createUpdateEffectivenessOutcome,
  getEffectivenessOutcome,
  deleteEffectivenessOutcome,
} from "../controller/trainingEffectivenessOutcomeController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const effectivenessOutcomeRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Training Effectiveness Outcome
 *   description: API operations for managing training effectiveness outcomes
 */

/**
 * @swagger
 * /api/effectivenessOutcome/create-update-effectiveness-outcome:
 *   post:
 *     summary: Create or update an effectiveness outcome
 *     tags: [Training Effectiveness Outcome]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/EffectivenessOutcome'
 *           example:
 *             employee_name: "John"
 *             surname: "Doe"
 *             employee_id: "E12345"
 *             role_name: "Manager"
 *             training_awareness_program: "Leadership Training"
 *             training_date: "2024-07-17"
 *             training_venue: "Conference Room 1"
 *             administration_and_resources: "All resources were provided"
 *             the_training_facilitator: "Excellent facilitator"
 *             study_material: "Comprehensive study material"
 *             training_assessment: "Positive feedback"
 *             any_other_comments: "None"
 *             organization: 1
 *             department: 2
 *     responses:
 *       200:
 *         description: The created or updated effectiveness outcome
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EffectivenessOutcome'
 */
effectivenessOutcomeRoutes.post(
  "/create-update-effectiveness-outcome",
  verifyToken,
  (req , res, next)=>{
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 216, "effectiveness_outcome")(req, res, next);
  },
  controllerWrapper(createUpdateEffectivenessOutcome)
);

/**
 * @swagger
 * /api/effectivenessOutcome/get-effectiveness-outcome:
 *   get:
 *     summary: Get all effectiveness outcomes
 *     tags: [Training Effectiveness Outcome]
 *     responses:
 *       200:
 *         description: A list of effectiveness outcomes
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/EffectivenessOutcome'
 */
effectivenessOutcomeRoutes.get(
  "/get-effectiveness-outcome",
  verifyToken,
  sideBarPermissionCheck("view", 216, "effectiveness_outcome"),
  controllerWrapper(getEffectivenessOutcome)
);

/**
 * @swagger
 * /api/effectivenessOutcome/get-effectiveness-outcome/{id}:
 *   get:
 *     summary: Get an effectiveness outcome by ID
 *     tags: [Training Effectiveness Outcome]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the effectiveness outcome
 *     responses:
 *       200:
 *         description: The requested effectiveness outcome
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EffectivenessOutcome'
 */
effectivenessOutcomeRoutes.get(
  "/get-effectiveness-outcome/:id",
  verifyToken,
  sideBarPermissionCheck("view", 216, "effectiveness_outcome"),
  controllerWrapper(getEffectivenessOutcome)
);

/**
 * @swagger
 * /api/effectivenessOutcome/delete-effectiveness-outcome/{id}:
 *   delete:
 *     summary: Delete an effectiveness outcome by ID
 *     tags: [Training Effectiveness Outcome]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the effectiveness outcome
 *     responses:
 *       200:
 *         description: The effectiveness outcome was deleted
 *         content:
 *           application/json:
 *             example:
 *               message: "Effectiveness outcome deleted successfully"
 */
effectivenessOutcomeRoutes.delete(
  "/delete-effectiveness-outcome/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 216, "effectiveness_outcome"),
  controllerWrapper(deleteEffectivenessOutcome)
);

export default effectivenessOutcomeRoutes;
