import { DataTypes } from "sequelize";
import sequelize from "./sequelize.js"; // Adjust the import path as necessary
import User from "./UserSchema.js";
import Meeting from "./MeetingSchema.js";
import Organization from "./OrganizationSchema.js";
import Department from "./DepartmentSchema.js";

const OperationalRisk = sequelize.define(
  "OperationalRisk",
  {
    id: {
      type: DataTypes.INTEGER,
      autoIncrement: true,
      primaryKey: true,
    },
    risk_register_name: {
      type: DataTypes.STRING(250),
      allowNull: true,
    },
    assessment_date: {
      type: DataTypes.DATEONLY,
      allowNull: true,
    },
    review_date: {
      type: DataTypes.DATEONLY,
      allowNull: true,
    },
    register_status: {
      type: DataTypes.STRING(255),
    },
    main_process: {
      type: DataTypes.INTEGER,
    },
    work_area_id: {
      type: DataTypes.INTEGER,
    },
    risk_context: {
      type: DataTypes.TEXT("long"),
      allowNull: true,
    },
    control_assessments: {
      type: DataTypes.TEXT("long"), // JSON stored as long text
      allowNull: true,
      get() {
        const value = this.getDataValue("control_assessments");
        return value ? JSON.parse(value) : null;
      },
      set(value) {
        this.setDataValue("control_assessments", JSON.stringify(value));
      },
    },
    opportunity_description: {
      type: DataTypes.TEXT("long"), // JSON stored as long text
      allowNull: true,
      get() {
        const value = this.getDataValue("opportunity_description");
        return value ? JSON.parse(value) : null;
      },
      set(value) {
        this.setDataValue("opportunity_description", JSON.stringify(value));
      },
    },
    risk_assessment_facilitator: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
    },
    register_assessment_facilitator: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
    },
    approver: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
    },
    meeting_id: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: Meeting,
        key: "id",
      },
    },
    venue: {
      type: DataTypes.STRING(250),
      allowNull: true,
    },
    time: {
      type: DataTypes.STRING(250),
      allowNull: true,
    },
    attendance_date: {
      type: DataTypes.DATEONLY,
      allowNull: true,
    },
    attendees: {
      type: DataTypes.TEXT("long"),
      allowNull: true,
    },
    unique_id: {
      type: DataTypes.STRING(250),
      allowNull: true,
    },
    reference_no: {
      type: DataTypes.STRING(250),
      allowNull: true,
    },
    organization: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: Organization,
        key: "id",
      },
    },
    department: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: Department,
        key: "id",
      },
    },
    approval_status: {
      type: DataTypes.ENUM("Pending Approval", "Approved", "Rejected"),
      defaultValue: "Pending Approval",
    },
    status: {
      type: DataTypes.STRING(255),
      allowNull: true,
      defaultValue: "Draft",
    },
    reject_comment: {
      type: DataTypes.TEXT("long"),
      allowNull: true,
    },
    created_at: {
      type: DataTypes.DATE,
      allowNull: true,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    created_by: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
    },
    updated_at: {
      type: DataTypes.DATE,
      allowNull: true,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    updated_by: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
    },
    deleted: {
      type: DataTypes.INTEGER,
      allowNull: true,
      defaultValue: 0,
    },
  },
  {
    tableName: "operational_risk",
    timestamps: false,
    charset: "utf8mb4",
    collate: "utf8mb4_general_ci",
  }
);

// Define associations
OperationalRisk.belongsTo(User, {
  foreignKey: "risk_assessment_facilitator",
  as: "RiskAssessmentFacilitator",
});
OperationalRisk.belongsTo(User, {
  foreignKey: "register_assessment_facilitator",
  as: "RegisterAssessmentFacilitator",
});
OperationalRisk.belongsTo(User, {
  foreignKey: "risk_register_approver",
  as: "RiskRegisterApprover",
});
OperationalRisk.belongsTo(Meeting, { foreignKey: "meeting_id", as: "Meeting" });
OperationalRisk.belongsTo(Organization, {
  foreignKey: "organization",
  as: "Organization",
});
OperationalRisk.belongsTo(Department, {
  foreignKey: "department",
  as: "Department",
});
OperationalRisk.belongsTo(User, { foreignKey: "created_by", as: "CreatedBy" });
OperationalRisk.belongsTo(User, { foreignKey: "updated_by", as: "UpdatedBy" });
// console.log(await OperationalRisk.sync({ alter: true }));
export default OperationalRisk;
