// TestingAndExaminationSchema.js
import sequelize from "./sequelize.js"; // Adjust the import path as necessary
import Organization from "./OrganizationSchema.js";
import User from "./UserSchema.js";
import Department from "./DepartmentSchema.js";
import EmployeeType from "./EmployeeTypeSchema.js"; // Assuming an EmployeeType schema exists
import ContractorRegistration from "./ContractorRegistrationSchema.js"; // Assuming a Contractor schema exists
import Location from "./LocationSchema.js"; // Assuming a Location schema exists
import Currency from "./CurrencySchema.js"; // Assuming a Currency schema exists
import { DataTypes } from "sequelize";
// may be not in use
const TestingAndExamination = sequelize.define(
  "TestingAndExamination",
  {
    id: {
      type: DataTypes.INTEGER,
      autoIncrement: true,
      primaryKey: true,
    },
    testing_and_examination_name: {
      type: DataTypes.STRING(250), // Testing and Examination Name
      allowNull: true,
    },
    testing_and_examination_type: {
      type: DataTypes.INTEGER, // Dropdown for Testing and Examination Type
      allowNull: true,
    },
    planned_testing_and_examination_start_date: {
      type: DataTypes.DATE, // Planned Testing and Examination Start Date
      allowNull: true,
    },
    planned_testing_and_examination_end_date: {
      type: DataTypes.DATE, // Planned Testing and Examination End Date
      allowNull: true,
    },
    testing_and_examination_source: {
      type: DataTypes.ENUM("Internal", "External"), // Testing and Examination Source
      allowNull: true,
    },
    // Internal Surveyor
    internal_testing_and_examination_ids: {
      type: DataTypes.INTEGER, // Surveyor Name fetched from Employee List
      allowNull: true,
      references: {
        model: EmployeeType,
        key: "id",
      },
    },
    // External Surveyor
    external_testing_and_examination_ids: {
      type: DataTypes.ARRAY(DataTypes.INTEGER), // Multi-select for external surveyors (fetched from contractors)
      allowNull: true,
      references: {
        model: ContractorRegistration,
        key: "id",
      },
    },
    testing_and_examination_owner: {
      type: DataTypes.STRING(250), // Testing and Examination Owner
      allowNull: true,
    },
    // Stakeholders Involved
    stakeholders_involved: {
      type: DataTypes.ENUM("Internal", "External"),
      allowNull: true,
    },
    internal_stakeholder_ids: {
      type: DataTypes.ARRAY(DataTypes.INTEGER), // Multi-select for internal stakeholders
      allowNull: true,
      references: {
        model: EmployeeType,
        key: "id",
      },
    },
    external_stakeholder_name: {
      type: DataTypes.STRING(250), // Free Text for External Stakeholder Name
      allowNull: true,
    },
    external_stakeholder_email: {
      type: DataTypes.STRING(250), // Email Input for External Stakeholder
      allowNull: true,
      validate: {
        isEmail: true, // Validate email format
      },
    },
    testing_and_examination_location_ids: {
      type: DataTypes.ARRAY(DataTypes.INTEGER), // Multi-select Dropdown for Testing and Examination Locations
      allowNull: true,
      references: {
        model: Location,
        key: "id",
      },
    },
    testing_and_examination_scope: {
      type: DataTypes.TEXT("long"), // Description Field for Testing and Examination Scope
      allowNull: true,
    },
    critical_business_processes: {
      type: DataTypes.TEXT, // Critical Business Processes
      allowNull: true,
    },

    // Technological Resources
    technological_resources: {
      type: DataTypes.TEXT, // Add More field for Technological Resources
      allowNull: true,
    },

    // Financial Resources
    financial_resources: {
      type: DataTypes.ENUM("Financial Resource Type", "Value", "Currency"),
      allowNull: true,
    },
    financial_resource_type: {
      type: DataTypes.ENUM("Operating Budget", "Capital Budget"), // Enum for Financial Resource Type
      allowNull: true,
    },
    value: {
      type: DataTypes.FLOAT, // Numerical Input for Financial Value
      allowNull: true,
      validate: {
        isFloat: true, // Validate float input
        min: 0, // Minimum value constraint
      },
    },
    currency_id: {
      type: DataTypes.INTEGER, // Dropdown for Currency, referencing prefilled dataset
      allowNull: true,
      references: {
        model: Currency,
        key: "id",
      },
    },
    evaluation_tool: {
      type: DataTypes.ENUM("Applicable", "Not Applicable"), // Dropdown for Evaluation Tool
      allowNull: true,
    },
    organization: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: Organization,
        key: "id",
      },
    },
    created_at: {
      type: DataTypes.DATE,
      allowNull: true,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    created_by: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
    },
    deleted: {
      type: DataTypes.TINYINT,
      allowNull: true,
      defaultValue: 0,
    },
    updated_at: {
      type: DataTypes.DATE,
      allowNull: true,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
      onUpdate: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    updated_by: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
    },
    department: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: Department,
        key: "id",
      },
    },
  },
  {
    tableName: "testing_and_examination",
    timestamps: false,
    charset: "utf8mb4",
    collate: "utf8mb4_general_ci",
  }
);

// Associations
TestingAndExamination.belongsTo(EmployeeType, {
  as: "internalTestingAndExamination",
  foreignKey: "internal_testing_and_examination_ids",
});
TestingAndExamination.belongsToMany(ContractorRegistration, {
  as: "testers",
  through: "TestingAndExaminationTesters",
  foreignKey: "testing_and_examination_id",
});
TestingAndExamination.belongsToMany(EmployeeType, {
  as: "internalStakeholders",
  through: "TestingAndExaminationInternalStakeholders",
  foreignKey: "testing_and_examination_id",
});
TestingAndExamination.belongsToMany(Location, {
  as: "testingAndExaminationLocations",
  through: "TestingAndExaminationLocationMapping",
  foreignKey: "testing_and_examination_id",
});
TestingAndExamination.belongsTo(Currency, {
  as: "currency",
  foreignKey: "currency_id",
});
TestingAndExamination.belongsTo(Organization, {
  as: "org",
  foreignKey: "organization",
});
TestingAndExamination.belongsTo(User, {
  as: "creator",
  foreignKey: "created_by",
});
TestingAndExamination.belongsTo(User, {
  as: "updater",
  foreignKey: "updated_by",
});
TestingAndExamination.belongsTo(Department, {
  as: "dep",
  foreignKey: "department",
});

export default TestingAndExamination;
