import axios from "axios";
import express from "express";
import supertest from "supertest";
import dotenv from "dotenv";
import { expect } from "chai";
import { describe, it } from "test";

dotenv.config();
const SWAGGER_URL = `http://192.168.1.52:2209/docs/swagger.json`;

const fetchSwagger = async () => {
  try {
    const response = await axios.get(SWAGGER_URL);
    return response.data;
  } catch (error) {
    console.error("Error fetching Swagger JSON:", error.message);
    throw error;
  }
};

const filterPathsByController = (swaggerDoc, controllerName) => {
  const filteredPaths = {};
  for (const path in swaggerDoc.paths) {
    for (const method in swaggerDoc.paths[path]) {
      const endpoint = swaggerDoc.paths[path][method];
      if (endpoint.tags && endpoint.tags.includes(controllerName)) {
        if (!filteredPaths[path]) {
          filteredPaths[path] = {};
        }
        filteredPaths[path][method] = endpoint;
      }
    }
  }
  return filteredPaths;
};

const generateTestsFromSwagger = (swaggerDoc) => {
  const tests = [];
  for (const path in swaggerDoc) {
    for (const method in swaggerDoc[path]) {
      const endpoint = swaggerDoc[path][method];
      const test = {
        method,
        path,
        payload: generatePayload(endpoint.parameters),
        expectedStatus: 200, // Adjust based on your swagger doc or default responses
        expectedResponse: {
          status: true,
          message: "Record Created successfully",
        }, // Adjust based on your swagger doc or example responses
      };
      tests.push(test);
    }
  }
  return tests;
};

const generatePayload = (parameters) => {
  const payload = {};
  if (!parameters) return payload;
  parameters.forEach((param) => {
    if (param.in === "body" && param.schema) {
      Object.assign(payload, param.schema.example || {});
    } else if (param.in === "query") {
      payload[param.name] = param.example || "";
    } else if (param.in === "path") {
      payload[param.name] = param.example || "";
    }
  });
  return payload;
};

const createServerWithController = (controller) => {
  const app = express();
  app.use(express.json());
  app.use("/api", controller);
  return app;
};

const runTests = async (controller, controllerName) => {
  try {
    const swaggerDoc = await fetchSwagger();
    const filteredPaths = filterPathsByController(swaggerDoc, controllerName);
    const tests = generateTestsFromSwagger(filteredPaths);

    const app = createServerWithController(controller);
    const request = supertest(app);

    describe(`${controllerName} Controller Tests`, function () {
      tests.forEach((test) => {
        it(`should handle ${test.method.toUpperCase()} ${
          test.path
        }`, async function () {
          const response = await request[test.method](test.path).send(
            test.payload,
          );
          expect(response.status).to.equal(test.expectedStatus);
          if (test.expectedResponse) {
            expect(response.body).to.deep.equal(test.expectedResponse);
          }
        });
      });
    });
  } catch (error) {
    console.error("Error running tests:", error.message);
  }
};

import { createUpdateCompetencyAssessment } from "./controller/competencyAssessmentController.js";

const controllerName = "Competency Assessment"; // Change to match your controller's tag

runTests(createUpdateCompetencyAssessment, controllerName)
  .then(() => console.log("All tests completed"))
  .catch((err) => console.error("Test execution failed:", err));

// const trainingResponseSection = [
//   {
//     0: "N/A",
//     1: "0",
//     2: "1",
//     category: "Administration and Resources",
//     items: [
//       "Opinion of the Training Facility on your first arrival",
//       "The physical conditions (parking, seating, ventilation, restrooms, policy display, noise levels and lighting)",
//       "The emergency evacuation was conducted prior to the training",
//     ],
//   },
//   {
//     0: "N/A",
//     1: "0",
//     2: "1",
//     3: "2",
//     4: "3",
//     category: "The Training Facilitator",
//     items: [
//       "Preparedness of the Training Facilitator",
//       "Knowledgeability of the Training Facilitator",
//       "Professional appearance of the Training Facilitator",
//       "Openness and responsiveness of the Training Facilitator to questions",
//       "Time management of the Training Facilitator",
//     ],
//   },
//   {
//     0: "N/A",
//     1: "0",
//     2: "1",
//     3: "2",
//     category: "Study Material",
//     items: [
//       "Relevance to content of modules/handouts/study guides",
//       "Professionalism of the study material",
//       "Availability of technical assistance on the content",
//       "Availability of required study tools and equipment where applicable",
//     ],
//   },
//   {
//     0: "N/A",
//     1: "0",
//     2: "1",
//     3: "2",
//     4: "3",
//     5: "4",
//     category: "Training Assessment",
//     items: [
//       "Clarity of the assessment requirements (e.g. minimum competence requirements)",
//       "Relevance to theory content",
//       "Duration of Assessment",
//       "Complexity level of Assessment",
//       "Availability of required assessment tools and equipment",
//       "Condition of the assessment tools and equipment",
//     ],
//   },
// ];

// function calculateAverages(trainingResponseSection) {
//   const results = {
//     categoryAverages: {},
//     overallAverage: 0,
//   };
//   let overallSum = 0;
//   let overallCount = 0;

//   for (const section of trainingResponseSection) {
//     let categorySum = 0;
//     let categoryCount = 0;

//     // Loop through the keys to find numeric scores
//     for (const key in section) {
//       if (!isNaN(key) && !isNaN(parseFloat(section[key]))) {
//         categorySum += parseFloat(section[key]);
//         categoryCount++;
//       }
//     }

//     // Calculate category average
//     if (categoryCount > 0) {
//       const categoryAverage = categorySum / categoryCount;
//       results.categoryAverages[section.category] = categoryAverage;

//       // Add to overall sum and count
//       overallSum += categorySum;
//       overallCount += categoryCount;
//     }
//   }

//   // Calculate overall average
//   if (overallCount > 0) {
//     results.overallAverage = overallSum / overallCount;
//   }

//   return results;
// }

// // Example Usage
// const averages = calculateAverages(trainingResponseSection);
// console.log(averages);
