import db from "../db-config.js";
import {
  checkCircularDependency,
  countQueryCondition,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  insertNotification,
  makeJoins,
  orderTasks,
  processesSingleDDRMDocument,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  uploadFile,
  whereCondition,
} from "../helper/general.js";
import { createQueryBuilder } from "../helper/queryBuilder.js";
import { sendResponse } from "../helper/wrapper.js";
import ActionTemplateList from "../sequelize/ActionTemplateListSchema.js";
import ActionTemplateComment from "../sequelize/ActionCommentSchema.js";

export const createUpdateActionTemplate = async (req, res) => {
  const { id, department, predecessor_task_template, successor_task_template , sidebar_id =  233} =
    req.body;
  // console.log(req.body);

  if ((predecessor_task_template || successor_task_template) && !id) {
    const circularDependency = await checkCircularDependency(
      id, // This might be undefined for new records
      predecessor_task_template,
      successor_task_template
    );
    if (circularDependency) {
      return sendResponse(
        res,
        400,
        "Circular dependency detected. Cannot create or update the record."
      );
    }
  }

  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }

  req.body.template_version = await uniqueIdGenerator(
    req.body.organization,
    department,
    "ACT",
    "action_template_list",
    "template_version"
  );

  if(!id){
    req.body.unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "ACT",
      "action_template_list",
      "template_version"
    );
  }

  // await db.query(`SELECT * FROM success`)
  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  if (req.files && req.files.attachment) {
    // req.body.attachment = await uploadFile(
    //   "action_template_list",
    //   req.files.attachment
    // );
    const file = req.files.attachment;
    const ddrm_id = await processesSingleDDRMDocument(
      "action_template_list",
      sidebar_id,
      file,
      req
    );
    req.body.ddrm_id = ddrm_id;
  }
  let arr = [];
  let created_user = req.user.sessionid;
  if (req.body.save_type == "draft") {
    req.body.status = "Draft";
    req.body.for_approval = 0;
  } else {
    if (created_user == 1) {
      req.body.status = "Approved";
      req.body.for_approval = 1;
    } else {
      const [teams] = await db.query(
        `SELECT * FROM teams WHERE JSON_CONTAINS(team_members, '${created_user}') AND deleted = 0`
      );
      if (teams.length > 0) {
        arr.push(teams[0].team_leader);
      }
      if (arr.length == 0 || arr[0] == null) {
        req.body.status = "Approved";
        req.body.for_approval = 1;
      } else {
        req.body.send_to_employee = arr;
        req.body.for_approval = 1;
      }
    }
  }
  if (id != undefined) {
    const [result] = await db.query(
      `SELECT * FROM action_template_list WHERE id = ${id}`
    );
    await db.query(
      `UPDATE action_template_list SET predecessor_task_template = ${result[0]?.predecessor_task_template} WHERE id = ${result[0]?.successor_task_template}`
    );
    await db.query(
      `UPDATE action_template_list SET successor_task_template = ${result[0]?.successor_task_template} WHERE id = ${result[0]?.predecessor_task_template}`
    );
  }
  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id
    ? updateQueryBuilder(ActionTemplateList, req.body)
    : createQueryBuilder(ActionTemplateList, req.body);
  let [result] = await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Action Template",
    id ? id : result.insertId
  );
  if (predecessor_task_template) {
    let predId = predecessor_task_template;
    await db.query(
      `UPDATE action_template_list SET successor_task_template = ${
        id ? id : result.insertId
      } WHERE id = ${predId}`
    );
  }
  if (successor_task_template) {
    let sucId = successor_task_template;
    await db.query(
      `UPDATE action_template_list SET predecessor_task_template = ${
        id ? id : result.insertId
      } WHERE id = ${sucId}`
    );
  }
  for (const val of arr) {
    await insertNotification(
      "Action Template Approval",
      `action/action-templates/view/${id ? id : result.insertId}`,
      val,
      "url",
      req.user.sessionid
    );
  }
  return sendResponse(res, 200, `Record ${status} successfully`);
};

function sortTasks(tasks) {
  const graph = new Map();
  const reverseGraph = new Map();

  // Build the graph and reverse graph
  tasks.forEach((task) => {
    graph.set(task.id, new Set());
    reverseGraph.set(task.id, new Set());
  });

  tasks.forEach((task) => {
    const successors = Array.isArray(task.successor_task_template)
      ? task.successor_task_template
      : [task.successor_task_template].filter(Boolean);
    const predecessors = Array.isArray(task.predecessor_task_template)
      ? task.predecessor_task_template
      : [task.predecessor_task_template].filter(Boolean);

    successors.forEach((succId) => {
      if (graph.has(succId)) {
        graph.get(task.id).add(succId);
        reverseGraph.get(succId).add(task.id);
      }
    });

    predecessors.forEach((predId) => {
      if (graph.has(predId)) {
        graph.get(predId).add(task.id);
        reverseGraph.get(task.id).add(predId);
      }
    });
  });

  function dfs(node, visited, stack, graph) {
    visited.add(node);
    for (let neighbor of graph.get(node)) {
      if (!visited.has(neighbor)) {
        dfs(neighbor, visited, stack, graph);
      }
    }
    stack.push(node);
  }

  function topologicalSort(graph) {
    const visited = new Set();
    const stack = [];
    for (let node of graph.keys()) {
      if (!visited.has(node)) {
        dfs(node, visited, stack, graph);
      }
    }
    return stack.reverse();
  }

  // Perform topological sort on both graphs
  const forwardOrder = topologicalSort(graph);
  const backwardOrder = topologicalSort(reverseGraph);

  // Combine both orders
  const scoreMap = new Map();
  forwardOrder.forEach((id, index) => {
    scoreMap.set(id, index);
  });
  backwardOrder.forEach((id, index) => {
    scoreMap.set(id, (scoreMap.get(id) || 0) + index);
  });

  // Final sorting based on combined scores
  const sortedIds = [...scoreMap.entries()]
    .sort((a, b) => a[1] - b[1])
    .map((entry) => entry[0]);

  // Map the sorted IDs back to the original task objects
  const taskMap = new Map(tasks.map((task) => [task.id, task]));
  return sortedIds.map((id) => taskMap.get(id));
}

export const getActionTemplateList = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "action_template_list",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = [
    "action_template_list.name",
    "action_template_list.visibility",
    "action_template_list.category",
    "action_template_list.description",
    // "action_template_list.subcategory",
    // "action_template_list.attachment",
    "action_template_list.dependencies",
    "action_template_list.task_type",
    "action_template_list.module_name",
    "action_template_list.sub_module_name",
    "action_template_list.record_name",
    "action_template_list.task_title",
    "action_template_list.task_description",
    "action_template_list.priority",
    "action_template_list.task_duration",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const filter = req.query.filter && JSON.parse(req.query.filter);

  let ownerCondition = "";
  if (filter?.for_approval == 1) {
    ownerCondition = req.user.isSuperAdmin
      ? ""
      : `AND JSON_CONTAINS(action_template_list.send_to_employee, ${req.user.sessionid} )`;
  }

  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "action_template_list.created_by = createdUser.id",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "action_template_list.organization = organization.id",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = action_template_list.updated_by",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "action_template_list.department = department.id",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = action_template_list.assigned_to",
    },
    {
      type: "left",
      targetTable: "category AS categoryTable",
      onCondition: "categoryTable.id = action_template_list.category",
    },
    {
      type: "left",
      targetTable: "category AS subcategoryTable",
      onCondition: "subcategoryTable.id = action_template_list.sub_category",
    },
    {
      type: "left",
      targetTable: "sidebar AS moduleTable",
      onCondition: "moduleTable.id = action_template_list.module_name",
    },
    {
      type: "left",
      targetTable: "sidebar AS subModuleTable",
      onCondition: "subModuleTable.id = action_template_list.sub_module_name",
    },
    {
      type: "left",
      targetTable: "roles",
      onCondition: "roles.id = action_template_list.role",
    },
    {
      type: "left",
      targetTable: "action_template_list as stt",
      onCondition: "stt.id = action_template_list.successor_task_template",
    },
    {
      type: "left",
      targetTable: "action_template_list as ptt",
      onCondition: "ptt.id = action_template_list.predecessor_task_template",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = action_template_list.ddrm_id",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const actionTemplateQuery = `SELECT action_template_list.* , createdUser.name AS created_by_name , createdUser.surname AS created_by_surname, createdUser.profile AS created_by_profile , organization.name AS organization_name , department.name AS department_name , CONCAT(u1.name , ' ' , u1.name) AS assigned_to_name , u1.profile AS assigned_to_profile , CONCAT(u2.name , ' ' ,u2.surname ) AS updated_by_name , u2.profile AS updated_by_profile , categoryTable.name AS category_name , subcategoryTable.name AS sub_category_name , moduleTable.title AS module , subModuleTable.title AS sub_module , roles.name AS role_name , stt.task_title AS successor_task_template_name , ptt.task_title AS predecessor_task_template_name, repository.url as attachment FROM action_template_list ${joinCondition} WHERE action_template_list.deleted = 0 ${searchCondition} ${ownerCondition} ${condition}`;
  let [actionTemplate] = await db.query(actionTemplateQuery);
  for (let act of actionTemplate) {
    act.updated_at = act.updated_by != null ? act.updated_at : null;
    let allSkills = [];
    if (act.skill) {
      [allSkills] = await db.query(
        `SELECT * FROM skills WHERE deleted = 0 AND id IN (${JSON.parse(
          act?.skill
        )})`
      );
    }
    act.skill_name = allSkills;
  }
  let levels = await orderTasks(actionTemplate);

  actionTemplate = await decodeAndParseFields(actionTemplate);
  const totalRecord = await countQueryCondition(actionTemplateQuery);
  return res.status(200).json({
    status: true,
    data: actionTemplate,
    total: totalRecord,
    ordering: levels,
  });
};

export const deleteActionTemplateList = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(ActionTemplateList, id);
  await insertActivityLog(req.user.sessionid, "delete", "Action Template", id);
  return sendResponse(res, 200, "Record deleted successfully");
};

export const approveRejectTemplateList = async (req, res) => {
  const { id } = req.body;
  req.body.action_template_id = id;
  // req.body.id = id;
  req.body.updated_by = req.user.sessionid;
  const { query, values } = updateQueryBuilder(ActionTemplateList, req.body);
  delete req.body.id;
  if ((req.body.status = "Rejected")) {
    const { query: query2, values: values2 } = createQueryBuilder(
      ActionTemplateComment,
      req.body
    );
    await db.query(query2, values2);
  }
  await db.query(query, values);
  return sendResponse(res, 200, "Record updated successfully");
};
