import db from "../db-config.js";
import {
  countQueryCondition,
  decodeSingle_statement,
  deleteSettingRecord,
  encodeSingle_statement,
  insertActivityLog,
  makeJoins,
  whereCondition,
  settingsUpdater,
  createQueryBuilder,
  searchConditionRecord,
  uniqueIdGenerator,
} from "../helper/general.js";
import AuditCategory from "../sequelize/AuditCategorySchema.js";
import { sendResponse } from "../helper/wrapper.js";

/**Function to create and update Audit Category */
export const createUpdatedAuditCategory = async (req, res) => {
  const { id, name, description, organization } = req.body;
  const encodedDescription = await encodeSingle_statement(description);
  req.body.description = encodedDescription;

  /**If id comes in body then it will update the query */
  if (id) {
    if (organization.length === 0) {
      return sendResponse(
        res,
        400,
        "Please Select at least one business Structure"
      );
    }
    const result = await settingsUpdater(
      AuditCategory,
      organization,
      req.body,
      req.user
    );

    if (!result) {
      return sendResponse(res, 404, "No data found");
    }

    /**Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "update",
      "Audit Category",
      `This user Update Audit Category which id is ${id}`
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const insertPromises = organization.map(async (element) => {
      /** Check if this organization already has the particular name */
      const [checkNameWithOrganization] = await db.query(
        `SELECT name FROM audit_category WHERE name = ? AND deleted = '0' AND organization = ?`,
        [name, element]
      );

      /** If record not found for that name with particular organization, insert the record */
      if (checkNameWithOrganization.length === 0) {
        const unique_id = await uniqueIdGenerator(
          element,
          req?.body?.department,
          "AuditCategory",
          "audit_category",
          "unique_id",
          "unique_id"
        );
        req.body.unique_id = unique_id;

        req.body.created_by = req.user.sessionid;
        req.body.organization = element;
        /** Insert Audit Category details */
        const { query, values } = createQueryBuilder(AuditCategory, {
          ...req.body,
          organization: element,
        });
        await db.query(query, values);
        /** Insert record for activity log */
        insertActivityLog(
          req.user.sessionid,
          "create",
          "Audit Category",
          `This user created a new Audit Category named ${name} for organization ${element}`
        );
      }
    });

    await Promise.all(insertPromises);

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all Audit Category */
export const viewAllAuditCategory = async (req, res) => {
  const condition = await whereCondition({
    table: "audit_category",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id: req.params.id,
    grouped: req.query.grouped,
    user: req.user,
  });

  /** If value come with any search condition then search that word */
  const searchTableName = [
    "audit_category.name",
    "audit_category.description",
    "users.name",
    "organization.name",
  ];
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = audit_category.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = audit_category.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  /**Record of all alert */
  const auditCategoryDataFetchQuery = `SELECT audit_category.unique_id,audit_category.id,audit_category.created_by as created_by_id,audit_category.name,audit_category.description,users.name as created_by,users.surname as created_by_surname ,users.profile as created_by_profile ,organization.name as organization_name,audit_category.organization
      FROM audit_category  
      ${joinsRecord} 
      where audit_category.deleted = 0 ${searchCondition} ${condition}`;

  const [auditCategoryDataFetch] = await db.query(auditCategoryDataFetchQuery);

  for (const item of auditCategoryDataFetch) {
    item.description = await decodeSingle_statement(item.description);
    if (req.query.grouped == "true") {
      const [organizations] = await db.query(
        `SELECT organization FROM audit_category WHERE deleted = 0 AND name = ?`,
        [item.name]
      );
      const arr = organizations.map((item) => item.organization);
      item.organizations = arr;
    }
  }

  /**Count all Audit Category */
  const totalRecord = await countQueryCondition(auditCategoryDataFetchQuery);

  return sendResponse(res, 200, auditCategoryDataFetch, totalRecord);
};

/**Function to delete a specific Audit Category */
export const deleteAuditCategory = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteSettingRecord("audit_category", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(req.user.sessionid, "delete", "Audit Category", id);
    return sendResponse(res, 200, "Record deleted successfully");
  } else {
    return sendResponse(res, 404, "No data found");
  }
};
