import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  createQueryBuilder,
  updateQueryBuilder,
  whereCondition,
  makeJoins,
  countQueryCondition,
  searchConditionRecord,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  decodeAndParseFields,
  encodeAndStringifyFields,
  insertActivityLog,
  getUserListByIds,
  uploadFile,
  processUploadDocuments,
  getRecord,
} from "../helper/general.js";
import BcpTesting from "../sequelize/BcpTestingSchema.js";
import {
  createUpdateActionFunction,
  getCustomActionFunction,
} from "./customActionCreationController.js";
import BcpTestingDocs from "../sequelize/BcpTestingDocsSchema.js";

export const createUpdateBcpTesting = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];
  const {
    id,
    department,
    reviewer = [],
    team_members = [],
    task_data,
    action_taken,
    sidebar_id = 213,
  } = req.body;

  req.body.reviewer = reviewer?.map((e) => e?.user_id);
  let organizationId = req.body.organization;
  req.body.team_members = team_members?.map((e) => e?.user_id);

  const data = {
    task_data,
    department,
    files: req.files,
    user: req.user,
    id: action_taken,
    organization: organizationId,
    sidebar_id,
  };

  if (
    req.body?.task_data &&
    req.body?.task_data[0] &&
    req.body?.task_data[0]?.role
  ) {
    const { status, arr } = await createUpdateActionFunction(data, req);
    if (status) {
      const action_taken = arr[0].id;
      req.body.action_taken = action_taken;
    }
  }

  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  // Image uploading section
  const documents = await processUploadDocuments(req, sidebar_id);

  if (id && documents.length) {
    const deleteQuery = `UPDATE bcp_testing_docs SET deleted = 1 where deleted = 0 AND bcp_testing_id = ?`;
    await db.query(deleteQuery, [id]);
  }

  const allFiles = [];
  documents.forEach((doc) => {
    allFiles.push(doc.filepath);
  });

  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }

  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id
    ? updateQueryBuilder(BcpTesting, req.body)
    : createQueryBuilder(BcpTesting, req.body);
  const [result] = await db.query(query, values);

  if (documents || Array.isArray(documents)) {
    for (const file of documents) {
      const title = file.title; // Use file name as title
      const ddrm_id = file?.ddrm_id;
      const id_doc = file.id;
      // Move the file to the public folder
      // Save file details for database
      const { query: docQuery, values: docValues } = createQueryBuilder(
        BcpTestingDocs,
        {
          doc_no: id_doc,
          bcp_testing_id: id ? id : result.insertId,
          title,
          ddrm_id,
        }
      );

      await db.query(docQuery, docValues);
    }
  }
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Bcp Testing",
    `This user ${status} a new Bcp Testing Record`
  );
  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getBcpTesting = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "bcp_testing",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = ["bcp_testing.plan_name", "bcp_testing.method"];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = bcp_testing.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = bcp_testing.organization",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = bcp_testing.location",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = bcp_testing.department",
    },
    {
      type: "left",
      targetTable: "bcp_management",
      onCondition: "bcp_management.id = bcp_testing.bcp_management_id",
    },
    {
      type: "left",
      targetTable: "users as test_coordinator",
      onCondition: "test_coordinator.id = bcp_testing.test_coordinator",
    },
    {
      type: "left",
      targetTable: "roles as test_coordinator_role",
      onCondition: "test_coordinator_role.id = test_coordinator.role",
    },
    {
      type: "left",
      targetTable: "department as test_coordinator_department",
      onCondition:
        "test_coordinator_department.id = test_coordinator.department",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const statusCondition = `AND bcp_management.status = 'approved'`;

  const bcpTesting = `SELECT bcp_testing.*,CONCAT(test_coordinator.name , ' ' , test_coordinator.surname) as test_coordinator_name , test_coordinator.profile as test_coordinator_profile, test_coordinator_role.name as test_coordinator_role , test_coordinator_department.name as test_coordinator_department, organization.name as organization_name , department.name as department_name , users.name as created_by_name , users.surname AS created_by_surname , users.profile AS created_by_profile, bcp_management.status as bcp_management_status,bcp_management.plan_name as bcp_management_plan_name, location.name AS location_name FROM bcp_testing ${joinCondition} WHERE bcp_testing.deleted = 0 ${searchCondition} ${statusCondition} ${condition}`;

  let [BcpTesting] = await db.query(bcpTesting);
  BcpTesting = await decodeAndParseFields(BcpTesting);

  for (let i = 0; i < BcpTesting.length; i++) {
    const id = BcpTesting[i].id;

    BcpTesting[i].reviewer = await getUserListByIds(BcpTesting[i].reviewer);
    BcpTesting[i].team_members = await getUserListByIds(
      BcpTesting[i].team_members
    );

    const [files] = await db.query(
      `SELECT bcp_testing_docs.*, repository.url as file, document_name as title FROM bcp_testing_docs LEFT JOIN repository ON repository.id = bcp_testing_docs.ddrm_id LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE bcp_testing_docs.deleted = 0 AND bcp_testing_id = ${id}`
    );

    BcpTesting[i].upload_documents = files;

    // action taken
    if (BcpTesting[i]?.action_taken) {
      const data = {
        params: { id: BcpTesting[i]?.action_taken },
        user: req.user,
        query: {},
      };
      const customActionData = await getCustomActionFunction(data);
      BcpTesting[i].task_data = customActionData;
    }
  }

  const totalRecord = await countQueryCondition(bcpTesting);
  return sendResponse(res, 200, BcpTesting, totalRecord);
};

export const deleteBcpTesting = async (req, res) => {
  const { id } = req.params;
  const [record] = await getRecord("bcp_testing", "id", id);
  if (record?.action_taken) {
    await deleteRecord("custom_action_creation", record?.action_taken);
  }
  await deleteRecord("bcp_testing", id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "bcp_testing",
    `This user deleted a new bcp_testing Record for organization ${id}`
  );
  return sendResponse(res, 200, "Record deleted successfully");
};
