import db from "../db-config.js";
  import { sendResponse } from "../helper/wrapper.js";
  import { 
    insertActivityLog, 
    searchConditionRecord, 
    makeJoins, 
    whereCondition, 
    countQueryCondition, 
    settingsUpdater,
    deleteSettingRecord, 
    encodeSingle_statement, 
    decodeSingle_statement, 
    uniqueIdGenerator,
  } from "../helper/general.js";
  import CommandCenterLocation from "../sequelize/CommandCenterLocationSchema.js";

  /** Function to create or update CommandCenterLocation */
  export const createUpdateCommandCenterLocation = async (req, res) => {
    const { id, name, description, department } = req.body;
    const encodedDescription = await encodeSingle_statement(description);

    if (id) {
      const result = await settingsUpdater(
        CommandCenterLocation,
        req.body.organization,
        req.body,
        req.user
      );
      if (!result) {
        return sendResponse(res, 404, "No data found");
      }
      await insertActivityLog(req.user.sessionid, "update", "CommandCenterLocation", `This user updated CommandCenterLocation with id: ${id}`);
      return sendResponse(res, 200, "Record updated successfully");
    } else {
      if(!id) {
        const unique_id = await uniqueIdGenerator(
          req.body.organization,
          department,
          "LCC",
          "command_center_location",
          "unique_id",
          "unique_id"
        );
        req.body.unique_id = unique_id;
      }    
      const insertPromises = req.body.organization.map(async (org) => {
        const [checkName] = await db.query(`SELECT name FROM command_center_location WHERE name = ? AND deleted = 0 AND organization = ?`, [name, org]);
        if (checkName.length === 0) {
          await db.query(`
            INSERT INTO command_center_location (name, description, created_by, organization, unique_id) VALUES (?, ?, ?, ?, ?)`, 
            [name, encodedDescription, req.user.sessionid, org, req.body.unique_id]);
          await insertActivityLog(req.user.sessionid, "create", "CommandCenterLocation", `This user created a new CommandCenterLocation '${name}' for organization ${org}`);
        }
      });
      await Promise.all(insertPromises);
      return sendResponse(res, 200, "Record created successfully");
    }
  };

  /** Function to view all or single CommandCenterLocation */
  export const viewCommandCenterLocation = async (req, res) => {
    const { id } = req.params;
    const condition = await whereCondition({
      table: "command_center_location",
      page: req.query.page,
      all: req.query.all,
      pageSize: req.query.pageSize,
      filter: req.query.filter,
      id,
      user: req.user,
      grouped: req.query.grouped,
    });

    const searchFields = [
      "command_center_location.name",
      "command_center_location.description",
      "users.name",
      "organization.name"
    ];
    let searchCondition = await searchConditionRecord(req.query.search, searchFields);
    const joins = [
      { type: "left", targetTable: "users", onCondition: "users.id = command_center_location.created_by" },
      { type: "left", targetTable: "organization", onCondition: "organization.id = command_center_location.organization" }
    ];
    const joinsRecord = await makeJoins(joins);

    const fetchQuery = `SELECT command_center_location.*, CONCAT(users.name, ' ', users.surname) AS created_by, organization.name AS organization_name FROM command_center_location ${joinsRecord} WHERE command_center_location.deleted = 0 ${searchCondition} ${condition}`;
    let [records] = await db.query(fetchQuery);

    for (const record of records) {
      record.description = await decodeSingle_statement(record.description);
       if (req.query.grouped == "true") {
        const [organizations] = await db.query(
          `SELECT organization FROM command_center_location WHERE deleted = 0 AND name = ?`,
          [record.name]
        );
        const arr = organizations.map((item) => item.organization);
        record.organizations = arr;
      }
    }

    const totalRecord = await countQueryCondition(fetchQuery);
    return sendResponse(res, 200, records, totalRecord);
  };

  /** Function to delete CommandCenterLocation */
  export const deleteCommandCenterLocation = async (req, res) => {
    const { id } = req.params;
    const deleteRecord = await deleteSettingRecord("command_center_location", id);
    if (deleteRecord) {
      await insertActivityLog(req.user.sessionid, "delete", "CommandCenterLocation", id);
      return sendResponse(res, 200, "Record deleted successfully");
    } else {
      return sendResponse(res, 404, "Record not found");
    }
  };