import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  getRecord,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import ContractorTrainingSchedule from "../sequelize/ContractorTrainingScheduleSchema.js";

import fs from "fs/promises";
import path from "path";
import ical from "ical-generator";
import sendEmail from "../helper/sendEmail.js";
import ContractorTrainingAttendance from "../sequelize/ContractorTrainingAttendanceSchema.js";
import chalk from "chalk";

export const createUpdateContractorTrainingSchedule = async (req, res) => {
  let { schedule_training, id, department, projects } = req.body;
  if (!projects) return sendResponse(res, 400, "Project is required");
  projects = typeof projects == "string" ? JSON.parse(projects) : projects;
  if (department) {
    req.body.organization = (await getOrganizationAccordingToDepartment(department))[0].organization;
  }
  let status = id ? "Updated" : "Created";
  for (let training of schedule_training) {
    training[id ? "updated_by" : "created_by"] = req.user.sessionid;
    const [trainingProgramRecord] = await getRecord("training_need_analysis", "id", training?.training_program);
    training.projects = projects;
    training.contractors = req.body.contractors;
    training.department = trainingProgramRecord?.department;
    training.organization = req.body.organization;
    training.department = req.body.department;

    training = await encodeAndStringifyFields(training);

    if (!id) {
      const unique_id = await uniqueIdGenerator(
        training.organization,
        training.department,
        "ContractorTrainingSchedule",
        "contractor_training_schedule",
        "unique_id",
        "unique_id"
      );
      training.unique_id = unique_id;
    }

    const { query, values } = id
      ? updateQueryBuilder(ContractorTrainingSchedule, { ...training, id })
      : createQueryBuilder(ContractorTrainingSchedule, training);
    const [result] = await db.query(query, values);

    if (!id) {
      const unique_id = await uniqueIdGenerator(
        req.body.organization,
        req.body.department,
        "ContractorTrainingAttendance",
        "contractor_attendance_recording",
        "unique_id",
        "unique_id"
      );

      const { query, values } = createQueryBuilder(ContractorTrainingAttendance, {
        unique_id: unique_id,
        organization: req.body.organization,
        department: req.body.department,
        training_program: result.insertId,
        contractors: req.body.contractors,
        projects: req.body.projects,
        status: "Pending",
      });
      await db.query(query, values);
    }
    for (const schedule of schedule_training) {
      const [fetchName] = await db.query(`SELECT * FROM training_need_analysis WHERE id = ?`, [
        schedule.training_program,
      ]);
      // console.log("TrainingProgram", schedule.training_program);
      // console.log(fetchName);
      // console.log(projects, "projects");
      for (let project of projects) {
        [project] = await db.query(`SELECT project_members  FROM project_registration WHERE id = ?`, [project]);
        let training_start_date = schedule.single_day ? schedule.single_day : schedule.multiple_day;
        let training_name = fetchName[0]?.training_awareness;
        let members = project[0]?.project_members;
        members = typeof members == "string" ? JSON.parse(members) : members;
        // console.log(members, "members");
        if (!members) continue;
        for (const member of members) {
          const mail = member.email;
          const name = member.name + " " + member.surname;
          // console.log(training_name, "trainingName");
          // console.log(training_start_date, "trainingStartDate");
          const sendRecordArray = {
            templateFileUrl: "training_schedule_template.html",
            training_name,
            training_start_date,
            name,
            templateName: "Training Schedule",
          };
          const info = await sendEmail("info@harmonyandhelp.com", mail, "Training Reminder", sendRecordArray);
          if (info) {
            console.log("Mail send successfully");
          }
        }
        let members2 = project[0]?.project_members2;
        members2 = typeof members2 == "string" ? JSON.parse(members2) : members2;
        // console.log(members2, "members2");
        if (!members2) continue;
        for (const member of members2) {
          const mail = member.email;
          const name = member.name + " " + member.surname;
          // console.log(training_name, "trainingName");
          // console.log(training_start_date, "trainingStartDate");
          const sendRecordArray = {
            templateFileUrl: "training_schedule_template.html",
            training_name,
            training_start_date,
            name,
            templateName: "Training Schedule",
          };
          const info = await sendEmail("info@harmonyandhelp.com", mail, "Training Reminder", sendRecordArray);
          if (info) {
            console.log("Mail send successfully");
          }
        }
      }
    }

    await insertActivityLog(req.user.sessionid, status, "Training Schedule", id ? id : result.insertId);
  }

  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getContractorTrainingSchedule = async (req, res) => {
  const { id } = req.params;
  const { training_type } = req.query;
  const trainingFilter = training_type ? `AND training_need_analysis.training_type = '${training_type}'` : "";
  const condition = await whereCondition({
    table: "contractor_training_schedule",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  const searchTableName = [
    "employeeUser.name",
    "employeeUser.surname",
    "employeeUser.employee_id",
    "employeeUser.gender",
    "employeeUser.disability",
    "employeeUser.race",
    "employeeUser.employee_type",
    "roles.name",
    "contractor_training_schedule.financial_year",
    "contractor_training_schedule.training_provider",
    "contractor_training_schedule.external_provider",
    "CONCAT(createdUser.name , ' ' , createdUser.surname)",
  ];

  const searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joinsRecord = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = contractor_training_schedule.created_by",
    },
    {
      type: "left",
      targetTable: "users as internal_employee",
      onCondition: "internal_employee.id = contractor_training_schedule.internal_emp_name",
    },
    {
      type: "left",
      targetTable: "department as internal_dep",
      onCondition: "internal_dep.id = internal_employee.department",
    },
    {
      type: "left",
      targetTable: "roles as internal_role",
      onCondition: "internal_role.id = internal_employee.role",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = contractor_training_schedule.organization",
    },
    {
      type: "left",
      targetTable: "department AS d",
      onCondition: "d.id = contractor_training_schedule.provider_department",
    },
    {
      type: "left",
      targetTable: "training_need_analysis",
      onCondition: "training_need_analysis.id = contractor_training_schedule.training_program",
    },
    {
      type: "left",
      targetTable: "purpose_of_training",
      onCondition: "purpose_of_training.id = training_need_analysis.training_purpose",
    },
    {
      type: "left",
      targetTable: "financial_year AS fy",
      onCondition: "fy.id = contractor_training_schedule.financial_year",
    },
  ];

  const joinCondition = await makeJoins(joinsRecord);

  const viewContractorTrainingScheduleQuery = `SELECT contractor_training_schedule.* , CONCAT(internal_employee.name , ' ' , internal_employee.surname) as internal_emp,internal_dep.name as internal_dep_name, internal_role.name as internal_role_name, createdUser.name as created_by_name, createdUser.surname as created_by_surname, createdUser.profile as created_by_profile , d.name AS provider_department_name , training_need_analysis.training_type , training_need_analysis.training_purpose , 
  purpose_of_training.name as training_purpose_name , organization.name AS organization_name , fy.name AS financial_year_name , training_need_analysis.training_name AS training_program_name FROM contractor_training_schedule   ${joinCondition} WHERE contractor_training_schedule.deleted = 0 ${trainingFilter} ${searchCondition} ${condition}`;

  let [viewContractorTrainingSchedule] = await db.query(viewContractorTrainingScheduleQuery);
  viewContractorTrainingSchedule = await decodeAndParseFields(viewContractorTrainingSchedule);
  for (const schedule of viewContractorTrainingSchedule) {
    const contractors = schedule.contractors;
    const projects = schedule.projects;
    schedule.contractor_details = [];
    schedule.project_details = [];
    if (contractors) {
      for (const contractor of contractors) {
        const [contractorDetails] = await db.query(`SELECT * FROM contractor_registration WHERE id = ${contractor}`);
        schedule.contractor_details.push({
          contractor_name: contractorDetails[0]?.contractor_name,
          contractor_cipc_registration_number: contractorDetails[0]?.cipc_registration_number,
        });
      }
    }
    if (projects) {
      for (const project of projects) {
        const [projectDetails] = await db.query(`SELECT * FROM project_registration WHERE id = ${project}`);
        schedule.project_details.push({
          contractor_name: projectDetails[0]?.project_title,
        });
      }
    }
    const employees = schedule.employees;
    const provider_department = schedule.provider_department;
    schedule.provider_department_name = "";
    if (provider_department) {
      if (schedule.training_provider === "Internal") {
        const [departmentRecord] = await getRecord("department", "id", provider_department);
        schedule.provider_department_name = departmentRecord?.name || "";
      } else if (schedule.training_provider === "External") {
        schedule.provider_department_name = Number(schedule.provider_department) ? "" : schedule.provider_department;
      }
    }
    let arr = [];
    for (let i = 0; i < employees.length; i++) {
      const employee = employees[i];
      // console.log(chalk.blue(JSON.stringify(employee)));
      // console.log(chalk.gray(JSON.stringify(employee?.user_id)));
      // console.log(chalk.gray(typeof(employee)));

      let empId = typeof employee === "object" ? employee?.user_id : employee;

      const [empDetails] = await db.query(
        `SELECT contractor_registration.*, organization.name AS organization_name FROM contractor_registration LEFT JOIN organization ON organization.id = contractor_registration.organization WHERE contractor_registration.id = ${empId}`
      );
      // console.log(empDetails, "empDetails");
      let obj = {};
      obj.user_id = empDetails[0]?.id;
      obj.name = empDetails[0]?.contractor_name;
      // obj.profile = empDetails[0]?.profile;
      // obj.surname = empDetails[0]?.surname;
      // obj.employee_id = empDetails[0]?.unique_id;
      // obj.gender = empDetails[0]?.gender;
      // obj.disability = empDetails[0]?.disability;
      // obj.race = empDetails[0]?.race;
      // obj.employee_type = empDetails[0]?.employee_type;
      obj.training_purpose = schedule.training_purpose;
      obj.training_type = schedule.training_type;
      obj.training_provider = schedule.training_provider;
      obj.training_program_name = schedule.training_program_name;
      obj.training_date = schedule.single_day ? schedule.single_day : schedule.multiple_day;
      obj.organization = empDetails[0]?.organization_name;
      arr.push(obj);
    }
    schedule.contractor_training_schedule = arr;
  }

  const totalRecord = await countQueryCondition(viewContractorTrainingScheduleQuery);

  return sendResponse(res, 200, viewContractorTrainingSchedule, totalRecord);
};
export const deleteContractorTrainingSchedule = async (req, res) => {
  const { id } = req.params;
  await deleteRecord("contractor_training_schedule", id);
  await insertActivityLog(
    req.user.sessionid,
    "Deleted",
    "Training Schedule",
    `This user deleted a Contractor Training Schedule Record with id ${id}`
  );
  return sendResponse(res, 200, "Record Deleted Successfully");
};

export const fetchContractorTrainingScheduleData = async (req, res) => {
  const [trainingSchedule] = await db.query(`SELECT * FROM contractor_training_schedule WHERE deleted = 0`);

  // Create a new calendar
  const calendar = ical({
    name: "Meeting Calendar",
    prodId: { company: "Harmony and Help", product: "SSHEQ" },
    timezone: "Asia/Kolkata",
    url: "https://ssheq.thewingshield.com/",
  });

  trainingSchedule.forEach((schedule) => {
    let eventOptions = {
      id: schedule.id,
      organizer: { name: "Harmony & Help", email: "info@harmonyandhelp.com" },
    };

    if (schedule.schedule_type === 0) {
      eventOptions.start = new Date(`${schedule.single_day}T${schedule.single_start_time}`);
      eventOptions.end = new Date(`${schedule.single_day}T${schedule.single_end_time}`);
    } else if (schedule.schedule_type === 1) {
      eventOptions.start = new Date(`${schedule.multiple_day}T${schedule.multiple_start_time}`);
      eventOptions.end = new Date(`${schedule.multiple_end_day}T${schedule.multiple_end_time}`);
    }

    const event = calendar.createEvent(eventOptions);

    // If the event is recurring, set the recurrence rule
    if (schedule?.repeat_event) {
      const rrule = {
        freq: schedule?.repeat_every_cycle?.toUpperCase(), // FREQ
        interval: schedule.repeat_every, // INTERVAL
      };

      // If weekly, add by day
      if (schedule.repeat_every_cycle.toLowerCase() === "weekly" && schedule.repeat_every_dwy) {
        rrule.byDay = schedule.repeat_every_dwy.split(",").map((day) => day.toUpperCase());
      }

      event.repeating(rrule);
    }
  });

  // Ensure the directory exists
  const calendarDir = path.join(process.cwd(), "public", "calendar");
  await fs.mkdir(calendarDir, { recursive: true });

  // Save the calendar to an .ics file
  const icsContent = calendar.toString();
  const writeFilePath = path.join(calendarDir, "contractor-training-schedule-data.ics");

  await fs.writeFile(writeFilePath, icsContent);

  res.status(200).json({
    status: true,
    message: "Contractor Training Schedule details saved to .ICS file",
    data: {
      filePath: "public/calendar/contractor-training-schedule-data.ics",
    },
  });
};
