import {
  decodeAndParseFields,
  decodeSingle_statement,
  getColorForCategory,
  getDateRangeForSQL,
  getOrganizationAccordingToDepartment,
  incidentCategoryConstant,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import db from "../db-config.js";
import moment from "moment";
import * as cheerio from "cheerio";

export const riskManagementDashboardData = async (req, res) => {
  const { all, filter } = req.query;

  if (!filter) {
    return sendResponse(res, 404, "Filter not found");
  }

  req.query.filter = JSON.parse(filter);
  // req.query.organization = req.query.filter?.organization || null;
  // if (!req.query.organization) {
  //   return sendResponse(res, 404, "Organization not found");
  // }

  // req.query.department = req.query.filter?.department || null;

  // Parse date range
  const dateRange = req.query.filter?.date_type || "All-Time";
  const start_date = req.query.filter?.start_date;
  const end_date = req.query.filter?.end_date;
  const dateFilter = getDateRangeForSQL(dateRange, start_date, end_date);

  /**Check record if organization is not coming then fetch organization according to department */
  // let organizationId = req.query.organization;
  // if (req.query.department) {
  //   const recordAccordingToOrganization =
  //     await getOrganizationAccordingToDepartment(req.query.department);
  //   organizationId = recordAccordingToOrganization[0].organization;
  // }

  // NEW DASHBOARD DATA
  const risksByBusinessChartData = await riskByBusinesses(dateFilter);
  const topTenRisksChartData = await topTenRisks(dateFilter);
  const risksByIssueTypeChartData = await risksByAllIssueType(dateFilter);
  const risksByRiskTypeChartData = await riskByRiskType(dateFilter);
  const risksByCategoryChartData = await riskByCategory(dateFilter);
  const risksByResidualRatingChartData = await riskByResidualRating(dateFilter);
  const risksPerWorkAreaChartData = await riskPerWorkArea(dateFilter);
  const risksPerMainProcessChartData = await riskPerMainProcess(dateFilter);
  const risksByReviewDateChartData = await riskByReviewDate(dateFilter);
  const risksByOverdueReviewDateChartData = await riskByOverdueReviewDate(
    dateFilter
  );

  const final_data = {
    // NEW DASHBOARD DATA
    risksByBusiness: risksByBusinessChartData,
    topTenRisks: topTenRisksChartData,
    risksByIssueType: risksByIssueTypeChartData,
    risksByRiskType: risksByRiskTypeChartData,
    risksByCategory: risksByCategoryChartData,
    risksByResidualRating: risksByResidualRatingChartData,
    risksPerWorkAres: risksPerWorkAreaChartData,
    risksPerMainProcess: risksPerMainProcessChartData,
    risksByReviewDate: risksByReviewDateChartData,
    risksByOverdueReviewDate: risksByOverdueReviewDateChartData,
  };
  return sendResponse(res, 200, final_data);
};

export const policyStatusAnalysisReportData = async (req, res) => {
  const { all, filter } = req.query;

  try {
    if (!filter) {
      return sendResponse(res, 404, "Filter not found");
    }

    req.query.filter = JSON.parse(filter);
    req.query.organization = req.query.filter?.organization || null;
    if (!req.query.organization) {
      return sendResponse(res, 404, "Organization not found");
    }

    req.query.department = req.query.filter?.department || null;

    /**Check record if organization is not coming then fetch organization according to department */
    let organizationId = req.query.organization;
    if (req.query.department) {
      const recordAccordingToOrganization =
        await getOrganizationAccordingToDepartment(req.query.department);
      organizationId = recordAccordingToOrganization[0].organization;
    }

    // Parse date range
    const dateRange = req.query.filter?.date_type || "All-Time";
    const start_date = req.query.filter?.start_date;
    const end_date = req.query.filter?.end_date;
    const dateFilter = getDateRangeForSQL(dateRange, start_date, end_date);

    const policy_status = [
      "pending",
      "reviewing",
      "approval",
      "approved",
      "rejected",
    ];

    // Function to get policy status data
    const getPolicyStatusQuery = (organizationId, dateFilter) => `
      SELECT 
      COUNT(id) AS count, 
      current_status
    FROM policy 
    WHERE organization = '${organizationId}' 
      AND ${dateFilter.getSQLWhereDateClause("created_at")} 
      AND deleted = 0
      GROUP BY policy.current_status
    `;

    const transformCombinedPolicyData = async (organizationId, dateFilter) => {
      try {
        const query = getPolicyStatusQuery(organizationId, dateFilter);
        const [results] = await db.query(query);

        // Calculate total count first
        const total = results.reduce(
          (sum, row) => sum + parseInt(row.count),
          0
        );

        // Create a map of status to count
        const statusCountMap = results.reduce((acc, row) => {
          acc[row.current_status] = parseInt(row.count);
          return acc;
        }, {});

        // Generate percentage array in specified order
        const data = policy_status.map((status) => {
          const count = statusCountMap[status] || 0;
          return total > 0 ? parseFloat(((count / total) * 100).toFixed(2)) : 0;
        });

        const labels = policy_status.map((item) => item.toUpperCase());
        return {
          data: data,
          labels: labels,
        };
      } catch (error) {
        console.error("Error transforming policy status data:", error);
        throw error;
      }
    };

    const formattedData = await transformCombinedPolicyData(
      organizationId,
      dateFilter
    );
    const policy_communication_data =
      await policyCommunicationStatusAnalysisReportData(
        organizationId,
        dateFilter
      );
    policy_communication_data.push({
      keys: ["Not Communicated", "Communicated", "Acknowledged"],
    });

    const final_data = {
      policy_status: formattedData,
      policy_communication_data: policy_communication_data,
    };
    return sendResponse(res, 200, final_data);
  } catch (error) {
    return sendResponse(res, 500, "Failed to retrieve policy status");
  }
};

const policyCommunicationStatusAnalysisReportData = async (
  organizationId,
  dateFilter
) => {
  try {
    if (!organizationId) {
      throw new Error("organizationId is required");
    }

    // Function to get all policies
    const getPolicyQuery = (organizationId, dateFilter) => `
      SELECT 
        policy_title, 
        SUM(CASE WHEN current_status = 'pending' THEN 1 ELSE 0 END) AS not_communicated,
        SUM(CASE WHEN current_status IN ('approval', 'rejected') THEN 1 ELSE 0 END) AS acknowledged,
        SUM(CASE WHEN policy_communicated_count > 0 THEN policy_communicated_count ELSE 0 END) AS communicated
      FROM policy 
      WHERE organization = '${organizationId}'
        AND ${dateFilter.getSQLWhereDateClause("created_at")} 
        AND deleted = 0
      GROUP BY policy_title
    `;

    const transformCombinedPolicyData = async (organizationId, dateFilter) => {
      try {
        const [[policyResults]] = await Promise.all([
          db.query(getPolicyQuery(organizationId, dateFilter)),
        ]);

        // Transform policyResults into the desired format
        const approvalStatus = policyResults.map((policy) => ({
          name: policy.policy_title,
          data: [
            policy.not_communicated,
            policy.communicated,
            policy.acknowledged,
          ],
          // keys: ["Not Communicated", "Communicated", "Acknowledged"],
        }));

        return approvalStatus;
      } catch (error) {
        console.error("Error fetching policies:", error);
        throw error;
      }
    };

    const formattedData = await transformCombinedPolicyData(
      organizationId,
      dateFilter
    );

    return formattedData;
  } catch (error) {
    throw error;
  }
};

/** Function to get Employees stats for Leadership Module  */
export const getLeadershipDashBoardData = async (req, res) => {
  const { all, filter } = req.query;

  if (!filter) {
    return sendResponse(res, 404, "Filter not found");
  }

  req.query.filter = JSON.parse(filter);
  req.query.organization = req.query.filter?.organization || null;
  if (!req.query.organization) {
    return sendResponse(res, 404, "Organization not found");
  }

  req.query.department = req.query.filter?.department || null;

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = req.query.organization;
  if (req.query.department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(req.query.department);
    organizationId = recordAccordingToOrganization[0].organization;
  }

  // Parse date range
  const dateRange = req.query.filter?.date_type || "All-Time";
  const start_date = req.query.filter?.start_date;
  const end_date = req.query.filter?.end_date;
  const dateFilter = getDateRangeForSQL(dateRange, start_date, end_date);

  const countQueryGenerator = (table, column, id, groupByColumn = column) => {
    let query = `SELECT COUNT(${column}) AS result FROM ${table} WHERE ${column} = '${id}'`;
    if (groupByColumn) {
      query += ` GROUP BY ${groupByColumn}`;
    }
    return query;
  };

  // Age Group data
  const ageData = async () => {
    const combinedQuery = `
    SELECT 
      COUNT(id) AS count, 
      date_of_birth,
      age
    FROM users 
    WHERE my_organization = '${organizationId}' 
      AND ${dateFilter.getSQLWhereDateClause("created_at")} 
      AND deleted = '0'
      AND (date_of_birth IS NOT NULL OR age IS NOT NULL)
    GROUP BY users.date_of_birth, users.age
  `;

    const [result] = await db.query(combinedQuery);

    // Initialize age categories and counts
    const ageCategories = ["12-20", "21-30", "30-35", "35-above"];
    const ageCounts = [0, 0, 0, 0]; // Initialize counts for each category

    // Process the result to count users in each age category
    result.forEach((user) => {
      let userAge = user.age;

      // If age is not available, calculate age from date_of_birth
      if (!userAge && user.date_of_birth) {
        // const birthDate = new Date(user.date_of_birth);
        const birthDate = moment(user.date_of_birth);
        // const ageDiff = new Date().getFullYear() - birthDate.getFullYear();
        const ageDiff = moment().diff(birthDate, "years");
        userAge = ageDiff; // Calculate age based on birth year
      }

      // Categorize the age
      if (userAge >= 12 && userAge <= 20) {
        ageCounts[0] += user.count; // 12-20
      } else if (userAge >= 21 && userAge <= 30) {
        ageCounts[1] += user.count; // 21-30
      } else if (userAge > 30 && userAge <= 35) {
        ageCounts[2] += user.count; // 30-35
      } else if (userAge > 35) {
        ageCounts[3] += user.count; // 35-above
      }
    });

    // Prepare the final age object
    const age = {
      name: "Age Group",
      data: ageCategories,
      values: ageCounts,
    };

    return age;
  };

  // Disability data
  const disabilityData = async () => {
    const combinedQuery = `
      SELECT 
        SUM(CASE WHEN disability IS NOT NULL THEN 1 ELSE 0 END) AS yes,
        SUM(CASE WHEN disability IS NULL THEN 1 ELSE 0 END) AS no
      FROM users 
      WHERE my_organization = '${organizationId}' 
        AND ${dateFilter.getSQLWhereDateClause("created_at")} 
        AND deleted = '0'
        AND (disability IS NOT NULL)
    `;

    const [result] = await db.query(combinedQuery);

    // Prepare the disability data object
    const disable = {};
    disable["name"] = "Disability Status";
    disable["data"] = ["Yes", "No"];

    // Ensure the values are in the correct order: [yes, no]
    disable["values"] = [result[0].yes, result[0].no];

    return disable;
  };

  // Race Data
  const raceData = async () => {
    // Step 1: Fetch the race categories from the race table
    const raceCategoriesQuery = `
      SELECT id, name 
      FROM race 
      WHERE organization = '${organizationId}' 
        AND deleted = 0;
    `;

    const [raceCategories] = await db.query(raceCategoriesQuery);

    // If no gender categories found, return zero values
    if (raceCategories.length === 0) {
      return {
        name: "Race",
        data: [],
        values: [],
      };
    }

    // If there are no race categories, return an empty object
    if (raceCategories.length === 0) {
      return {};
    }

    // Step 2: Construct the dynamic SQL query for counting users by race
    const raceCountCases = raceCategories
      .map((race) => {
        return `SUM(CASE WHEN users.race = '${race.id
          }' THEN 1 ELSE 0 END) AS ${race.name.toUpperCase()}`;
      })
      .join(", ");

    const combinedQuery = `
      SELECT 
        ${raceCountCases}
      FROM users
      WHERE my_organization = '${organizationId}' 
        AND ${dateFilter.getSQLWhereDateClause("created_at")} 
        AND deleted = '0'
        AND (race IS NOT NULL AND race > 0)
    `;

    // Step 3: Execute the query
    const [result] = await db.query(combinedQuery);

    // Step 4: Prepare the race data object
    const race = {};
    race["name"] = "Race";
    race["data"] = raceCategories.map((race) => race.name); // Get race names
    race["values"] = raceCategories.map(
      (race) => result[0][race.name.toUpperCase()] || 0
    ); // Get counts, default to 0 if undefined

    return race;
  };

  // Gender Data
  const genderData = async () => {
    // Step 1: Fetch the gender categories from the gender table
    const genderCategoriesQuery = `
      SELECT id, name
      FROM gender
      WHERE organization = '${organizationId}'
        AND deleted = 0;
    `;

    const [genderCategories] = await db.query(genderCategoriesQuery);

    // If no gender categories found, return zero values
    if (genderCategories.length === 0) {
      return {
        name: "Gender",
        data: [],
        values: [],
      };
    }

    // Step 2: Construct the dynamic SQL query for counting users by gender
    const genderCountCases = genderCategories
      .map((gender) => {
        return `SUM(CASE WHEN users.gender = '${gender.id
          }' THEN 1 ELSE 0 END) AS ${gender.name.toUpperCase()}`;
      })
      .join(", ");

    const combinedQuery = `
      SELECT
        ${genderCountCases}
      FROM users
      WHERE my_organization = '${organizationId}'
        AND ${dateFilter.getSQLWhereDateClause("created_at")}
        AND deleted = '0'
        AND (gender IS NOT NULL AND gender > 0)
    `;

    // Step 3: Execute the query
    const [result] = await db.query(combinedQuery);

    // Step 4: Prepare the gender data object
    const gender = {};
    gender["name"] = "Gender";
    gender["data"] = genderCategories.map((gender) => gender.name); // Get gender names
    gender["values"] = genderCategories.map(
      (gender) => result[0][gender.name.toUpperCase()] || 0
    ); // Get counts, default to 0 if undefined

    return gender;
  };

  // Employee Type Data
  const employmentTypeData = async () => {
    // FIRST GET DISTINCT EMPLOYMENT TYPES THAT EXIST IN USER DATA
    const typeQuery = `
        SELECT DISTINCT employee_type
        FROM users
        WHERE my_organization = '${organizationId}'
            AND ${dateFilter.getSQLWhereDateClause("created_at")}
            AND deleted = '0'
            AND employee_type IS NOT NULL
    `;
    const [existingTypes] = await db.query(typeQuery);

    if (existingTypes.length === 0) {
      return { name: "Employment Types", data: [], values: [] };
    }

    // EXTRACT TYPE VALUES FOR FILTERING
    const validTypes = existingTypes
      .map((t) => `'${t.employee_type}'`)
      .join(",");

    // STEP 1: Get total users WITH THESE SPECIFIC TYPES
    const totalUsersQuery = `
        SELECT COUNT(*) AS total_users
        FROM users
        WHERE my_organization = '${organizationId}'
            AND ${dateFilter.getSQLWhereDateClause("created_at")}
            AND deleted = '0'
            AND employee_type IN (${validTypes})
    `;

    // STEP 2: Get counts for these types
    const employmentTypeQuery = `
        SELECT 
            employee_type,
            COUNT(*) AS type_count
        FROM users
        WHERE my_organization = '${organizationId}'
            AND ${dateFilter.getSQLWhereDateClause("created_at")}
            AND deleted = '0'
            AND employee_type IN (${validTypes})
        GROUP BY employee_type
    `;

    // Execute both queries
    const [[totalUsersResult], [typeCounts]] = await Promise.all([
      db.query(totalUsersQuery),
      db.query(employmentTypeQuery),
    ]);

    const totalUsers = totalUsersResult[0].total_users;

    // Create map for quick lookup of counts
    const typeCountMap = new Map();
    typeCounts.forEach((item) => {
      typeCountMap.set(item.employee_type, item.type_count);
    });

    // Prepare data with original type order and proper percentages
    const employmentTypeData = existingTypes.map((type) => {
      const count = typeCountMap.get(type.employee_type) || 0;
      return {
        type: type.employee_type,
        count: count,
        percentage:
          totalUsers > 0
            ? parseFloat(((count / totalUsers) * 100).toFixed(2))
            : 0,
      };
    });

    return {
      name: "Employment Types",
      data: employmentTypeData.map((item) => item.type),
      values: employmentTypeData.map((item) => item.percentage),
      counts: employmentTypeData.map((item) => item.count),
      totalUsers: totalUsers,
    };
  };

  // Employee Department Data
  const employeeDepartmentData = async () => {
    // FIRST FETCH DEPARTMENTS
    const departmentsQuery = `
        SELECT id, name
        FROM department
        WHERE organization = '${organizationId}'
            AND ${dateFilter.getSQLWhereDateClause("created_at")}
            AND deleted = 0;
    `;
    const [departmentData] = await db.query(departmentsQuery);

    if (departmentData.length === 0) {
      return { name: "Department", data: [], values: [] };
    }

    // GET DEPARTMENT IDs TO FILTER USER COUNTS
    const departmentIds = departmentData
      .map((dept) => `'${dept.id}'`)
      .join(",");

    // COUNT ONLY USERS IN THESE SPECIFIC DEPARTMENTS
    const totalUsersQuery = `
        SELECT COUNT(*) AS total_users
        FROM users
        WHERE my_organization = '${organizationId}'
            AND ${dateFilter.getSQLWhereDateClause("created_at")}
            AND deleted = '0'
            AND department IN (${departmentIds}) 
    `;

    const [totalUsersResult] = await db.query(totalUsersQuery);
    const totalUsers = totalUsersResult[0].total_users;

    // Step 3: Build dynamic cases for department counts
    const departmentCountCases = departmentData
      .map((dept) => {
        const sanitizedName = dept.name.replace(/\s+/g, "_").toUpperCase();
        return `SUM(CASE WHEN users.department = '${dept.id}' THEN 1 ELSE 0 END) AS \`${sanitizedName}\``;
      })
      .join(", ");

    const combinedQuery = `
        SELECT ${departmentCountCases}
        FROM users
        WHERE my_organization = '${organizationId}'
            AND ${dateFilter.getSQLWhereDateClause("created_at")}
            AND deleted = '0'
            AND department IS NOT NULL
    `;

    // Step 4: Execute the query
    // console.log("combinedQuery: ", combinedQuery);
    const [result] = await db.query(combinedQuery);
    const row = result[0];
    // console.log("result: ", result);

    // Step 5: Prepare response with percentages
    const department = {
      name: "Department",
      data: departmentData.map((dept) => dept.name),
      values: departmentData.map((dept) => {
        const sanitizedName = dept.name.replace(/\s+/g, "_").toUpperCase();
        const count = row[sanitizedName] || 0;
        return totalUsers > 0
          ? parseFloat(((count / totalUsers) * 100).toFixed(2))
          : 0;
      }),
      totalUsers, // Optional inclusion remains unchanged
    };

    return department;
  };

  // Employee Demographics
  const employeeDemographicsData = async () => {
    // Query to get all employee types with their counts
    const employeeTypeQuery = `
      SELECT 
        employee_type,
        COUNT(users.id) AS type_count
      FROM users 
      WHERE my_organization = '${organizationId}'
        AND ${dateFilter.getSQLWhereDateClause("created_at")}
        AND deleted = '0'
        AND employee_type IS NOT NULL
      GROUP BY employee_type
    `;

    // Query to get disability count
    const disabilityQuery = `
      SELECT 
        COUNT(*) AS disability_count
      FROM users
      WHERE my_organization = '${organizationId}'
        AND ${dateFilter.getSQLWhereDateClause("created_at")}
        AND deleted = '0'
        AND disability IS NOT NULL
    `;

    // Execute both queries
    const [employeeTypeResults, [disabilityResult]] = await Promise.all([
      db.query(employeeTypeQuery),
      db.query(disabilityQuery),
    ]);

    // Prepare the final result in the desired format
    const demographicsData = [
      // Employee Type Distribution
      ...employeeTypeResults[0].map((type) => ({
        x: type.employee_type,
        y: type.type_count,
      })),

      // Disability Status
      {
        x: "Disability Status",
        y: disabilityResult[0].disability_count,
      },
    ];

    return demographicsData;
  };

  // Meeting Attendees
  const meetingMinutes = async () => {
    // Query to get all meeting notes status with their counts
    const meetingNotesQuery = `
      SELECT 
        status,
        COUNT(meeting_notes.id) AS type_count
      FROM meeting_notes
      LEFT JOIN meeting ON meeting.id = meeting_notes.meeting_id 
      WHERE meeting.organization = '${organizationId}'
        AND ${dateFilter.getSQLWhereDateClause("meeting_notes.created_at")}
        AND meeting_notes.deleted = '0'
      GROUP BY status
    `;

    // Execute the query
    const [meetingNotesResults] = await db.query(meetingNotesQuery);

    // Prepare the meeting notes status data object
    const meetingNotes = {};
    meetingNotes["name"] = "Meeting Action Status Report";
    meetingNotes["data"] = meetingNotesResults
      .map((note) => note?.status?.toUpperCase())
      .filter((e) => e); // Get status names
    meetingNotes["values"] = meetingNotesResults.map((note) => note.type_count); // Get counts

    return meetingNotes;
  };

  // meetingAttendeesDepartment();
  const dashboardData = [];
  const age = await ageData();
  const disable = await disabilityData();
  const race = await raceData();
  const gender = await genderData();
  const employment = await employmentTypeData();
  const department = await employeeDepartmentData();
  const demographics = await employeeDemographicsData();
  const meetingMinute = await meetingMinutes();

  dashboardData.push(age);
  dashboardData.push(disable);
  dashboardData.push(race);
  dashboardData.push(gender);
  dashboardData.push(employment);
  dashboardData.push(department);
  dashboardData.push(demographics);
  dashboardData.push(meetingMinute);

  return sendResponse(res, 200, dashboardData);
};

/** Function to get Dashboard data for Objective & Target */
export const getObjectiveTargetDashboardData = async (req, res) => {
  const { all, filter } = req.query;

  if (!filter) {
    return sendResponse(res, 404, "Filter not found");
  }

  req.query.filter = JSON.parse(filter);
  req.query.organization = req.query.filter?.organization || null;
  if (!req.query.organization) {
    return sendResponse(res, 404, "Organization not found");
  }

  req.query.department = req.query.filter?.department || null;

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = req.query.organization;
  if (req.query.department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(req.query.department);
    organizationId = recordAccordingToOrganization[0].organization;
  }

  // Parse date range
  const dateRange = req.query.filter?.date_type || "All-Time";
  const start_date = req.query.filter?.start_date;
  const end_date = req.query.filter?.end_date;
  const dateFilter = getDateRangeForSQL(dateRange, start_date, end_date);

  const objectiveStatusReportData = await objectiveTargetStatusReport(
    organizationId,
    dateFilter
  );
  const targetStatusReportData = await targetAchievementStatusReport(
    organizationId,
    dateFilter
  );
  const objectiveTargetPlanStatusReportData =
    await objectiveTargetPlanStatusReport(organizationId, dateFilter);

  const finalData = [];

  finalData.push(objectiveStatusReportData);
  finalData.push(targetStatusReportData);
  finalData.push(objectiveTargetPlanStatusReportData);

  return sendResponse(res, 200, finalData);
};

const objectiveTargetStatusReport = async (organizationId, dateFilter) => {
  try {
    if (!organizationId) {
      throw new Error("organizationId is required");
    }

    // Step 1: Fetch the Objective & Target Status from database
    const objectiveTargetStatusQuery = `
    SELECT COUNT(*) AS count, status
    FROM objective_setting
    WHERE organization = '${organizationId}'
      AND ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
    GROUP BY status;
  `;

    // Step 2: Execute the query
    const [objectiveTargetStatusResult] = await db.query(
      objectiveTargetStatusQuery
    );

    // Step 3: If no Objective & Target Status found, return zero values
    if (objectiveTargetStatusResult.length === 0) {
      return {
        name: "Objective & Target Status",
        data: [],
        values: [],
      };
    }

    // Step 4: Prepare the Objective & Target Status data object
    const objectiveTargetStatus = {};
    objectiveTargetStatus["name"] = "Objective & Target Status";
    objectiveTargetStatus["data"] = objectiveTargetStatusResult.map(
      (item) => item.status
    ); // Get original objectiveTargetStatus names

    // Step 5: Calculate percentages and counts
    const totalCount = objectiveTargetStatusResult.reduce(
      (sum, result) => sum + (result.count || 0),
      0
    );

    // Initialize arrays for percentages and counts
    objectiveTargetStatus["values"] = [];
    objectiveTargetStatus["counts"] = [];

    objectiveTargetStatusResult.forEach((result) => {
      const count = result.count || 0; // Directly access the count property

      // Calculate percentage, rounded to 2 decimal places
      const percentage = Math.round((count / totalCount) * 10000) / 100;

      // Push percentage and count to their respective arrays
      objectiveTargetStatus["values"].push(percentage);
      objectiveTargetStatus["counts"].push(count);
    });

    return objectiveTargetStatus;
  } catch (error) {
    throw error;
  }
};

const targetAchievementStatusReport = async (organizationId, dateFilter) => {
  // console.log('dateFilter: ', dateFilter);
  try {
    if (!organizationId) {
      throw new Error("organizationId is required");
    }

    const targetAchievementStatusQuery = `
      SELECT end_date, cac.updated_at, cac.status
      FROM custom_action_creation cac
      INNER JOIN 
        objective_setting 
          ON JSON_CONTAINS(objective_setting.action_taken, CAST(cac.id AS JSON))
      WHERE cac.organization = '${organizationId}'
        AND ${dateFilter.getSQLWhereDateClause("cac.created_at")}
        AND cac.deleted = 0
      GROUP BY cac.status, cac.end_date, cac.updated_at;
    `;

    // console.log('targetAchievementStatusQuery: ', targetAchievementStatusQuery);
    const [results] = await db.query(targetAchievementStatusQuery);
    // console.log('results: ', results);

    let achievedCount = 0;
    let failedCount = 0;
    let notAchievedCount = 0;
    const currentDate = new Date();

    for (const row of results) {
      const endDate = new Date(row.end_date);
      const updatedAt = new Date(row.updated_at);
      const status = row.status;
      const isEndDateLapsed = currentDate > endDate;

      if (status === "Completed") {
        // Handle completed tasks
        if (updatedAt <= endDate) {
          achievedCount++;
        } else {
          notAchievedCount++;
        }
      } else if (
        isEndDateLapsed &&
        ["In-Progress", "Pending", "To-Do"].includes(status)
      ) {
        // New failure condition for lapsed tasks with non-completed statuses
        failedCount++;
      } else {
        // Handle non-completed tasks within deadline
        notAchievedCount++;
      }
    }

    // Calculate percentages
    const total = achievedCount + failedCount + notAchievedCount;
    const calculatePercentage = (count) =>
      total > 0 ? parseFloat(((count / total) * 100).toFixed(2)) : 0;

    return {
      name: "Target Achievement Status",
      data: [
        "Target Achieved",
        "Target Failed to Achieve",
        "Target Not Achieved",
      ],
      values: [
        calculatePercentage(achievedCount),
        calculatePercentage(failedCount),
        calculatePercentage(notAchievedCount),
      ],
      counts: [achievedCount, failedCount, notAchievedCount],
    };
  } catch (error) {
    throw error;
  }
};

const objectiveTargetPlanStatusReport = async (organizationId, dateFilter) => {
  // console.log("dateFilter: ", dateFilter);
  try {
    if (!organizationId) {
      throw new Error("organizationId is required");
    }

    // Parse dates and adjust start date if range > 1 year
    let startDate = new Date(dateFilter.startDate);
    const endDate = new Date(dateFilter.endDate);

    // Calculate if range exceeds 12 months
    const monthDifference =
      (endDate.getFullYear() - startDate.getFullYear()) * 12 +
      (endDate.getMonth() - startDate.getMonth());

    // If range > 12 months, show latest 12 months
    if (monthDifference > 11) {
      startDate = new Date(endDate);
      startDate.setDate(1); // Set the day to 1 first
      startDate.setFullYear(endDate.getFullYear() - 1); // Go back one year
      startDate.setMonth(endDate.getMonth() + 1); // Go back one month
    }

    // Generate months between adjusted start and end dates
    const allMonths = generateAllMonths(startDate, endDate);
    // console.log("allMonths: ", allMonths);

    // Initialize monthMap with zero counts
    const monthMap = new Map();
    allMonths.forEach((month) => {
      monthMap.set(month, { achieved: 0, failed: 0, notAchieved: 0 });
    });

    // SQL query
    const targetAchievementStatusQuery = `
      SELECT cac.created_at, cac.end_date, cac.updated_at, cac.status
      FROM custom_action_creation cac
      INNER JOIN objective_setting 
        ON JSON_CONTAINS(objective_setting.action_taken, CAST(cac.id AS JSON))
      WHERE cac.organization = '${organizationId}'
        AND ${dateFilter.getSQLWhereDateClause("cac.created_at")}
        AND cac.deleted = 0
      GROUP BY cac.status, cac.end_date, cac.updated_at, cac.created_at;
    `;

    const [results] = await db.query(targetAchievementStatusQuery);
    const currentDate = new Date();

    for (const row of results) {
      const createdDate = new Date(row.created_at + " UTC");
      const monthYear = createdDate.toLocaleDateString("en-US", {
        month: "short",
        year: "numeric",
      });

      if (!monthMap.has(monthYear)) continue;

      const endDateRow = new Date(row.end_date + " UTC");
      const updatedAt = new Date(row.updated_at + " UTC");
      const status = row.status;
      const isEndDateLapsed = currentDate > endDateRow;

      let category;
      if (status === "Completed") {
        category = updatedAt <= endDateRow ? "achieved" : "notAchieved";
      } else if (
        isEndDateLapsed &&
        ["In-Progress", "Pending", "To-Do"].includes(status)
      ) {
        category = "failed";
      } else {
        category = "notAchieved";
      }

      monthMap.get(monthYear)[category]++;
    }

    // // Format final output
    // let formattedData = [
    //   {
    //     name: "Target Achieved",
    //     data: allMonths.map((month) => monthMap.get(month).achieved),
    //   },
    //   {
    //     name: "Target Failed to Achieve",
    //     data: allMonths.map((month) => monthMap.get(month).failed),
    //   },
    //   {
    //     name: "Target Not Achieved",
    //     data: allMonths.map((month) => monthMap.get(month).notAchieved),
    //   },
    //   {
    //     keys: allMonths,
    //   }
    // ];

    // Format final output
    const formattedData = {
      name: "Objective Target Plan Status Report",
      data: [
        {
          name: "Target Achieved",
          counts: allMonths.map((month) => monthMap.get(month).achieved),
        },
        {
          name: "Target Failed to Achieve",
          counts: allMonths.map((month) => monthMap.get(month).failed),
        },
        {
          name: "Target Not Achieved",
          counts: allMonths.map((month) => monthMap.get(month).notAchieved),
        },
      ],
      keys: allMonths,
    };

    // return formattedData = {
    //   name: "Target Achievement Status",
    //   data: ["Target Achieved", "Target Failed to Achieve", "Target Not Achieved"],
    //   values: [
    //     allMonths.map((month) => monthMap.get(month).achieved),
    //     allMonths.map((month) => monthMap.get(month).failed),
    //     allMonths.map((month) => monthMap.get(month).notAchieved),
    //   ],
    //   counts: [
    //     allMonths.map((month) => monthMap.get(month).achieved),
    //     allMonths.map((month) => monthMap.get(month).failed),
    //     allMonths.map((month) => monthMap.get(month).notAchieved),
    //   ],
    // };
    return formattedData;
  } catch (error) {
    throw error;
  }
};

function generateAllMonths(start, end) {
  // console.log("start, end: ", start, end);

  // If the difference between dates is less than 15 days
  const diffInDays = (new Date(end) - new Date(start)) / (1000 * 60 * 60 * 24);

  if (diffInDays <= 15) {
    // Use the end date's month for short ranges
    const endDate = new Date(end);
    return [
      endDate.toLocaleDateString("en-US", {
        month: "short",
        year: "numeric",
        timeZone: "UTC",
      }),
    ];
  }

  // For longer ranges, get all months between dates
  const months = [];
  const current = new Date(start);
  current.setDate(1); // Start from the first day of the month
  current.setUTCHours(0, 0, 0, 0);
  const endDate = new Date(end);
  endDate.setDate(1); // Ensure the end date is also the first day of the month

  // Loop through all months, including the start and end months
  while (current <= endDate) {
    months.push(
      current.toLocaleDateString("en-US", {
        month: "short",
        year: "numeric",
        timeZone: "UTC",
      })
    );
    current.setMonth(current.getMonth() + 1); // Move to the next month
  }

  return months;
}

/** *********************  AUDIT MODULE ***************************** */
export const auditManagementDashboardData = async (req, res) => {
  const { all, filter } = req.query;

  if (!filter) {
    return sendResponse(res, 404, "Filter not found");
  }

  req.query.filter = JSON.parse(filter);

  // Parse date range
  const dateRange = req.query.filter?.date_type || "All-Time";
  const start_date = req.query.filter?.start_date;
  const end_date = req.query.filter?.end_date;
  const dateFilter = getDateRangeForSQL(dateRange, start_date, end_date);

  const auditByBusinessesChartData = await auditByBusinesses(dateFilter);
  const auditByAuditTypeOriginBasedChartData =
    await auditByAuditTypeOriginBased(dateFilter);
  const auditByStatusChartData = await auditByStatus(dateFilter);
  const auditByCategoryChartData = await auditByCategory(dateFilter);
  const auditByOverdueChartData = await auditByOverdueAuditStartDate(
    dateFilter
  );
  const monitoringByAnnualTrendChartData = await monitoringByAnnualTrend();
  const complianceByPercentageSuccessRateChartData = await complianceByPercentageSuccessRate(dateFilter);
  const complianceByPercentageSuccessRateTimelineChartData = await complianceByPercentageSuccessRateTimeline(dateFilter);
  const monitoringByBusinessesChartData = await monitoringByBusinesses(
    dateFilter
  );
  const monitoringByMonitoringTypeChartData = await monitoringByMonitoringType(
    dateFilter
  );
  const monitoringByCategoryChartData = await monitoringByCategory(dateFilter);
  const monitoringByStatusChartData = await monitoringByStatus(dateFilter);
  const monitoringByResponsiblePersonBasedOnTypeChartData =
    await monitoringByResponsiblePersonBasedOnType(dateFilter);
  const monitoringByResponsiblePersonBasedOnStatusChartData =
    await monitoringByResponsiblePersonBasedOnStatus(dateFilter);
  const complianceByBusinessesChartData = await complianceByBusiness(
    dateFilter
  );
  const complianceByMonitoringTypeChartData = await complianceByMonitoringType(
    dateFilter
  );
  const complianceByResponsiblePersonChartData = await complianceByResponsiblePerson(dateFilter);

  const finalData = {
    auditByBusinesses: auditByBusinessesChartData,
    auditByAuditTypeOriginBased: auditByAuditTypeOriginBasedChartData,
    auditByStatus: auditByStatusChartData,
    auditByCategory: auditByCategoryChartData,
    auditByOverdue: auditByOverdueChartData,
    monitoringByAnnualTrend: monitoringByAnnualTrendChartData,
    complianceByPercentageSuccessRate: complianceByPercentageSuccessRateChartData,
    complianceByPercentageSuccessRateTimeline: complianceByPercentageSuccessRateTimelineChartData,
    monitoringByBusiness: monitoringByBusinessesChartData,
    monitoringByMonitoringType: monitoringByMonitoringTypeChartData,
    monitoringByCategory: monitoringByCategoryChartData,
    monitoringByStatus: monitoringByStatusChartData,
    monitoringByResponsiblePersonBasedOnType:
      monitoringByResponsiblePersonBasedOnTypeChartData,
    monitoringByResponsiblePersonBasedOnStatus:
      monitoringByResponsiblePersonBasedOnStatusChartData,
    complianceByBusiness: complianceByBusinessesChartData,
    complianceByMonitoringType: complianceByMonitoringTypeChartData,
    complianceByResponsiblePerson: complianceByResponsiblePersonChartData,
  };

  return sendResponse(res, 200, finalData);
};

/*********************************************************************/
/********************** 🌟 NEW DASHBOARD 🌟 *************************/
/*********************** 📌 FUNCTIONS 📌 **************************/
/*********************************************************************/

/** *********************  RISK MODULE ***************************** */

/** All Risks by Business Organizations */
const riskByBusinesses = async (dateFilter) => {
  try {
    // Query to get all organizations
    const getOrganizationQuery = `
      SELECT
        id, parent_id, name, level, created_at, updated_at, unique_id
      FROM organization
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
    `;
    // Query to get all risks based on organization
    const getRiskQuery = (riskType, organizationId) => `
      SELECT
        COUNT(id) AS count
      FROM ${riskType}_risk
      WHERE organization = '${organizationId}' 
      AND ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
      GROUP BY id;
    `;

    const riskTypes = ["strategic", "tactical", "operational"];
    const chartSeries = [];
    let chartXaxis = []; // To store organization names

    const [orgData] = await db.query(getOrganizationQuery);

    if (orgData && orgData.length > 0) {
      // Prepare xaxis categories (Organization Names)
      chartXaxis = orgData.map((org) => org.name);

      // Loop through risk types to create series
      for (const riskType of riskTypes) {
        const riskDataForType = []; // Data for current risk type across all organizations
        for (const organization of orgData) {
          const query = getRiskQuery(riskType, organization.id);
          const [results] = await db.query(query);
          let totalCount = 0;
          if (results && results.length > 0 && results[0].count !== null) {
            totalCount = results[0].count;
          }
          riskDataForType.push(totalCount); // Push count for current organization
        }
        const categoryName =
          riskType.charAt(0).toUpperCase() + riskType.slice(1);
        chartSeries.push({
          name: categoryName,
          type: "bar", // Assuming bar chart type as per example
          data: riskDataForType,
          total: riskDataForType.reduce((sum, count) => sum + count, 0), // Calculate total
        });
      }
    }

    const chartData = {
      data: chartSeries,
      xaxis: chartXaxis,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** Top 10 Risks by Risk Categories */
const topTenRisks = async (dateFilter) => {
  try {
    const riskTypes = ["strategic", "tactical", "operational"];
    const chartSeries = [];
    let chartXaxis = []; // To store top 10 risk register names
    const riskRegisterCountsByType = {}; // To store counts per risk type and register name

    // Initialize data structure to hold counts for each risk type
    riskTypes.forEach((riskType) => {
      riskRegisterCountsByType[riskType] = {};
    });

    for (const riskType of riskTypes) {
      // Modified query to get risk register names and count
      const getRiskQuery = (riskType) => `
        SELECT
          risk_register_name,
          COUNT(id) AS count
        FROM ${riskType}_risk
        WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
        AND deleted = 0
        GROUP BY risk_register_name
      `;

      const [results] = await db.query(getRiskQuery(riskType));

      if (results && results.length > 0) {
        for (const result of results) {
          const riskName = result.risk_register_name;
          const count = parseInt(result.count, 10);
          riskRegisterCountsByType[riskType][riskName] = count; // Store count for each risk type and name
        }
      }
    }

    // Calculate total count for each risk_register_name across all risk types and sort
    const riskRegisterTotalCounts = {};
    for (const riskType of riskTypes) {
      for (const riskName in riskRegisterCountsByType[riskType]) {
        if (!riskRegisterTotalCounts[riskName]) {
          riskRegisterTotalCounts[riskName] = 0;
        }
        riskRegisterTotalCounts[riskName] +=
          riskRegisterCountsByType[riskType][riskName];
      }
    }

    const sortedRiskNames = Object.entries(riskRegisterTotalCounts)
      .sort(([, a], [, b]) => b - a) // Sort by total count descending
      .slice(0, 10) // Get top 10
      .map(([name, count]) => name); // Extract just the names

    chartXaxis = sortedRiskNames; // Set xaxis to top 10 risk register names

    // Create chart series for each risk type
    for (const riskType of riskTypes) {
      const riskDataForType = [];
      for (const riskName of sortedRiskNames) {
        riskDataForType.push(riskRegisterCountsByType[riskType][riskName] || 0); // Get count or 0 if no data
      }
      const categoryName = riskType.charAt(0).toUpperCase() + riskType.slice(1);
      chartSeries.push({
        name: categoryName,
        type: "bar",
        data: riskDataForType,
        total: riskDataForType.reduce((sum, count) => sum + count, 0), // Calculate total for each risk type
      });
    }

    const chartData = {
      data: chartSeries,
      xaxis: chartXaxis,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** All Risks by Issue(Risk) Types */
const risksByAllIssueType = async (dateFilter) => {
  try {
    const getRiskQuery = (riskType) => `
      SELECT
        COUNT(id) AS count
      FROM ${riskType}_risk
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
      GROUP BY id;
    `;

    const riskTypes = ["strategic", "tactical", "operational"];
    const chartSeries = [];
    let chartCategories = riskTypes.map(
      (type) => type.charAt(0).toUpperCase() + type.slice(1)
    );
    const riskCounts = {}; // To store counts for each risk type

    const dataArray = []; // Array to hold data for the series

    for (const riskType of riskTypes) {
      const [results] = await db.query(getRiskQuery(riskType));
      let totalCount = 0;
      if (results && results.length > 0) {
        // Sum up the counts from all rows
        totalCount = results.reduce((sum, row) => sum + (row.count || 0), 0);
      }
      riskCounts[riskType] = totalCount; // Store count for each risk type
      dataArray.push(totalCount); // Push total count to dataArray
    }

    // Structure data for ApexCharts series - now only one series
    chartSeries.push({
      name: "Count", // Generic series name
      type: "bar",
      data: dataArray, // Data array now contains counts for all risk types
    });

    const chartData = {
      data: chartSeries,
      xaxis: chartCategories, // Categories are risk types on xaxis
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** All Risk Identifications by Risk Type */
const riskByRiskType = async (dateFilter) => {
  try {
    // Get Risk Identifications by Risk Type - Modified to fetch risk_register_name
    const getRiskIdentificationQuery = (riskType) => `
      SELECT
        COUNT(identification.id) AS count, risk.risk_register_name
      FROM ${riskType}_risk_identification AS identification
      INNER JOIN ${riskType}_risk AS risk ON identification.risk_register_id = risk.id
      WHERE ${dateFilter.getSQLWhereDateClause("identification.created_at")}
      AND ${dateFilter.getSQLWhereDateClause(
      "risk.created_at"
    )} AND risk.deleted = 0
      AND identification.deleted = 0
      GROUP BY risk.risk_register_name
    `;

    const riskTypes = ["strategic", "tactical", "operational"];
    const chartSeries = [];

    for (const riskType of riskTypes) {
      const [riskIdentificationResults] = await db.query(
        getRiskIdentificationQuery(riskType)
      );
      const categoryName = riskType.charAt(0).toUpperCase() + riskType.slice(1);
      const seriesData = [];
      const chartCategories = [];

      if (riskIdentificationResults && riskIdentificationResults.length > 0) {
        riskIdentificationResults.forEach((row) => {
          chartCategories.push(row.risk_register_name || "Unknown Risk");
          seriesData.push(parseInt(row.count, 10) || 0);
        });
      }

      chartSeries.push({
        name: categoryName,
        type: "bar",
        data: seriesData,
        xaxis: chartCategories, // X-axis specific to this risk type
        total: seriesData.reduce((sum, count) => sum + count, 0),
      });
    }

    const chartData = {
      data: chartSeries,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** All Risk Categories (Pie Chart) */
const riskByCategory = async (dateFilter) => {
  try {
    const getRiskQuery = (riskType) => `
      SELECT
        COUNT(id) AS count
      FROM ${riskType}_risk
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
      GROUP BY id;
    `;

    const riskTypes = ["strategic", "tactical", "operational"];
    const chartData = {
      data: [],
      labels: [],
    };

    for (const riskType of riskTypes) {
      const [results] = await db.query(getRiskQuery(riskType));
      let totalCount = 0;
      if (results && results.length > 0) {
        // Sum up the counts from all rows
        totalCount = results.reduce((sum, row) => sum + (row.count || 0), 0);
      }
      chartData.data.push(totalCount);
      chartData.labels.push(
        riskType.charAt(0).toUpperCase() + riskType.slice(1)
      );
    }

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** Risks by Rating (Residual Rating of Risks) (Pie Chart) */
const riskByResidualRating = async (dateFilter) => {
  try {
    // Query to get count of risks for each residual ranking
    const getRiskQuery = `
      SELECT
        COUNT(residual_ranking) AS count,
        residual_ranking
      FROM operational_risk_identification
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
      GROUP BY residual_ranking
      ORDER BY
        CASE residual_ranking
          WHEN 'Low' THEN 1
          WHEN 'Medium' THEN 2
          WHEN 'High' THEN 3
          WHEN 'Very High' THEN 4
          ELSE 5 -- Handle unexpected values if any
        END;
    `;

    const [results] = await db.query(getRiskQuery);

    const data = [];
    const labels = [];
    const rankingMap = {
      Low: 0,
      Medium: 1,
      High: 2,
      "Very High": 3,
    };
    const orderedLabels = ["Low", "Medium", "High", "Very High"];
    const counts = [0, 0, 0, 0];

    results.forEach((row) => {
      const ranking = row.residual_ranking;
      const count = parseInt(row.count, 10);
      if (rankingMap.hasOwnProperty(ranking)) {
        counts[rankingMap[ranking]] = count;
      }
    });

    orderedLabels.forEach((label) => {
      labels.push(label);
      data.push(counts[rankingMap[label]]);
    });

    return {
      data: data,
      labels: labels,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Risks per Work Area (Operational Risks) */
const riskPerWorkArea = async (dateFilter) => {
  try {
    // Query to get All Operational Work areas
    const getWorkAreaQuery = `
    SELECT id, name 
    FROM operational_work_area 
    WHERE ${dateFilter.getSQLWhereDateClause("created_at")} AND deleted = 0;
    `;

    // Query to get all Operational Activities
    const getActivityQuery = `
    SELECT id, work_id 
    FROM operational_activities 
    WHERE ${dateFilter.getSQLWhereDateClause("created_at")} AND deleted = 0;`;

    // Query to get all Operational Risk Identifications
    const getRiskIdentificationQuery = `
      SELECT activity_id, residual_ranking
      FROM operational_risk_identification 
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")} AND deleted = 0;`;

    // Execute the queries
    const [workAreaResult] = await db.query(getWorkAreaQuery);
    const [activityResult] = await db.query(getActivityQuery);
    const [riskIdentificationResult] = await db.query(
      getRiskIdentificationQuery
    );

    const workAreas = workAreaResult;
    const activities = activityResult;
    const riskIdentifications = riskIdentificationResult;
    const riskRankings = ["Low", "Medium", "High", "Very High"];
    const responseData = [];

    for (const ranking of riskRankings) {
      const rankingData = {
        name: ranking,
        type: "bar",
        data: Array(workAreas.length).fill(0),
        total: 0,
      };

      for (let i = 0; i < workAreas.length; i++) {
        const workArea = workAreas[i];
        let count = 0;

        // Find activities for the current work area
        const workAreaActivities = activities.filter(
          (activity) => activity.work_id === workArea.id
        );

        // Count risk identifications for the current ranking within these activities
        for (const activity of workAreaActivities) {
          const riskCount = riskIdentifications.filter(
            (risk) =>
              risk.activity_id === activity.id &&
              risk.residual_ranking === ranking
          ).length;
          count += riskCount;
        }

        rankingData.data[i] = count;
        rankingData.total += count;
      }
      responseData.push(rankingData);
    }

    return {
      data: responseData,
      xaxis: workAreas.map((wa) => wa.name),
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Risks per Main Process (Operational Risks) */
const riskPerMainProcess = async (dateFilter) => {
  try {
    // Query to get all Main Process
    const getMainProcessQuery = `
    SELECT id, name 
    FROM main_process 
    WHERE ${dateFilter.getSQLWhereDateClause("created_at")} AND deleted = 0;`;

    // Query to get All Risks
    const getRiskQuery = `
    SELECT id, main_process 
    FROM operational_risk 
    WHERE ${dateFilter.getSQLWhereDateClause("created_at")} AND deleted = 0;`;

    // Query to get All Risks Identifications
    const getRiskIdentificationQuery = `
      SELECT risk_register_id, residual_ranking
      FROM operational_risk_identification 
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")} AND deleted = 0;`;

    // Execute the queries
    const [mainProcessResult] = await db.query(getMainProcessQuery);
    const [riskResult] = await db.query(getRiskQuery);
    const [riskIdentificationResult] = await db.query(
      getRiskIdentificationQuery
    );

    const mainProcesses = mainProcessResult;
    const risks = riskResult;
    const riskIdentifications = riskIdentificationResult;
    const riskRankings = ["Low", "Medium", "High", "Very High"];
    const responseData = [];

    for (const ranking of riskRankings) {
      const rankingData = {
        name: ranking,
        type: "bar",
        data: Array(mainProcesses.length).fill(0),
        total: 0,
      };

      for (let i = 0; i < mainProcesses.length; i++) {
        const mainProcess = mainProcesses[i];
        let count = 0;

        // Find risks for the current main process
        const mainProcessRisks = risks.filter(
          (risk) => risk.main_process === mainProcess.id
        );

        // Count risk identifications for the current ranking within these risks
        for (const risk of mainProcessRisks) {
          const riskCount = riskIdentifications.filter(
            (identification) =>
              identification.risk_register_id === risk.id &&
              identification.residual_ranking === ranking
          ).length;
          count += riskCount;
        }

        rankingData.data[i] = count;
        rankingData.total += count;
      }
      responseData.push(rankingData);
    }

    return {
      data: responseData,
      xaxis: mainProcesses.map((mp) => mp.name),
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Risks (Risk Registers) by Upcoming Review Date */
const riskByReviewDate = async (dateFilter) => {
  try {
    const getRiskQuery = (riskType) => `
      SELECT
        review_date
      FROM ${riskType}_risk
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
    `;

    const riskTypes = ["strategic", "tactical", "operational"];
    const reviewDateGroups = {
      Overdue: 0,
      "0-30 Days": 0,
      "31-60 Days": 0,
      "61-90 Days": 0,
      "90+ Days": 0,
      "No Date": 0,
    };

    const today = new Date();
    today.setHours(0, 0, 0, 0); // Normalize today's date

    for (const riskType of riskTypes) {
      const [riskResult] = await db.query(getRiskQuery(riskType));
      const risks = riskResult;

      for (const risk of risks) {
        const reviewDate = new Date(risk.review_date);

        if (!risk.review_date) {
          reviewDateGroups["No Date"]++;
        } else {
          reviewDate.setHours(0, 0, 0, 0); // Normalize review date

          if (reviewDate < today) {
            reviewDateGroups["Overdue"]++;
          } else {
            const timeDifference = reviewDate.getTime() - today.getTime();
            const daysDifference = Math.ceil(
              timeDifference / (1000 * 3600 * 24)
            );

            if (daysDifference >= 0 && daysDifference <= 30) {
              reviewDateGroups["0-30 Days"]++;
            } else if (daysDifference > 30 && daysDifference <= 60) {
              reviewDateGroups["31-60 Days"]++;
            } else if (daysDifference > 60 && daysDifference <= 90) {
              reviewDateGroups["61-90 Days"]++;
            } else if (daysDifference > 90) {
              reviewDateGroups["90+ Days"]++;
            }
          }
        }
      }
    }

    return {
      data: [
        reviewDateGroups.Overdue,
        reviewDateGroups["0-30 Days"],
        reviewDateGroups["31-60 Days"],
        reviewDateGroups["61-90 Days"],
        reviewDateGroups["90+ Days"],
        reviewDateGroups["No Date"],
      ],
      xaxis: [
        "Overdue",
        "0-30 Days",
        "31-60 Days",
        "61-90 Days",
        "90+ Days",
        "No Date",
      ],
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Risks (Risk Registers) by Overdue Review Date */
const riskByOverdueReviewDate = async (dateFilter) => {
  try {
    const getRiskQuery = (riskType) => `
      SELECT
        review_date
      FROM ${riskType}_risk
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
    `;

    const riskTypes = ["strategic", "tactical", "operational"];
    const overdueGroups = {
      "90+ Days": 0,
      "0-30 Days": 0,
      "31-60 Days": 0,
      "61-90 Days": 0,
    };

    const today = new Date();
    today.setHours(0, 0, 0, 0); // Normalize today's date

    for (const riskType of riskTypes) {
      const [riskResult] = await db.query(getRiskQuery(riskType));
      const risks = riskResult;

      for (const risk of risks) {
        const reviewDate = new Date(risk.review_date);

        if (risk.review_date) {
          reviewDate.setHours(0, 0, 0, 0); // Normalize review date

          if (reviewDate < today) {
            const timeDifference = today.getTime() - reviewDate.getTime();
            const daysOverdue = Math.floor(timeDifference / (1000 * 3600 * 24));

            if (daysOverdue >= 0 && daysOverdue <= 30) {
              overdueGroups["0-30 Days"]++;
            } else if (daysOverdue > 30 && daysOverdue <= 60) {
              overdueGroups["31-60 Days"]++;
            } else if (daysOverdue > 60 && daysOverdue <= 90) {
              overdueGroups["61-90 Days"]++;
            } else if (daysOverdue > 90) {
              overdueGroups["90+ Days"]++;
            }
          }
        }
      }
    }

    return {
      data: [
        overdueGroups["90+ Days"],
        overdueGroups["0-30 Days"],
        overdueGroups["31-60 Days"],
        overdueGroups["61-90 Days"],
      ],
      xaxis: ["90+ Days", "0-30 Days", "31-60 Days", "61-90 Days"],
    };
  } catch (error) {
    throw error;
  }
};


/** ****************** AUDIT MODULE FUNCTIONS ************************** */

/** No.of Audits by Business Organizations */
const auditByBusinesses = async (dateFilter) => {
  try {
    // Query to get the count of audits for each organization
    const getAuditCountByOrganizationQuery = `
      SELECT
        o.id AS organization_id,
        o.name AS organization_name,
        COUNT(a.id) AS audit_count
      FROM organization o
      LEFT JOIN audit_scheduling a ON o.id = a.organization
      WHERE ${dateFilter.getSQLWhereDateClause("o.created_at")}
        AND o.deleted = 0
        AND ${dateFilter.getSQLWhereDateClause("a.created_at")}
        AND a.schedule_type = 'audit'
        AND a.deleted = 0
      GROUP BY o.id, o.name
      ORDER BY o.name;
    `;

    const [results] = await db.query(getAuditCountByOrganizationQuery);

    const chartXaxis = [];
    const chartSeries = [
      {
        name: "Number of Audits",
        type: "bar",
        data: [],
        total: 0,
      },
    ];

    if (results && results.length > 0) {
      results.forEach((row) => {
        chartXaxis.push(row.organization_name);
        chartSeries[0].data.push(row.audit_count);
        chartSeries[0].total += row.audit_count;
      });
    }

    const chartData = {
      data: chartSeries,
      xaxis: chartXaxis,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** No. of Audits based on Audit Type Origin based (Internal & External) */
const auditByAuditTypeOriginBased = async (dateFilter) => {
  try {
    // query to get the count of audits grouped by source
    const getAuditCountBySourceQuery = `
      SELECT source, COUNT(id) AS count
      FROM audit_scheduling
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND schedule_type = 'audit'
      AND deleted = 0
      GROUP BY source
    `;

    const [auditResults] = await db.query(getAuditCountBySourceQuery);

    const auditCounts = [];
    const auditSourceLabels = [];

    if (auditResults && auditResults.length > 0) {
      auditResults.forEach((result) => {
        auditCounts.push(result.count);
        auditSourceLabels.push(result.source);
      });
    }

    const chartData = {
      data: auditCounts,
      labels: auditSourceLabels,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** No. of audits based on status (Audit Schedule Status) */
const auditByStatus = async (dateFilter) => {
  try {
    // query to get all audits with organization information
    const getAuditQuery = `
      SELECT COUNT(asched.id) as count, asched.status, org.name as organization_name
      FROM audit_scheduling asched
      INNER JOIN organization org ON asched.organization = org.id
      WHERE ${dateFilter.getSQLWhereDateClause("asched.created_at")}
      AND asched.schedule_type = 'audit'
      AND asched.deleted = 0
      GROUP BY asched.status, org.name
      ORDER BY asched.status, org.name;
    `;

    // Query to get all organization names for xaxis
    const getOrganizationsQuery = `
      SELECT name 
      FROM organization 
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0 
      ORDER BY name;
    `;

    const status = ["To-do", "In progress", "Executed", "Overdue"];
    const organizationsResult = await db.query(getOrganizationsQuery);
    const organizations = (
      organizationsResult.rows || organizationsResult[0]
    ).map((org) => org.name);
    const queryResult = await db.query(getAuditQuery);
    const rows = queryResult.rows || queryResult[0];

    const finalResponse = {
      data: [],
      xaxis: organizations,
    };

    for (const stat of status) {
      const statusData = {
        name: stat,
        data: new Array(organizations.length).fill(0), // Initialize with 0 for each organization
      };

      for (const row of rows) {
        if (row.status === stat) {
          const orgIndex = organizations.indexOf(row.organization_name);
          if (orgIndex !== -1) {
            statusData.data[orgIndex] = parseInt(row.count);
          }
        }
      }
      finalResponse.data.push(statusData);
    }

    return finalResponse;
  } catch (error) {
    throw error;
  }
};

/** No. of audits based on audit category */
const auditByCategory = async (dateFilter) => {
  try {
    // query to get all audit types
    const getAuditTypeQuery = `
      SELECT id, name
      FROM audit_type
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
    `;

    const [auditTypes] = await db.query(getAuditTypeQuery);

    const auditCounts = [];
    const auditTypeLabels = [];
    const auditTypeColors = []; // Array to store hex color codes

    if (auditTypes && auditTypes.length > 0) {
      for (const auditType of auditTypes) {
        const hexColor = await getColorForCategory(auditType.id, "audit_type");

        // query to get the count of audits for the current audit type
        const getAuditCountQuery = `
          SELECT COUNT(id) AS count
          FROM audit_scheduling
          WHERE type = '${auditType.id}'
          AND ${dateFilter.getSQLWhereDateClause("created_at")}
          AND schedule_type = 'audit'
          AND deleted = 0
          GROUP BY type;
        `;

        const [auditResult] = await db.query(getAuditCountQuery);
        const count = auditResult[0]?.count || 0;

        auditCounts.push(count);
        auditTypeLabels.push(auditType.name);
        auditTypeColors.push(hexColor);
      }
    }

    const chartData = {
      data: auditCounts,
      labels: auditTypeLabels,
      colors: auditTypeColors,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** No.of audits based on passed Planned Audit Start Date (Overdue Audits) */
const auditByOverdueAuditStartDate = async (dateFilter) => {
  try {
    // query to get all audits by planned start date
    const getAuditQuery = `
      SELECT planned_start_date
      FROM audit_scheduling
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND schedule_type = 'audit'
      AND deleted = 0
    `;

    const overdueGroups = {
      "90+ Days": 0,
      "0-30 Days": 0,
      "31-60 Days": 0,
      "61-90 Days": 0,
    };

    const queryResult = await db.query(getAuditQuery);
    const rows = queryResult.rows || queryResult[0];

    const today = new Date();

    for (const row of rows) {
      const plannedStartDate = new Date(row.planned_start_date);
      const timeDifference = today.getTime() - plannedStartDate.getTime();
      const daysDifference = Math.ceil(timeDifference / (1000 * 3600 * 24));

      if (daysDifference > 90) {
        overdueGroups["90+ Days"]++;
      } else if (daysDifference >= 0 && daysDifference <= 30) {
        overdueGroups["0-30 Days"]++;
      } else if (daysDifference >= 31 && daysDifference <= 60) {
        overdueGroups["31-60 Days"]++;
      } else if (daysDifference >= 61 && daysDifference <= 90) {
        overdueGroups["61-90 Days"]++;
      }
    }

    return {
      data: [
        overdueGroups["90+ Days"],
        overdueGroups["0-30 Days"],
        overdueGroups["31-60 Days"],
        overdueGroups["61-90 Days"],
      ],
      xaxis: ["90+ Days", "0-30 Days", "31-60 Days", "61-90 Days"],
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Monitoring Audit based on status grouped by Annually */
const monitoringByAnnualTrend = async () => {
  try {
    // Define all possible statuses
    const allStatuses = ["To-do", "In progress", "Executed", "Overdue"];

    // Query to get all monitoring audits by status and year
    const getMonitoringQuery = `
      SELECT status, EXTRACT(YEAR FROM created_at) as year, COUNT(id) as count
      FROM audit_scheduling
      WHERE schedule_type = 'monitoring'
      AND deleted = 0
      GROUP BY status, EXTRACT(YEAR FROM created_at)
      ORDER BY status, year;
    `;

    const queryResult = await db.query(getMonitoringQuery);
    const rows = queryResult.rows || queryResult[0];

    const annualTrendData = {};
    const years = new Set();

    // Populate annualTrendData with existing data
    for (const row of rows) {
      const { status, year, count } = row;
      years.add(parseInt(year));
      if (!annualTrendData[status]) {
        annualTrendData[status] = {};
      }
      annualTrendData[status][year] = parseInt(count);
    }

    // Get sorted unique years
    const sortedYears = Array.from(years).sort();
    const responseData = [];

    // Iterate through all possible statuses
    for (const status of allStatuses) {
      const dataPoints = [];
      // Iterate through all sorted years
      for (const year of sortedYears) {
        // If data exists for the current status and year, use it; otherwise, use 0
        dataPoints.push(annualTrendData[status]?.[year] || 0);
      }
      responseData.push({
        name: status,
        data: dataPoints,
      });
    }

    return {
      data: responseData,
      xaxis: sortedYears,
    };
  } catch (error) {
    throw error;
  }
};

/** Percentage of Success of Compliance Data for a period of Time
 * (Compliance Annual Trends - % Compliance) in Monitoring Audit
 */
const complianceByPercentageSuccessRate = async (dateFilter) => {
  try {
    // query to get compliance data
    const getAuditComplianceQuery = `
      SELECT id, status, evaluation_tool_status, evaluation_tool, audit_record, created_at
      FROM audit_scheduling
      WHERE status = 'Executed'
      AND evaluation_tool_status = 'Added'
      AND schedule_type = 'monitoring'
      AND deleted = 0
      AND ${dateFilter.getSQLWhereDateClause("created_at")};
    `;
    const [auditSchedules] = await db.query(getAuditComplianceQuery);

    const evaluationToolIds = new Set(auditSchedules.map(schedule => schedule.evaluation_tool).filter(id => id));
    const auditRecordIds = new Set(auditSchedules.map(schedule => schedule.audit_record).filter(id => id));

    const evaluationToolsMap = {};
    if (evaluationToolIds.size > 0) {
      // query to get evaluation tool questions
      const evaluationToolQuery = `
        SELECT id, name, quiz, passing_marks
        FROM audit_template
        WHERE deleted = 0 AND id IN (?) AND quiz = 1;
      `;
      const [evaluationTools] = await db.query(evaluationToolQuery, [Array.from(evaluationToolIds)]);
      evaluationTools.forEach(tool => {
        evaluationToolsMap[tool.id] = tool;
      });
    }

    const auditRecordsMap = {};
    if (auditRecordIds.size > 0) {
      // query to get earned points for success calculation
      const auditRecordQuery = `
        SELECT id, total_point, earned_points
        FROM audit_record
        WHERE deleted = 0 AND id IN (?);
      `;
      const [auditRecords] = await db.query(auditRecordQuery, [Array.from(auditRecordIds)]);
      auditRecords.forEach(record => {
        auditRecordsMap[record.id] = record;
      });
    }

    const yearlySuccess = {};

    for (const schedule of auditSchedules) {
      const year = new Date(schedule.created_at).getFullYear();
      if (!yearlySuccess[year]) {
        yearlySuccess[year] = { total: 0, success: 0 };
      }
      yearlySuccess[year].total++;

      const evaluationTool = evaluationToolsMap[schedule.evaluation_tool];
      const auditRecord = auditRecordsMap[schedule.audit_record];

      if (evaluationTool && auditRecord && auditRecord.earned_points >= evaluationTool.passing_marks) {
        yearlySuccess[year].success++;
      }
    }

    const years = Object.keys(yearlySuccess).sort();
    const data = years.map(year => {
      const { total, success } = yearlySuccess[year];
      return total > 0 ? Math.round((success / total) * 100) : 0;
    });
    const xaxis = years;

    return {
      data,
      xaxis
    };
  } catch (error) {
    throw error;
  }
};

/** Percentage of Success of Compliance Data for a period of Time
 * (Compliance Annual Trends - % Compliance) in Monitoring Audit for Last 5 Years
 */
const complianceByPercentageSuccessRateTimeline = async (dateFilter) => {
  try {
    const currentDate = new Date();
    const currentYear = currentDate.getFullYear();
    const currentMonth = currentDate.getMonth(); // 0-indexed

    const timelineData = [];
    const timelineLabels = [];

    for (let year = currentYear - 4; year <= currentYear; year++) {
      for (let month = 0; month < 12; month++) {
        // For the current year, only process up to the current month
        if (year === currentYear && month > currentMonth) {
          break;
        }

        const startDate = new Date(year, month, 1);
        const endDate = new Date(year, month + 1, 0); // Last day of the month

        // Create a temporary date filter for the current month
        const monthlyDateFilter = {
          getSQLWhereDateClause: (dateField) => {
            return `${dateField} >= '${startDate.toISOString().slice(0, 10)}' AND ${dateField} <= '${endDate.toISOString().slice(0, 10)}'`;
          },
        };

        // query to get compliance data for the specific month
        const getAuditComplianceQuery = `
          SELECT id, status, evaluation_tool_status, evaluation_tool, audit_record, created_at
          FROM audit_scheduling
          WHERE status = 'Executed'
          AND evaluation_tool_status = 'Added'
          AND schedule_type = 'monitoring'
          AND deleted = 0
          AND ${monthlyDateFilter.getSQLWhereDateClause("created_at")};
        `;
        const [auditSchedules] = await db.query(getAuditComplianceQuery);

        const evaluationToolIds = new Set(auditSchedules.map(schedule => schedule.evaluation_tool).filter(id => id));
        const auditRecordIds = new Set(auditSchedules.map(schedule => schedule.audit_record).filter(id => id));

        const evaluationToolsMap = {};
        if (evaluationToolIds.size > 0) {
          const evaluationToolQuery = `
            SELECT id, name, quiz, passing_marks
            FROM audit_template
            WHERE deleted = 0 AND id IN (?) AND quiz = 1;
          `;
          const [evaluationTools] = await db.query(evaluationToolQuery, [Array.from(evaluationToolIds)]);
          evaluationTools.forEach(tool => {
            evaluationToolsMap[tool.id] = tool;
          });
        }

        const auditRecordsMap = {};
        if (auditRecordIds.size > 0) {
          const auditRecordQuery = `
            SELECT id, total_point, earned_points
            FROM audit_record
            WHERE deleted = 0 AND id IN (?);
          `;
          const [auditRecords] = await db.query(auditRecordQuery, [Array.from(auditRecordIds)]);
          auditRecords.forEach(record => {
            auditRecordsMap[record.id] = record;
          });
        }

        let totalAudits = 0;
        let successfulAudits = 0;

        for (const schedule of auditSchedules) {
          totalAudits++;
          const evaluationTool = evaluationToolsMap[schedule.evaluation_tool];
          const auditRecord = auditRecordsMap[schedule.audit_record];

          if (evaluationTool && auditRecord && auditRecord.earned_points >= evaluationTool.passing_marks) {
            successfulAudits++;
          }
        }

        const successRate = totalAudits > 0 ? Math.round((successfulAudits / totalAudits) * 100) : 0;
        const monthName = new Intl.DateTimeFormat('en-US', { month: 'short' }).format(startDate);
        timelineLabels.push(`${monthName}-${year}`);
        timelineData.push(successRate);
      }
    }

    return {
      data: timelineData,
      xaxis: timelineLabels,
    };
  } catch (error) {
    throw error;
  }
};


/** No.of Monitoring Audits by Business Organizations  */
const monitoringByBusinesses = async (dateFilter) => {
  try {
    // Query to get the count of audits for each organization
    const getAuditCountByOrganizationQuery = `
      SELECT
        o.id AS organization_id,
        o.name AS organization_name,
        COUNT(a.id) AS audit_count
      FROM organization o
      LEFT JOIN audit_scheduling a ON o.id = a.organization
      WHERE ${dateFilter.getSQLWhereDateClause("o.created_at")}
        AND o.deleted = 0
        AND ${dateFilter.getSQLWhereDateClause("a.created_at")}
        AND a.schedule_type = 'monitoring'
        AND a.deleted = 0
      GROUP BY o.id, o.name
      ORDER BY o.name;
    `;

    const [results] = await db.query(getAuditCountByOrganizationQuery);

    const chartXaxis = [];
    const chartSeries = [
      {
        name: "Number of Monitoring Audits",
        type: "bar",
        data: [],
        total: 0,
      },
    ];

    if (results && results.length > 0) {
      results.forEach((row) => {
        chartXaxis.push(row.organization_name);
        chartSeries[0].data.push(row.audit_count);
        chartSeries[0].total += row.audit_count;
      });
    }

    const chartData = {
      data: chartSeries,
      xaxis: chartXaxis,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** No. of Monitoring Audits based on Audit Type Origin Based (Internal & External) */
const monitoringByMonitoringType = async (dateFilter) => {
  try {
    // query to get the count of audits grouped by source
    const getMonitoringAuditCountBySourceQuery = `
      SELECT source, COUNT(id) AS count
      FROM audit_scheduling
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND schedule_type = 'monitoring'
      AND deleted = 0
      GROUP BY source
    `;

    const [monitoringAuditResults] = await db.query(
      getMonitoringAuditCountBySourceQuery
    );

    const monitoringAuditCounts = [];
    const monitoringAuditSourceLabels = [];

    if (monitoringAuditResults && monitoringAuditResults.length > 0) {
      monitoringAuditResults.forEach((result) => {
        monitoringAuditCounts.push(result.count);
        monitoringAuditSourceLabels.push(result.source);
      });
    }

    const chartData = {
      data: monitoringAuditCounts,
      labels: monitoringAuditSourceLabels,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** No. of Monitoring Audits based on Monitoring Category */
const monitoringByCategory = async (dateFilter) => {
  try {
    // query to get all monitoring audit types
    const getAuditTypeQuery = `
      SELECT id, name
      FROM audit_type
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
    `;

    const [monitoringAuditTypes] = await db.query(getAuditTypeQuery);

    const monitoringAuditCounts = [];
    const monitoringAuditSourceLabels = [];
    const monitoringAuditTypeColors = [];

    if (monitoringAuditTypes && monitoringAuditTypes.length > 0) {
      for (const monitoringAuditType of monitoringAuditTypes) {
        const hexColor = await getColorForCategory(
          monitoringAuditType.id,
          "monitoring_type"
        );

        // query to get the count of audits for the current audit type
        const getAuditCountQuery = `
          SELECT COUNT(id) AS count
          FROM audit_scheduling
          WHERE type = '${monitoringAuditType.id}'
          AND ${dateFilter.getSQLWhereDateClause("created_at")}
          AND schedule_type = 'monitoring'
          AND deleted = 0
          GROUP BY type;
        `;

        const [monitoringAuditResults] = await db.query(getAuditCountQuery);
        const count = monitoringAuditResults[0]?.count || 0;

        monitoringAuditCounts.push(count);
        monitoringAuditSourceLabels.push(monitoringAuditType.name);
        monitoringAuditTypeColors.push(hexColor);
      }
    }

    const chartData = {
      data: monitoringAuditCounts,
      labels: monitoringAuditSourceLabels,
      colors: monitoringAuditTypeColors,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** No. of Monitoring Audits by status (Audit Schedule Status) */
const monitoringByStatus = async (dateFilter) => {
  try {
    // Define all possible monitoring statuses
    const allStatuses = ["To-do", "In progress", "Executed", "Overdue"];
    const finalResponse = {
      data: [],
      labels: [],
    };

    // Initialize data and labels with all possible statuses and zero counts
    for (const status of allStatuses) {
      finalResponse.labels.push(status);
      finalResponse.data.push(0);
    }

    // query to get monitoring audit counts by status
    const getMonitoringQuery = `
      SELECT COUNT(id) as count, status
      FROM audit_scheduling
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND schedule_type = 'monitoring'
      AND deleted = 0
      GROUP BY status
      ORDER BY status;
    `;

    const queryResult = await db.query(getMonitoringQuery);
    const rows = queryResult.rows || queryResult[0];

    // Update counts based on the database results
    for (const row of rows) {
      const statusIndex = finalResponse.labels.indexOf(row.status);
      if (statusIndex !== -1) {
        finalResponse.data[statusIndex] = parseInt(row.count);
      }
    }

    return finalResponse;
  } catch (error) {
    throw error;
  }
};

/** No. of Monitoring Audits as per Responsible Person group by Monitoring Types */
const monitoringByResponsiblePersonBasedOnType = async (dateFilter) => {
  try {
    // query to get monitoring audits for responsible persons with audit type name
    const getAuditQueryByResponsiblePerson = `
      SELECT atype.name AS monitoring_type, CONCAT(users.name, ' ', users.surname) AS responsible_person, atype.id AS monitoring_type_id
      FROM audit_scheduling asched
      LEFT JOIN users ON asched.owner = users.id
      LEFT JOIN audit_type atype ON asched.type = atype.id
      WHERE ${dateFilter.getSQLWhereDateClause("asched.created_at")}
      AND asched.schedule_type = 'monitoring'
      AND asched.deleted = 0
      AND users.deleted = '0'
      GROUP BY atype.name, CONCAT(users.name, ' ', users.surname), atype.id
      ORDER BY atype.name, CONCAT(users.name, ' ', users.surname);
    `;

    const queryResult = await db.query(getAuditQueryByResponsiblePerson);
    const rows = queryResult.rows || queryResult[0];

    const responsiblePersons = [];
    const monitoringTypesMap = new Map(); // Use a Map to store type and its ID
    const data = [];
    const monitoringTypeColors = []; // Array to store colors for monitoring types

    // Extract unique responsible persons and monitoring types with their IDs
    rows.forEach((row) => {
      if (
        row.responsible_person &&
        !responsiblePersons.includes(row.responsible_person)
      ) {
        responsiblePersons.push(row.responsible_person);
      }
      if (row.monitoring_type && !monitoringTypesMap.has(row.monitoring_type)) {
        monitoringTypesMap.set(row.monitoring_type, row.monitoring_type_id);
      }
    });

    // Initialize data array and fetch colors
    for (const [typeName, typeId] of monitoringTypesMap) {
      const color = await getColorForCategory(
        typeId,
        "monitoring_type_responsible_person"
      );
      monitoringTypeColors.push(color); // Add color to the array
      data.push({
        name: typeName,
        type: "bar",
        data: new Array(responsiblePersons.length).fill(0),
        total: 0,
      });
    }

    const monitoringTypes = Array.from(monitoringTypesMap.keys());

    // Populate data array with counts
    rows.forEach((row) => {
      const ownerIndex = responsiblePersons.indexOf(row.responsible_person);
      const typeIndex = monitoringTypes.indexOf(row.monitoring_type);
      if (ownerIndex !== -1 && typeIndex !== -1) {
        data[typeIndex].data[ownerIndex]++;
        data[typeIndex].total++;
      }
    });

    return {
      data: data,
      xaxis: responsiblePersons.sort(), // Sort responsible persons alphabetically
      colors: monitoringTypeColors, // Include the colors array in the response
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Monitoring Audits as per Responsible Person group by Monitoring Status  */
const monitoringByResponsiblePersonBasedOnStatus = async (dateFilter) => {
  try {
    // query to get monitoring audits for responsible persons with status
    const getAuditQueryByResponsiblePerson = `
      SELECT asched.status, CONCAT(users.name, ' ', users.surname) AS responsible_person
      FROM audit_scheduling asched
      LEFT JOIN users ON asched.owner = users.id
      WHERE ${dateFilter.getSQLWhereDateClause("asched.created_at")}
      AND asched.schedule_type = 'monitoring'
      AND asched.deleted = 0
      AND users.deleted = '0'
      GROUP BY asched.status, CONCAT(users.name, ' ', users.surname)
      ORDER BY asched.status, CONCAT(users.name, ' ', users.surname);
    `;

    const queryResult = await db.query(getAuditQueryByResponsiblePerson);
    const rows = queryResult.rows || queryResult[0];

    const responsiblePersons = [];
    const statusTypes = ["To-do", "In progress", "Executed", "Overdue"];
    const data = [];

    // Extract unique responsible persons
    rows.forEach((row) => {
      if (
        row.responsible_person &&
        !responsiblePersons.includes(row.responsible_person)
      ) {
        responsiblePersons.push(row.responsible_person);
      }
    });
    responsiblePersons.sort(); // Sort responsible persons alphabetically for consistent xaxis

    // Initialize data array for each status
    statusTypes.forEach((status) => {
      data.push({
        name: status,
        type: "bar",
        data: new Array(responsiblePersons.length).fill(0),
        total: 0,
      });
    });

    // Populate data array with counts
    rows.forEach((row) => {
      const ownerIndex = responsiblePersons.indexOf(row.responsible_person);
      const statusIndex = statusTypes.findIndex(
        (status) => status === row.status
      );
      if (ownerIndex !== -1 && statusIndex !== -1) {
        data[statusIndex].data[ownerIndex]++;
        data[statusIndex].total++;
      }
    });

    return {
      data: data,
      xaxis: responsiblePersons,
    };
  } catch (error) {
    throw error;
  }
};

/** % of Compliance Monitoring by Business Organizations*/
const complianceByBusiness = async (dateFilter) => {
  try {
    // Query to get the ID of "Compliance Monitoring" audit type
    const getAuditTypeQuery = `
      SELECT id
      FROM audit_type
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
        AND deleted = 0
        AND name = 'Compliance Monitoring'
    `;

    const [auditTypeResult] = await db.query(getAuditTypeQuery);
    const complianceMonitoringType = auditTypeResult[0]?.id || null;

    if (!complianceMonitoringType) {
      return { data: [], xaxis: [] }; // Return empty if "Compliance Monitoring" type is not found
    }

    // Query to get the count of compliance monitoring audits for each organization
    const getAuditCountByOrganizationQuery = `
      SELECT
        o.name AS organization_name,
        COUNT(a.id) AS audit_count
      FROM organization o
      LEFT JOIN audit_scheduling a ON o.id = a.organization
      WHERE ${dateFilter.getSQLWhereDateClause("o.created_at")}
        AND o.deleted = 0
        AND ${dateFilter.getSQLWhereDateClause("a.created_at")}
        AND a.schedule_type = 'monitoring'
        AND a.deleted = 0
        AND a.type = ?
      GROUP BY o.name
      ORDER BY o.name;
    `;

    const auditCountResult = await db.query(getAuditCountByOrganizationQuery, [
      complianceMonitoringType,
    ]);
    const rows = auditCountResult.rows || auditCountResult[0];

    const finalResponse = {
      data: [
        {
          name: "Percentage of Compliance Monitoring Audits",
          type: "bar",
          data: [],
          total: 0,
        },
      ],
      xaxis: [],
    };

    let totalAudits = 0;
    for (const row of rows) {
      const count = parseInt(row.audit_count);
      totalAudits += count;
    }
    finalResponse.data[0].total = totalAudits;

    for (const row of rows) {
      const count = parseInt(row.audit_count);
      const percentage = totalAudits > 0 ? (count / totalAudits) * 100 : 0;
      finalResponse.data[0].data.push(parseFloat(percentage.toFixed(2))); // Store percentage with 2 decimal places
      finalResponse.xaxis.push(row.organization_name);
    }

    return finalResponse;
  } catch (error) {
    throw error;
  }
};

/** No.of Compliance Monitoring Audits by Monitoring Type */
const complianceByMonitoringType = async (dateFilter) => {
  try {
    // Query to get the ID of "Compliance Monitoring" audit type
    const getAuditTypeQuery = `
      SELECT id, name
      FROM audit_type
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
        AND deleted = 0
        AND name = 'Compliance Monitoring'
    `;

    const [auditTypeResult] = await db.query(getAuditTypeQuery);
    const complianceMonitoringTypeId = auditTypeResult[0]?.id || null;

    if (!complianceMonitoringTypeId) {
      return { data: [], labels: [], colors: [] }; // Return empty if "Compliance Monitoring" type is not found
    }

    // Query to get the count of compliance monitoring audits by monitoring type
    const getAuditCountByMonitoringTypeQuery = `
      SELECT
        audit_scheduling.type as type_id,
        audit_type.name as type_name,
        COUNT(audit_scheduling.id) AS audit_count
      FROM audit_scheduling
      LEFT JOIN audit_type ON audit_type.id = audit_scheduling.type
      WHERE ${dateFilter.getSQLWhereDateClause("audit_scheduling.created_at")}
        AND audit_scheduling.schedule_type = 'monitoring'
        AND audit_scheduling.deleted = 0
        AND audit_type.deleted = 0
        AND audit_scheduling.type = ?
      GROUP BY audit_scheduling.type, type_name
      ORDER BY type_name;
    `;

    const auditCountResult = await db.query(
      getAuditCountByMonitoringTypeQuery,
      [complianceMonitoringTypeId]
    );
    const rows = auditCountResult.rows || auditCountResult[0];

    const finalResponse = {
      data: [],
      labels: [],
      colors: [], // Array to store colors
    };

    for (const row of rows) {
      finalResponse.data.push(parseInt(row.audit_count));
      finalResponse.labels.push(row.type_name);
      // Fetch color for the underlying monitoring type
      const color = await getColorForCategory(
        row.type_id,
        "compliance_monitoring_type"
      );
      finalResponse.colors.push(color);
    }

    return finalResponse;
  } catch (error) {
    throw error;
  }
};

/** No. of Compliance Monitoring Audits by Responsible Person */
const complianceByResponsiblePerson = async (dateFilter) => {
  try {
    // query to get monitoring audits for responsible persons
    const getAuditQueryByResponsiblePerson = `
      SELECT CONCAT(users.name, ' ', users.surname) AS responsible_person
      FROM audit_scheduling asched
      LEFT JOIN users ON asched.owner = users.id
      WHERE ${dateFilter.getSQLWhereDateClause("asched.created_at")}
      AND asched.schedule_type = 'monitoring'
      AND asched.deleted = 0
      AND users.deleted = '0'
      GROUP BY responsible_person
      ORDER BY responsible_person;
    `;

    const [rows] = await db.query(getAuditQueryByResponsiblePerson);

    const responsiblePersonCounts = {};

    rows.forEach((row) => {
      const responsiblePerson = row?.responsible_person;
      if (responsiblePerson) {
        responsiblePersonCounts[responsiblePerson] = (responsiblePersonCounts[responsiblePerson] || 0) + 1;
      }
    });

    const responsiblePersons = Object.keys(responsiblePersonCounts).sort();
    const dataValues = responsiblePersons.map((person) => responsiblePersonCounts[person]);
    const totalAudits = dataValues.reduce((sum, count) => sum + count, 0);

    return {
      data: [
        {
          name: "Percentage of Compliance Monitoring Audits",
          type: "bar",
          data: dataValues,
          total: totalAudits,
        },
      ],
      xaxis: responsiblePersons,
    };
  } catch (error) {
    console.error("Error in complianceByResponsiblePerson:", error);
    throw error;
  }
};

/** ****************** HSE APPOINTMENT MODULE ************************** */
export const hseAppointmentDashboardData = async (req, res) => {
  const { all, filter } = req.query;

  if (!filter) {
    return sendResponse(res, 404, "Filter not found");
  }

  req.query.filter = JSON.parse(filter);

  // Parse date range
  const dateRange = req.query.filter?.date_type || "All-Time";
  const start_date = req.query.filter?.start_date;
  const end_date = req.query.filter?.end_date;
  const dateFilter = getDateRangeForSQL(dateRange, start_date, end_date);

  const hseAppointmentByBusinessChartData = await hseAppointmentByBusiness(
    dateFilter
  );
  const hseAppointmentByStatusChartData = await hseAppointmentByStatus(
    dateFilter
  );
  const hseAppointmentByAgeingUpcomingDateChartData =
    await hseAppointmentByAgeingBasedOnUpcomingAppointment(dateFilter);
  const hseAppointmentByAgeingOverdueChartData =
    await hseAppointmentByAgeingOverdueDays(dateFilter);
  const hseAppointmentByAppointeeChartData = await hseAppointmentsByAppointee(
    dateFilter
  );

  const finalData = {
    hseAppointmentByBusiness: hseAppointmentByBusinessChartData,
    hseAppointmentByStatus: hseAppointmentByStatusChartData,
    hseAppointmentByAgeingUpcomingDate:
      hseAppointmentByAgeingUpcomingDateChartData,
    hseAppointmentByAgeingOverdueDate: hseAppointmentByAgeingOverdueChartData,
    hseAppointmentByAppointee: hseAppointmentByAppointeeChartData,
  };

  return sendResponse(res, 200, finalData);
};

/** No. of HSE Appointments as per Business */
const hseAppointmentByBusiness = async (dateFilter) => {
  try {
    // query to get all appointments by Business
    const getHseAppointmentsByBusinessQuery = `
      SELECT
        o.id AS organization_id,
        o.name AS organization_name,
        COUNT(a.id) AS appointment_count
      FROM organization o
      LEFT JOIN hse_appointment a ON o.id = a.organization
      WHERE ${dateFilter.getSQLWhereDateClause("a.created_at")}
        AND a.deleted = 0
        AND o.deleted = 0
      GROUP BY o.id, o.name
      ORDER BY o.name;
    `;

    const [results] = await db.query(getHseAppointmentsByBusinessQuery);

    const chartXaxis = [];
    const chartSeries = [
      {
        name: "Number of HSE Appointments",
        type: "bar",
        data: [],
        total: 0,
      },
    ];

    if (results && results.length > 0) {
      results.forEach((row) => {
        chartXaxis.push(row.organization_name);
        chartSeries[0].data.push(row.appointment_count);
        chartSeries[0].total += row.appointment_count;
      });
    }

    const chartData = {
      data: chartSeries,
      xaxis: chartXaxis,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** HSE Appointments by Active/Inactive Status w.r.t effective and end date of appointment */
const hseAppointmentByStatus = async (dateFilter) => {
  try {
    // query to get all appointments by Business
    const getHseAppointmentsByStatusQuery = `
      SELECT effective_date, end_date
      FROM hse_appointment
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0;
    `;

    const todayDate = new Date();
    const [results] = await db.query(getHseAppointmentsByStatusQuery);

    let activeCount = 0;
    let inactiveCount = 0;

    if (results && results.length > 0) {
      results.forEach((row) => {
        const effectiveDate = new Date(row.effective_date);
        const endDate = row.end_date ? new Date(row.end_date) : null;

        // Check if today's date falls within the effective and end dates
        if (todayDate >= effectiveDate && (!endDate || todayDate <= endDate)) {
          activeCount++;
        } else {
          inactiveCount++;
        }
      });
    }

    const chartData = {
      data: [activeCount, inactiveCount],
      labels: ["Active", "Inactive"],
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** No.of HSE Appointments by Ageing: Upcoming & Overdue Appointments */
const hseAppointmentByAgeingBasedOnUpcomingAppointment = async (dateFilter) => {
  try {
    // query to get hse appointments by ageing
    const getHseAppointmentsByAgeingQuery = `
      SELECT effective_date
      FROM hse_appointment
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0;
    `;

    const [results] = await db.query(getHseAppointmentsByAgeingQuery);
    const todayDate = new Date();
    const overdueGroups = {
      Overdue: 0,
      "0-30 Days": 0,
      "31-60 Days": 0,
      "61-90 Days": 0,
      "90+ Days": 0,
      "No Date": 0,
    };

    if (results && results.length > 0) {
      results.forEach((row) => {
        const { effective_date } = row;

        if (!effective_date) {
          overdueGroups["No Date"]++;
        } else {
          const effectiveDate = new Date(effective_date);
          const timeDifference = effectiveDate.getTime() - todayDate.getTime();
          const daysDifference = Math.ceil(timeDifference / (1000 * 3600 * 24));

          if (daysDifference < 0) {
            overdueGroups["Overdue"]++;
          } else if (daysDifference >= 0 && daysDifference <= 30) {
            overdueGroups["0-30 Days"]++;
          } else if (daysDifference > 30 && daysDifference <= 60) {
            overdueGroups["31-60 Days"]++;
          } else if (daysDifference > 60 && daysDifference <= 90) {
            overdueGroups["61-90 Days"]++;
          } else if (daysDifference > 90) {
            overdueGroups["90+ Days"]++;
          }
        }
      });
    }

    const chartData = {
      data: [
        overdueGroups["Overdue"],
        overdueGroups["90+ Days"],
        overdueGroups["0-30 Days"],
        overdueGroups["31-60 Days"],
        overdueGroups["61-90 Days"],
        overdueGroups["No Date"],
      ],
      xaxis: [
        "Overdue",
        "90+ Days",
        "0-30 Days",
        "31-60 Days",
        "61-90 Days",
        "No Date",
      ],
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** No.of HSE Appointments by Ageing: Overdue Appointment Days */
const hseAppointmentByAgeingOverdueDays = async (dateFilter) => {
  try {
    // query to get overdue hse appointments
    const getOverdueHseAppointmentsQuery = `
      SELECT effective_date
      FROM hse_appointment
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
        AND effective_date < CURDATE()
        AND deleted = 0;
    `;

    const [results] = await db.query(getOverdueHseAppointmentsQuery);
    const todayDate = new Date();
    const overdueGroups = {
      "0-30 Days": 0,
      "31-60 Days": 0,
      "61-90 Days": 0,
      "90+ Days": 0,
    };

    if (results && results.length > 0) {
      results.forEach((row) => {
        const { effective_date } = row;

        if (effective_date) {
          const effectiveDate = new Date(effective_date);
          const timeDifference = todayDate.getTime() - effectiveDate.getTime();
          const daysDifference = Math.floor(
            timeDifference / (1000 * 3600 * 24)
          );

          if (daysDifference >= 0 && daysDifference <= 30) {
            overdueGroups["0-30 Days"]++;
          } else if (daysDifference > 30 && daysDifference <= 60) {
            overdueGroups["31-60 Days"]++;
          } else if (daysDifference > 60 && daysDifference <= 90) {
            overdueGroups["61-90 Days"]++;
          } else if (daysDifference > 90) {
            overdueGroups["90+ Days"]++;
          }
        }
      });
    }

    const chartData = {
      data: [
        overdueGroups["90+ Days"],
        overdueGroups["0-30 Days"],
        overdueGroups["31-60 Days"],
        overdueGroups["61-90 Days"],
      ],
      xaxis: ["90+ Days", "0-30 Days", "31-60 Days", "61-90 Days"],
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** No. of HSE Appointments for each Appointee (Appointed person) */
const hseAppointmentsByAppointee = async (dateFilter) => {
  try {
    // query to get appointed user from hse appointments
    const getAppointedUserQuery = `
      SELECT COUNT(ha.id) as total_appointments, ha.appointed_user_id, CONCAT(u.name, ' ', u.surname) AS appointed_name
      FROM hse_appointment ha
      LEFT JOIN users u ON u.id = ha.appointed_user_id
      WHERE ${dateFilter.getSQLWhereDateClause("ha.created_at")}
      AND ha.deleted = 0
      AND u.deleted = '0'
      GROUP BY ha.appointed_user_id, appointed_name
      ORDER BY appointed_name;
    `;

    const [results] = await db.query(getAppointedUserQuery);

    const chartXaxis = [];
    const chartSeries = [
      {
        name: "Number of HSE Appointments",
        type: "bar",
        data: [],
        total: 0,
      },
    ];

    if (results && results.length > 0) {
      results.forEach((row) => {
        chartXaxis.push(row.appointed_name);
        chartSeries[0].data.push(row.total_appointments);
        chartSeries[0].total += row.total_appointments;
      });
    }

    const chartData = {
      data: chartSeries,
      xaxis: chartXaxis,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** ****************** NCR & STOP CERTIFICATE MODULE ************************** */

export const ncrStopCertificateDashboardData = async (req, res) => {
  const { all, filter } = req.query;

  if (!filter) {
    return sendResponse(res, 404, "Filter not found");
  }

  req.query.filter = JSON.parse(filter);

  // Parse date range
  const dateRange = req.query.filter?.date_type || "All-Time";
  const start_date = req.query.filter?.start_date;
  const end_date = req.query.filter?.end_date;
  const dateFilter = getDateRangeForSQL(dateRange, start_date, end_date);

  const ncrByStatusChartData = await ncrByStatus(dateFilter);
  const ncrDigitalDataByStatusChartData = await ncrDigitalDataByStatus(dateFilter);
  const ncrByOriginChartData = await ncrByOrigin(dateFilter);
  const ncrByClassificationChartData = await ncrByClassification(dateFilter);
  const ncrByCategoryChartData = await ncrByCategory(dateFilter);
  const ncrByBusinessChartData = await ncrByBusiness(dateFilter);
  const stopCertByBusinessChartData = await stopCertByBusiness(dateFilter);
  const stopCertByIssueFromChartData = await stopCertByIssueFrom(dateFilter);
  const stopCertificateByStatusChartData = await stopCertificateByStatus(dateFilter);
  const stopCertByIssueToChartData = await stopCertByIssueTo(dateFilter);

  const finalData = {
    ncrByStatus: ncrByStatusChartData,
    ncrDigitalDataByStatus: ncrDigitalDataByStatusChartData,
    ncrByOrigin: ncrByOriginChartData,
    ncrByClassification: ncrByClassificationChartData,
    ncrByCategory: ncrByCategoryChartData,
    ncrByBusiness: ncrByBusinessChartData,
    stopCertByBusiness: stopCertByBusinessChartData,
    stopCertByIssueFrom: stopCertByIssueFromChartData,
    stopCertificateByStatus: stopCertificateByStatusChartData,
    stopCertByIssueTo: stopCertByIssueToChartData,
  };

  return sendResponse(res, 200, finalData);
};

/** NCR records by Status (Pie Chart) */
const ncrByStatus = async (dateFilter) => {
  try {
    // query to get NCR records by status
    const getNcrByStatusQuery = `
      SELECT signature_type, ddrm_id
      FROM ncr_recording
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0;
    `;

    const [results] = await db.query(getNcrByStatusQuery);

    let closedCount = 0;
    let inProgressCount = 0;

    if (results && results.length > 0) {
      results.forEach((row) => {
        const { signature_type, ddrm_id } = row;

        if (signature_type === 'Manual' && ddrm_id !== null && Number.isInteger(ddrm_id)) {
          closedCount++;
        } else {
          inProgressCount++;
        }
      });
    }

    const chartData = {
      data: [closedCount, inProgressCount],
      labels: ["Closed", "In Progress"],
    };

    return chartData;

  } catch (error) {
    throw error;
  }
};

/** NCR Digital Data by Status "In Progress Count", "Closed Count", "Current Month", "Current Month Closed"*/
const ncrDigitalDataByStatus = async (dateFilter) => {
  try {
    const todayDate = new Date();
    const currentYear = todayDate.getFullYear();
    const currentMonth = todayDate.getMonth();
    const firstDayOfMonth = new Date(currentYear, currentMonth, 1);
    const lastDayOfMonth = new Date(currentYear, currentMonth + 1, 0);

    // query to get NCR records
    const getNcrRecordsQuery = `
      SELECT signature_type, ddrm_id, created_at
      FROM ncr_recording
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0;
    `;

    const [results] = await db.query(getNcrRecordsQuery);

    let inProgressCount = 0;
    let closedCount = 0;
    let currentMonthCount = 0;
    let currentMonthClosedCount = 0;

    if (results && results.length > 0) {
      results.forEach((row) => {
        const { signature_type, ddrm_id, created_at } = row;
        const recordCreatedAt = new Date(created_at);

        const isClosed = signature_type === 'Manual' && ddrm_id !== null && Number.isInteger(ddrm_id);

        if (isClosed) {
          closedCount++;
          const isCurrentMonthClosed = recordCreatedAt >= firstDayOfMonth && recordCreatedAt <= lastDayOfMonth;
          if (isCurrentMonthClosed) {
            currentMonthClosedCount++;
          }
        } else {
          inProgressCount++;
        }

        const isCurrentMonth = recordCreatedAt >= firstDayOfMonth && recordCreatedAt <= lastDayOfMonth;
        if (isCurrentMonth) {
          currentMonthCount++;
        }
      });
    }

    return {
      "in_progress_count": inProgressCount,
      "closed_count": closedCount,
      "current_month": currentMonthCount,
      "current_month_closed": currentMonthClosedCount,
    };

  } catch (error) {
    throw error;
  }
};

/** NCR records based on Origin */
const ncrByOrigin = async (dateFilter) => {
  try {
    // query to get ncr records by origin
    const getNcrByOriginQuery = `
      SELECT origin_ncr
      FROM ncr_recording
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0;
    `;

    const [results] = await db.query(getNcrByOriginQuery);

    const originCounts = {
      "Legal Non-Compliance": 0,
      "System Non-Conformance": 0,
      "Customer Non-Compliance": 0,
      Others: 0,
    };

    if (results && results.length > 0) {
      results.forEach((row) => {
        const { origin_ncr } = row;

        if (origin_ncr === "Legal Non-Compliance") {
          originCounts["Legal Non-Compliance"]++;
        } else if (origin_ncr === "System Non-Conformance") {
          originCounts["System Non-Conformance"]++;
        } else if (origin_ncr === "Customer Non-Compliance") {
          originCounts["Customer Non-Compliance"]++;
        } else {
          originCounts["Others"]++;
        }
      });
    }

    const chartData = {
      data: [
        originCounts["Legal Non-Compliance"],
        originCounts["System Non-Conformance"],
        originCounts["Customer Non-Compliance"],
        originCounts["Others"],
      ],
      labels: [
        "Legal (Non-Compliance)",
        "System (Non-Conformance)",
        "Customer Non-Compliance",
        "Others",
      ],
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** NCR records based on Classification */
const ncrByClassification = async (dateFilter) => {
  try {
    // query to get ncr records by classification
    const getNcrByClassificationQuery = `
      SELECT classification, COUNT(id) AS count
      FROM ncr_recording
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
      GROUP BY classification;
    `;

    const [results] = await db.query(getNcrByClassificationQuery);

    const classificationCounts = {
      Major: 0,
      Minor: 0,
      Observation: 0,
    };

    if (results && results.length > 0) {
      results.forEach((row) => {
        if (classificationCounts.hasOwnProperty(row.classification)) {
          classificationCounts[row.classification] = row.count;
        }
      });
    }

    const chartData = {
      data: [
        {
          name: "Number of NCR Records",
          type: "bar",
          data: [
            classificationCounts["Major"],
            classificationCounts["Minor"],
            classificationCounts["Observation"],
          ],
          total: Object.values(classificationCounts).reduce(
            (sum, count) => sum + count,
            0
          ),
        },
      ],
      xaxis: ["Major", "Minor", "Observation"],
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** NCR records by Category (Pie Chart)*/
const ncrByCategory = async (dateFilter) => {
  try {
    // query to get ncr records based on categories
    const getNcrByCategoriesQuery = `
      SELECT ncr_categories
      FROM ncr_recording
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0;
    `;

    // get all categories
    const getCategoriesQuery = `
      SELECT id, name
      FROM incident_category
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
    `;

    const [ncrRecords] = await db.query(getNcrByCategoriesQuery);
    const [categoriesResult] = await db.query(getCategoriesQuery);

    const categoryCounts = {};
    categoriesResult.forEach((category) => {
      categoryCounts[category.name] = 0;
    });

    if (ncrRecords && ncrRecords.length > 0) {
      ncrRecords.forEach((record) => {
        try {
          const categoryIds = JSON.parse(record.ncr_categories);
          if (Array.isArray(categoryIds)) {
            categoryIds.forEach((categoryId) => {
              const category = categoriesResult.find(
                (cat) => cat.id === categoryId
              );
              if (category) {
                categoryCounts[category.name]++;
              }
            });
          }
        } catch (error) {
          console.error(
            "Error parsing ncr_categories:",
            error,
            record.ncr_categories
          );
          // Handle cases where ncr_categories might not be a valid JSON array
        }
      });
    }

    const chartData = {
      data: Object.values(categoryCounts),
      labels: Object.keys(categoryCounts),
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** NCR Records by Business Overview grouped by status */
const ncrByBusiness = async (dateFilter) => {
  try {
    // query to get count of ncr records based on organizations and status
    const getNcrByBusinessQuery = `
      SELECT
        o.name AS organization_name,
        ncr.signature_type,
        ncr.ddrm_id
      FROM organization o
      LEFT JOIN ncr_recording ncr ON o.id = ncr.organization
      WHERE ${dateFilter.getSQLWhereDateClause("ncr.created_at")}
        AND ncr.deleted = 0
        AND o.deleted = 0;
    `;

    const [results] = await db.query(getNcrByBusinessQuery);

    const organizationStatusCounts = {};

    if (results && results.length > 0) {
      results.forEach((row) => {
        const { organization_name, signature_type, ddrm_id } = row;
        const isClosed = signature_type === 'Manual' && ddrm_id !== null && Number.isInteger(ddrm_id);
        const status = isClosed ? 'Closed' : 'In Progress';

        if (!organizationStatusCounts[organization_name]) {
          organizationStatusCounts[organization_name] = {
            'In Progress': 0,
            'Closed': 0,
          };
        }
        organizationStatusCounts[organization_name][status]++;
      });
    }

    const chartXaxis = Object.keys(organizationStatusCounts).sort(); // Sort organizations alphabetically
    const inProgressData = [];
    const closedData = [];

    chartXaxis.forEach((orgName) => {
      inProgressData.push(organizationStatusCounts[orgName]['In Progress'] || 0);
      closedData.push(organizationStatusCounts[orgName]['Closed'] || 0);
    });

    const chartData = {
      data: [
        {
          name: 'In Progress',
          type: 'bar',
          data: inProgressData,
          total: inProgressData.reduce((sum, count) => sum + count, 0),
        },
        {
          name: 'Closed',
          type: 'bar',
          data: closedData,
          total: closedData.reduce((sum, count) => sum + count, 0),
        },
      ],
      xaxis: chartXaxis,
    };

    return chartData;

  } catch (error) {
    throw error;
  }
};

/** Stop Certificate records by Business */
const stopCertByBusiness = async (dateFilter) => {
  try {
    // get stop certificate records by Business Organizations
    const getStopCertQuery = `
      SELECT
        o.id AS organization_id,
        o.name AS organization_name,
        COUNT(ncr.id) AS stop_cert_count
      FROM organization o
      LEFT JOIN ncr_recording ncr ON o.id = ncr.organization
      WHERE ${dateFilter.getSQLWhereDateClause("ncr.created_at")}
        AND ncr.deleted = 0
        AND o.deleted = 0
        AND ncr.stop_certificate_used = 1
      GROUP BY o.id, o.name
      ORDER BY o.name;
    `;

    const [results] = await db.query(getStopCertQuery);

    const chartXaxis = [];
    const chartSeries = [
      {
        name: "Number of Stop Certificates Used",
        type: "bar",
        data: [],
        total: 0,
      },
    ];

    if (results && results.length > 0) {
      results.forEach((row) => {
        chartXaxis.push(row.organization_name);
        chartSeries[0].data.push(row.stop_cert_count);
        chartSeries[0].total += row.stop_cert_count;
      });
    }

    const chartData = {
      data: chartSeries,
      xaxis: chartXaxis,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** Stop Certificate records based on issued from */
const stopCertByIssueFrom = async (dateFilter) => {
  try {
    // query to get stop certificates based on issued from
    const getStopCertQuery = `
      SELECT
        COUNT(ncr.id) AS stop_cert_count,
        ncr.issuer_user_id,
        CONCAT(u.name, ' ', u.surname) AS issuer_name
      FROM ncr_recording ncr
      LEFT JOIN users u ON ncr.issuer_user_id = u.id
      WHERE ${dateFilter.getSQLWhereDateClause("ncr.created_at")}
        AND ncr.deleted = 0
        AND ncr.stop_certificate_used = 1
        AND u.deleted = '0'
      GROUP BY ncr.issuer_user_id, issuer_name
      ORDER BY issuer_name;
    `;

    const [results] = await db.query(getStopCertQuery);

    const chartXaxis = [];
    const chartSeries = [
      {
        name: "Number of Stop Certificates Issued",
        type: "bar",
        data: [],
        total: 0,
      },
    ];

    if (results && results.length > 0) {
      results.forEach((row) => {
        chartXaxis.push(row.issuer_name);
        chartSeries[0].data.push(row.stop_cert_count);
        chartSeries[0].total += row.stop_cert_count;
      });
    }

    const chartData = {
      data: chartSeries,
      xaxis: chartXaxis,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** Stop Certificate records based on Status (Closed/Open) */
const stopCertificateByStatus = async (dateFilter) => {
  try {
    const getStopCertQuery = `
      SELECT
        ncr.issuer_date,
        ncr.date_lifted
      FROM ncr_recording ncr
      WHERE ${dateFilter.getSQLWhereDateClause("ncr.created_at")}
        AND ncr.deleted = 0
        AND ncr.stop_certificate_used = 1
        GROUP BY ncr.issuer_date, ncr.date_lifted;
    `;

    const [ncrRecords] = await db.query(getStopCertQuery);

    const currentDate = new Date();
    let openCount = 0;
    let closedCount = 0;

    ncrRecords.forEach((ncr) => {
      const issuerDate = ncr.issuer_date ? new Date(ncr.issuer_date) : null;
      const dateLifted = ncr.date_lifted ? new Date(ncr.date_lifted) : null;

      if (issuerDate) {
        if (!dateLifted || currentDate >= issuerDate && currentDate < dateLifted) {
          openCount++;
        } else if (dateLifted && currentDate >= dateLifted) {
          closedCount++;
        } else if (dateLifted === null && currentDate >= issuerDate) {
          openCount++;
        } else {
          closedCount++; // Consider as closed if issuer date is in the future
        }
      } else {
        closedCount++; // Consider as closed if no issuer date
      }
    });

    const data = [openCount, closedCount];
    const labels = ["Open", "Closed"];

    return {
      data: data,
      labels: labels,
    };
  } catch (error) {
    throw error;
  }
};

/** Stop Certificate records based on issued to */
const stopCertByIssueTo = async (dateFilter) => {
  try {
    // query to get stop certificates based on issued from
    const getStopCertQuery = `
      SELECT
        COUNT(ncr.id) AS stop_cert_count,
        ncr.recipient_user_id,
        CONCAT(u.name, ' ', u.surname) AS recipient_name
      FROM ncr_recording ncr
      LEFT JOIN users u ON ncr.recipient_user_id = u.id
      WHERE ${dateFilter.getSQLWhereDateClause("ncr.created_at")}
        AND ncr.deleted = 0
        AND ncr.stop_certificate_used = 1
        AND u.deleted = '0'
      GROUP BY ncr.recipient_user_id, recipient_name
      ORDER BY recipient_name;
    `;

    const [results] = await db.query(getStopCertQuery);

    const chartXaxis = [];
    const chartSeries = [
      {
        name: "Number of Stop Certificates Issued",
        type: "bar",
        data: [],
        total: 0,
      },
    ];

    if (results && results.length > 0) {
      results.forEach((row) => {
        chartXaxis.push(row.recipient_name);
        chartSeries[0].data.push(row.stop_cert_count);
        chartSeries[0].total += row.stop_cert_count;
      });
    }

    const chartData = {
      data: chartSeries,
      xaxis: chartXaxis,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** ****************** INCIDENT MODULE ************************** */

export const incidentDashboardData = async (req, res) => {
  const { all, filter } = req.query;

  if (!filter) {
    return sendResponse(res, 404, "Filter not found");
  }

  req.query.filter = JSON.parse(filter);

  // Parse date range
  const dateRange = req.query.filter?.date_type || "All-Time";
  const start_date = req.query.filter?.start_date;
  const end_date = req.query.filter?.end_date;
  const dateFilter = getDateRangeForSQL(dateRange, start_date, end_date);

  const incidentOccurrenceByBusinessChartData =
    await incidentOccurrenceByBusiness(dateFilter);
  const incidentOccurrenceByStatusChartData = await incidentOccurrenceByStatus(
    dateFilter
  );
  const incidentOccurrenceByTypeChartData = await incidentOccurrenceByType(
    dateFilter
  );
  const incidentOccurrenceByYearMonthChartData =
    await incidentOccurrenceByYearMonth(dateFilter);
  const incidentOccurrenceByReportableToChartData =
    await incidentOccurrenceByReportableTo(dateFilter);
  const incidentOccurrenceByDayOfWeekChartData =
    await incidentOccurrenceByDayOfWeek(dateFilter);
  const incidentOccurrenceByTimeAtHourChartData =
    await incidentOccurrenceByTimeAtHour(dateFilter);
  const incidentOccurrencesByPersonsInvolvedChartData =
    await incidentOccurrencesByPersonsInvolved(dateFilter);
  const incidentOccurrencesByPersonsInvolvedTypeChartData =
    await incidentOccurrencesByPersonsInvolvedType(dateFilter);
  const incidentOccurrencesByInjuredExposedTypeChartData =
    await incidentOccurrencesByInjuredExposedType(dateFilter);
  const iodByClassificationChartData = await iodByClassification(dateFilter);
  const personExperienceByIODChartData = await personExperienceByIOD(
    dateFilter
  );
  const iodByPersonsChartData = await iodByPersons(dateFilter);
  const iodByCategoryChartData = await iodByCategory(dateFilter);
  const iodByTypeChartData = await iodByType(dateFilter);
  const occupationalDiseasesByCategoryChartData =
    await occupationalDiseasesByCategory(dateFilter);
  const occupationalDiseasesByDescriptionChartData =
    await occupationalDiseasesByDescription(dateFilter);
  const occupationalDiseasesBySourceOfExposureChartData =
    await occupationalDiseasesBySourceOfExposure(dateFilter);
  const occupationalDiseasesByClinicChartData =
    await occupationalDiseasesByClinic(dateFilter);
  const assetIncidentByTypeChartData = await assetIncidentByType(dateFilter);
  const assetIncidentByOwnerChartData = await assetIncidentByOwner(dateFilter);
  const assetIncidentBySeverityChartData = await assetIncidentBySeverity(
    dateFilter
  );
  const vehicleIncidentByTypeChartData = await vehicleIncidentByType(
    dateFilter
  );
  const vehicleIncidentBySeverityChartData = await vehicleIncidentBySeverity(
    dateFilter
  );
  const vehicleIncidentByOwnerChartData = await vehicleIncidentByOwner(
    dateFilter
  );
  const vehicleIncidentByDamagedPartsChartData =
    await vehicleIncidentByDamagedParts(dateFilter);
  const nearMissByOrganizationChartData = await nearMissByOrganization(
    dateFilter
  );
  const nearMissTrendsChartData = await nearMissTrends(dateFilter);
  const nearMissByCategoryAndSeverityChartData =
    await nearMissByCategoryAndSeverity(dateFilter);
  const nearMissByProbabilityChartData = await nearMissByProbability(
    dateFilter
  );
  const securityIncidentByTypeChartData = await securityIncidentByType(
    dateFilter
  );
  const securityIncidentBySuspectTypeChartData =
    await securityIncidentBySuspectType(dateFilter);
  const securityIncidentOccurrenceAtHourChartData =
    await securityIncidentOccurrenceAtHour(dateFilter);
  const securityIncidentOccurrenceByDayOfWeekChartData =
    await securityIncidentOccurrenceByDayOfWeek(dateFilter);
  const costBreakdown = await getCostBreakDown(dateFilter);
  const estimatedVsActualCostByYear = await getEstimatedVsActualCostByYear(
    dateFilter
  );
  const actualCostByType = await getActualCostByType(dateFilter);
  const actualCostByProperty = await getActualCostByProperty(dateFilter);
  const environmentOccurrenceByConsequences =
    await getEnvironmentOccurrenceByConsequences(dateFilter);
  const { emissionResult, leakageResult, wasteResult } =
    await getEnvironmentOccurrenceByLeakageAndEmission(dateFilter);

  const finalData = {
    incidentOccurrenceByBusiness: incidentOccurrenceByBusinessChartData,
    incidentOccurrenceByStatus: incidentOccurrenceByStatusChartData,
    incidentOccurrenceByType: incidentOccurrenceByTypeChartData,
    incidentOccurrenceByYearMonth: incidentOccurrenceByYearMonthChartData,
    incidentOccurrenceByReportableTo: incidentOccurrenceByReportableToChartData,
    incidentOccurrenceByDayOfWeek: incidentOccurrenceByDayOfWeekChartData,
    incidentOccurrenceByTimeAtHour: incidentOccurrenceByTimeAtHourChartData,
    incidentOccurrencesByPersonsInvolved:
      incidentOccurrencesByPersonsInvolvedChartData,
    incidentOccurrencesByPersonsInvolvedType:
      incidentOccurrencesByPersonsInvolvedTypeChartData,
    incidentOccurrencesByInjuredExposedType:
      incidentOccurrencesByInjuredExposedTypeChartData,
    iodByClassification: iodByClassificationChartData,
    personExperienceByIOD: personExperienceByIODChartData,
    iodByPersons: iodByPersonsChartData,
    iodByCategory: iodByCategoryChartData,
    iodByType: iodByTypeChartData,
    occupationalDiseasesByCategory: occupationalDiseasesByCategoryChartData,
    occupationalDiseasesByDescription:
      occupationalDiseasesByDescriptionChartData,
    occupationalDiseasesBySourceOfExposure:
      occupationalDiseasesBySourceOfExposureChartData,
    occupationalDiseasesByClinic: occupationalDiseasesByClinicChartData,
    assetIncidentByType: assetIncidentByTypeChartData,
    assetIncidentByOwner: assetIncidentByOwnerChartData,
    assetIncidentBySeverity: assetIncidentBySeverityChartData,
    vehicleIncidentByType: vehicleIncidentByTypeChartData,
    vehicleIncidentBySeverity: vehicleIncidentBySeverityChartData,
    vehicleIncidentByOwner: vehicleIncidentByOwnerChartData,
    vehicleIncidentByDamagedParts: vehicleIncidentByDamagedPartsChartData,
    nearMissByOrganization: nearMissByOrganizationChartData,
    nearMissTrends: nearMissTrendsChartData,
    nearMissByCategoryAndSeverity: nearMissByCategoryAndSeverityChartData,
    nearMissByProbability: nearMissByProbabilityChartData,
    securityIncidentByType: securityIncidentByTypeChartData,
    securityIncidentCommittedBy: securityIncidentBySuspectTypeChartData,
    securityIncidentOccurrenceAtHour: securityIncidentOccurrenceAtHourChartData,
    securityIncidentOccurrenceByDayOfWeek:
      securityIncidentOccurrenceByDayOfWeekChartData,
    costBreakdown,
    estimatedVsActualCostByYear,
    actualCostByType,
    actualCostByProperty,
    environmentOccurrenceByConsequences,
    environmentOccurrenceByLeakage: leakageResult,
    environmentOccurrenceByEmission: emissionResult,
    environmentOccurrenceByWasteClassification: wasteResult,
  };

  return sendResponse(res, 200, finalData);
};

/** No. of Incident Occurrences by Business Organizations */
const incidentOccurrenceByBusiness = async (dateFilter) => {
  try {
    // Query to get all organizations
    const getOrganizationQuery = `
      SELECT
        id, name
      FROM organization
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
    `;

    const [organizationsResult] = await db.query(getOrganizationQuery);
    const organizations = organizationsResult || [];

    const incidentCounts = [];
    const organizationNames = [];
    let totalIncidents = 0;

    for (const organization of organizations) {
      // query to get all incidents by Business
      const getIncidentQuery = `
        SELECT
          COUNT(id) AS count
        FROM incident
        WHERE organization = ?
        AND ${dateFilter.getSQLWhereDateClause("created_at")}
        AND deleted = 0
        GROUP by id;
      `;

      const [incidentResult] = await db.query(getIncidentQuery, [
        organization.id,
      ]);
      const count = incidentResult?.[0]?.count || 0;

      incidentCounts.push(count);
      organizationNames.push(organization.name);
      totalIncidents += count;
    }

    const chartData = {
      data: [
        {
          name: "Number of Incidents",
          type: "bar",
          data: incidentCounts,
          total: totalIncidents,
        },
      ],
      xaxis: organizationNames,
    };

    return chartData;
  } catch (error) {
    throw error;
  }
};

/** No. of Incident Occurrences by Occurrence Status */
const incidentOccurrenceByStatus = async (dateFilter) => {
  try {
    const statuses = ["Created", "In Progress", "Investigation Completed"];
    const statusCounts = [0, 0, 0];

    let incidentDataFetchQuery = `SELECT
      incident.id,
      incident.incident_category,
      incident.created_at
      FROM incident
      WHERE incident.deleted = 0`;

    if (dateFilter?.startDate && dateFilter?.endDate) {
      incidentDataFetchQuery += ` AND incident.created_at >= '${dateFilter.startDate}' AND incident.created_at <= '${dateFilter.endDate}'`;
    } else if (dateFilter?.startDate) {
      incidentDataFetchQuery += ` AND incident.created_at >= '${dateFilter.startDate}'`;
    } else if (dateFilter?.endDate) {
      incidentDataFetchQuery += ` AND incident.created_at <= '${dateFilter.endDate}'`;
    }

    const [incidents] = await db.query(incidentDataFetchQuery);

    for (const incident of incidents) {
      let currentStatus = "Created"; // Default status
      let atLeastOneTechnicalFormCompleted = false;

      try {
        incident.incident_category = JSON.parse(
          incident?.incident_category || ""
        );
      } catch (error) {
        console.error("Error parsing incident_category:", error, incident);
        continue;
      }

      if (incident.incident_category && incident.incident_category.length > 0) {
        const incidentCategoryQuery = `SELECT id, name FROM incident_category WHERE deleted = 0 AND id IN (${incident.incident_category})`;
        const [incidentCategoryList] = await db.query(incidentCategoryQuery);

        for (const category of incidentCategoryList) {
          if (incidentCategoryConstant[category.name]) {
            const incidentCategoryFormList =
              incidentCategoryConstant[category.name];
            for (const formConfig of incidentCategoryFormList) {
              if (formConfig.tableName) {
                const FormCheckQuery = `SELECT id, save_type FROM ${formConfig.tableName} WHERE incident_id = ${incident.id} AND deleted = 0 ORDER BY id DESC LIMIT 1`;
                const [formRecord] = await db.query(FormCheckQuery);

                if (formRecord.length > 0) {
                  if (formRecord[0].save_type === null) {
                    currentStatus = "In Progress";
                    break; // If any form has null save_type, it's In Progress
                  } else if (formRecord[0].save_type === "Completed") {
                    atLeastOneTechnicalFormCompleted = true;
                  }
                }
                // If formRecord.length === 0, the status remains 'Created' (default)
              }
            }
          }
          if (currentStatus === "In Progress") break; // Optimization: If status is already 'In Progress', no need to check other categories
        }

        if (
          currentStatus !== "In Progress" &&
          atLeastOneTechnicalFormCompleted
        ) {
          currentStatus = "In Progress"; // Set to In Progress if at least one form is completed and not already set by null save_type
        }
      }

      // Check for Investigation Completed status
      const investigationCheckQuery = `SELECT id FROM investigation WHERE incident_id = ${incident.id} AND is_done = 1 AND deleted = 0`;
      const [investigationRecord] = await db.query(investigationCheckQuery);

      if (investigationRecord.length > 0) {
        currentStatus = "Investigation Completed";
      }

      const statusIndex = statuses.indexOf(currentStatus);
      if (statusIndex !== -1) {
        statusCounts[statusIndex]++;
      }
    }

    return {
      data: statusCounts,
      labels: statuses,
    };
  } catch (error) {
    console.error("Error in incidentOccurrenceByStatus:", error);
    throw error;
  }
};

/** No. of Incident Occurrences by Nature (Type/Category) */
const incidentOccurrenceByType = async (dateFilter) => {
  try {
    // query to get incident categories
    const incidentCategoryQuery = `
      SELECT id, name
      FROM incident_category
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0;
    `;
    const [incidentCategories] = await db.query(incidentCategoryQuery);

    // Create a map to store category names by ID and initialize counts
    const categoryNameMap = {};
    const categoryCounts = {};
    incidentCategories.forEach((category) => {
      categoryNameMap[category.id] = category.name;
      categoryCounts[category.id] = 0;
    });

    // query to get incident
    const incidentDataFetchQuery = `SELECT
        incident.id,
        incident.incident_category
        FROM incident
        WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
        AND deleted = 0
      `;
    const [incidents] = await db.query(incidentDataFetchQuery);

    // Count incidents per category
    for (const incident of incidents) {
      try {
        const categoryIds = JSON.parse(incident?.incident_category || "");
        if (Array.isArray(categoryIds)) {
          categoryIds.forEach((categoryId) => {
            if (categoryCounts.hasOwnProperty(categoryId)) {
              categoryCounts[categoryId]++;
            }
          });
        }
      } catch (error) {
        console.error("Error parsing incident_category:", error, incident);
        // Handle error appropriately, maybe skip this incident or log
      }
    }

    // Prepare the response data and labels
    const data = Object.values(categoryCounts);
    const labels = Object.keys(categoryCounts).map((id) => categoryNameMap[id]);

    return {
      data,
      labels,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Incident Occurrences by Year/Month */
const incidentOccurrenceByYearMonth = async (dateFilter) => {
  try {
    // query to get incident
    const incidentDataFetchQuery = `
      SELECT
        id,
        incident_actual_date
        FROM incident
        WHERE ${dateFilter.getSQLWhereDateClause("incident_actual_date")}
        AND deleted = 0
      `;
    const [incidents] = await db.query(incidentDataFetchQuery);

    const monthlyCounts = {};

    for (const incident of incidents) {
      const actualDate = incident.incident_actual_date;
      if (actualDate) {
        const date = new Date(actualDate);
        const year = date.getFullYear();
        // const month = date.getMonth(); // 0-indexed
        const monthShort = new Intl.DateTimeFormat("en-US", {
          month: "short",
        }).format(date);
        const yearFull = year;
        const yearMonth = `${monthShort}-${yearFull}`;

        monthlyCounts[yearMonth] = (monthlyCounts[yearMonth] || 0) + 1;
      }
    }

    const sortedMonths = Object.keys(monthlyCounts).sort((a, b) => {
      const [monthA, yearA] = a.split("-");
      const [monthB, yearB] = b.split("-");

      const dateA = new Date(`${monthA} 01, ${yearA}`);
      const dateB = new Date(`${monthB} 01, ${yearB}`);

      return dateA - dateB;
    });

    const data = sortedMonths.map((month) => monthlyCounts[month]);
    const xaxis = sortedMonths;

    return {
      data,
      xaxis,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Incident Occurrences by Reportable To (Organization) */
const incidentOccurrenceByReportableTo = async (dateFilter) => {
  try {
    // query to get incident Occurrences
    const getIncidentNotifyPersonQuery = `
      SELECT notification
      FROM incident
      WHERE ${dateFilter.getSQLWhereDateClause("incident_actual_date")}
      AND deleted = 0;
    `;
    const [incidents] = await db.query(getIncidentNotifyPersonQuery);

    const notifiedUserCounts = {};
    const userIdsToFetch = new Set();

    for (const incident of incidents) {
      try {
        const notificationIds = JSON.parse(incident?.notification || "");
        if (Array.isArray(notificationIds)) {
          notificationIds.forEach((userId) => {
            userIdsToFetch.add(userId);
            notifiedUserCounts[userId] = (notifiedUserCounts[userId] || 0) + 1;
          });
        }
      } catch (error) {
        console.error("Error parsing notification:", error, incident);
        // Handle error appropriately
      }
    }

    const userIdsArray = Array.from(userIdsToFetch);
    let userNamesMap = {};

    if (userIdsArray.length > 0) {
      const getUsersQuery = `
        SELECT id, CONCAT(name, ' ', surname) as notified_person
        FROM users
        WHERE deleted = '0' AND id IN (?)
      `;
      const [users] = await db.query(getUsersQuery, [userIdsArray]);
      users.forEach((user) => {
        userNamesMap[user.id] = user.notified_person;
      });
    }

    const data = [];
    const labels = [];

    for (const userId in notifiedUserCounts) {
      if (userNamesMap[userId]) {
        data.push(notifiedUserCounts[userId]);
        labels.push(userNamesMap[userId]);
      }
    }

    return {
      data,
      labels,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Incident Occurrences by Final Level (Different Levels) */
// ! PENDING
const incidentOccurrenceByFinalLevel = async (dateFilter) => {
  try {

  } catch (error) {
    throw error;
  }
}

/** No. of Incident Occurrences by Day of Week (e.g. Sun, Mon, ...) */
const incidentOccurrenceByDayOfWeek = async (dateFilter) => {
  try {
    // query to get incident
    const incidentDataFetchQuery = `
      SELECT
        id,
        incident_actual_date
        FROM incident
        WHERE ${dateFilter.getSQLWhereDateClause("incident_actual_date")}
        AND deleted = 0
      `;
    const [incidents] = await db.query(incidentDataFetchQuery);

    const dayOfWeekCounts = {
      Sunday: 0,
      Monday: 0,
      Tuesday: 0,
      Wednesday: 0,
      Thursday: 0,
      Friday: 0,
      Saturday: 0,
    };

    for (const incident of incidents) {
      const actualDate = incident.incident_actual_date;
      if (actualDate) {
        const date = new Date(actualDate);
        const dayIndex = date.getDay(); // 0 for Sunday, 1 for Monday, etc.
        const daysOfWeek = [
          "Sunday",
          "Monday",
          "Tuesday",
          "Wednesday",
          "Thursday",
          "Friday",
          "Saturday",
        ];
        const dayName = daysOfWeek[dayIndex];
        dayOfWeekCounts[dayName]++;
      }
    }

    const data = Object.values(dayOfWeekCounts);
    const xaxis = Object.keys(dayOfWeekCounts);

    return {
      data,
      xaxis,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Incident Occurrences by Time (at hour) */
const incidentOccurrenceByTimeAtHour = async (dateFilter) => {
  try {
    // query to get incident
    const incidentDataFetchQuery = `
      SELECT
        id,
        incident_actual_time
        FROM incident
        WHERE ${dateFilter.getSQLWhereDateClause("incident_actual_date")}
        AND deleted = 0
      `;
    const [incidents] = await db.query(incidentDataFetchQuery);

    const hourlyCounts = {};
    for (let i = 0; i < 24; i++) {
      hourlyCounts[i] = 0;
    }

    for (const incident of incidents) {
      const actualTime = incident.incident_actual_time;
      if (actualTime) {
        const date = new Date(`2000-01-01 ${actualTime}`); // Use a dummy date to parse the time
        const hour = date.getHours();
        if (hourlyCounts.hasOwnProperty(hour)) {
          hourlyCounts[hour]++;
        }
      }
    }

    const data = Object.values(hourlyCounts);
    const labels = Object.keys(hourlyCounts).map((hour) => parseInt(hour)); // Ensure labels are numbers

    return {
      data,
      labels,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Incident Occurrences by Persons Involved */
const incidentOccurrencesByPersonsInvolved = async (dateFilter) => {
  try {
    // query to get incident
    const incidentDataFetchQuery = `
      SELECT
        id,
        incident_persons_involved
        FROM incident
        WHERE ${dateFilter.getSQLWhereDateClause("incident_actual_date")}
        AND deleted = 0
      `;
    const [incidents] = await db.query(incidentDataFetchQuery);

    const userIds = new Set();
    const contractorIds = new Set();
    const personCounts = {};

    for (const incident of incidents) {
      const personsInvolved = incident?.incident_persons_involved;
      if (Array.isArray(personsInvolved)) {
        personsInvolved.forEach((person) => {
          if (person?.person_type == "Internal" && person?.user_id) {
            userIds.add(person.user_id);
          } else if (
            person?.person_type == "External" &&
            person?.is_contractor == "1" &&
            person?.contractor_id
          ) {
            contractorIds.add(person.contractor_id);
          } else if (person?.person_type == "Both" && person?.user_id) {
            userIds.add(person.user_id);
          } else if (
            person?.person_type == "Both" &&
            person?.is_contractor == "1" &&
            person?.contractor_id
          ) {
            contractorIds.add(person.contractor_id);
          }
        });
      }
    }

    const usersMap = {};
    if (userIds.size > 0) {
      const usersQuery = `
        SELECT id, CONCAT(name, ' ', surname) as notified_person
        FROM users
        WHERE deleted = '0' AND id IN (?)
      `;
      const [usersResult] = await db.query(usersQuery, [Array.from(userIds)]);
      usersResult.forEach((user) => {
        usersMap[user.id] = user.notified_person;
      });
    }

    const contractorsMap = {};
    if (contractorIds.size > 0) {
      const contractorQuery = `
        SELECT id, contractor_name as contractor_name
        FROM contractor_registration
        WHERE deleted = 0 AND id IN (?)
      `;
      const [contractorsResult] = await db.query(contractorQuery, [
        Array.from(contractorIds),
      ]);
      contractorsResult.forEach((contractor) => {
        contractorsMap[contractor.id] = contractor.contractor_name;
      });
    }

    for (const incident of incidents) {
      const personsInvolved = incident?.incident_persons_involved;
      if (Array.isArray(personsInvolved)) {
        personsInvolved.forEach((person) => {
          let personName = null;
          if (
            person?.person_type == "Internal" &&
            person?.user_id &&
            usersMap[person.user_id]
          ) {
            personName = usersMap[person.user_id];
          } else if (
            person?.person_type == "External" &&
            person?.is_contractor == "0" &&
            person?.external_person
          ) {
            personName = person.external_person;
          } else if (
            person?.person_type == "External" &&
            person?.is_contractor == "1" &&
            person?.contractor_id &&
            contractorsMap[person.contractor_id]
          ) {
            personName = contractorsMap[person.contractor_id];
          } else if (person?.person_type == "Both") {
            const names = [];
            if (person?.user_id && usersMap[person.user_id]) {
              names.push(usersMap[person.user_id]);
            }
            if (person?.is_contractor == "0" && person?.external_person) {
              names.push(person.external_person);
            } else if (
              person?.is_contractor == "1" &&
              person?.contractor_id &&
              contractorsMap[person.contractor_id]
            ) {
              names.push(contractorsMap[person.contractor_id]);
            }
            personName = names.join(" and ");
          }

          if (personName) {
            personCounts[personName] = (personCounts[personName] || 0) + 1;
          }
        });
      }
    }

    const data = [
      {
        name: "Number of Incidents",
        type: "bar",
        data: Object.values(personCounts),
        total: incidents.length,
      },
    ];
    const xaxis = Object.keys(personCounts);

    return {
      data,
      xaxis,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Incident Occurrences by Persons Involved Type (Person's Role at the time of Incident) */
const incidentOccurrencesByPersonsInvolvedType = async (dateFilter) => {
  try {
    // query to get incident
    const incidentDataFetchQuery = `
      SELECT
        id,
        incident_persons_involved
        FROM incident
        WHERE ${dateFilter.getSQLWhereDateClause("incident_actual_date")}
        AND deleted = 0
      `;
    const [incidents] = await db.query(incidentDataFetchQuery);

    const roleCounts = {};

    for (const incident of incidents) {
      const personsInvolved = incident?.incident_persons_involved;
      if (Array.isArray(personsInvolved)) {
        personsInvolved.forEach((person) => {
          let roleName = null;
          if (
            person?.person_type == "Internal" &&
            person?.internal_incident_role_name
          ) {
            roleName = person.internal_incident_role_name;
          } else if (
            person?.person_type == "External" &&
            person?.external_incident_role_name
          ) {
            roleName = person.external_incident_role_name;
          } else if (person?.person_type == "Both") {
            const roles = [];
            if (person?.internal_incident_role_name) {
              roles.push(person.internal_incident_role_name);
            }
            if (person?.external_incident_role_name) {
              roles.push(person.external_incident_role_name);
            }
            roleName = roles.join(" and ");
          }

          if (roleName) {
            roleCounts[roleName] = (roleCounts[roleName] || 0) + 1;
          }
        });
      }
    }

    const data = [
      {
        name: "Number of Incidents",
        type: "bar",
        data: Object.values(roleCounts),
        total: incidents.length,
      },
    ];
    const xaxis = Object.keys(roleCounts);

    return {
      data,
      xaxis,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Incident Occurrences by Persons Injured/Exposed */
const incidentOccurrencesByInjuredExposedType = async (dateFilter) => {
  try {
    // query to get injured persons
    const injuredPersonsQuery = `
      SELECT injured_persons
      FROM injury_on_duty
      WHERE ${dateFilter.getSQLWhereDateClause("incident_date")}
      AND deleted = 0;
    `;
    const [injuredRecords] = await db.query(injuredPersonsQuery);

    const injuredPersonCounts = {};
    const userIds = new Set();

    for (const record of injuredRecords) {
      let injuredPersonsData = record?.injured_persons;
      try {
        let injuredPersons = [];
        if (injuredPersonsData && injuredPersonsData !== "null") {
          injuredPersons = JSON.parse(injuredPersonsData);
        }
        if (Array.isArray(injuredPersons)) {
          injuredPersons.forEach((person) => {
            if (person?.user_id) {
              userIds.add(person.user_id);
            }
          });
        }
      } catch (error) {
        console.error("Error parsing injured_persons:", error, record);
        // Handle error appropriately
      }
    }

    const usersMap = {};
    if (userIds.size > 0) {
      const usersQuery = `
        SELECT id, CONCAT(name, ' ', surname) as notified_person
        FROM users
        WHERE deleted = '0' AND id IN (?)
      `;
      const [usersResult] = await db.query(usersQuery, [Array.from(userIds)]);
      usersResult.forEach((user) => {
        usersMap[user.id] = user.notified_person;
      });
    }

    for (const record of injuredRecords) {
      let injuredPersonsData = record?.injured_persons;
      try {
        let injuredPersons = [];
        if (injuredPersonsData && injuredPersonsData !== "null") {
          injuredPersons = JSON.parse(injuredPersonsData);
        }
        if (Array.isArray(injuredPersons)) {
          injuredPersons.forEach((person) => {
            if (person?.user_id && usersMap[person.user_id]) {
              const userName = usersMap[person.user_id];
              injuredPersonCounts[userName] =
                (injuredPersonCounts[userName] || 0) + 1;
            }
          });
        }
      } catch (error) {
        console.error("Error processing injured_persons:", error, record);
      }
    }

    const data = [
      {
        name: "Number of Incidents",
        type: "bar",
        data: Object.values(injuredPersonCounts),
        total: injuredRecords.length,
      },
    ];
    const xaxis = Object.keys(injuredPersonCounts);

    return {
      data,
      xaxis,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Employees/Non-Employees by IOD(Injured On Duty) */
// ! PENDING
const employeesByIOD = async (dateFilter) => {
  try {
    // query to get employees by IOD
    const employeesByIODQuery = `
      SELECT 
    `;
  } catch (error) {
    throw error;
  }
};

/** No. of IOD(Injured On Duty) incidents based on Classification (Non-Section 24/Section 24) */
const iodByClassification = async (dateFilter) => {
  try {
    // query to get injured persons
    const injuredPersonsQuery = `
      SELECT injured_persons
      FROM injury_on_duty
      WHERE ${dateFilter.getSQLWhereDateClause("incident_date")}
      AND deleted = 0;
    `;
    const [injuredRecords] = await db.query(injuredPersonsQuery);

    const classificationCounts = {
      "Non-Section 24": 0,
      "Section 24": 0,
    };

    for (const record of injuredRecords) {
      let injuredPersonsData = record?.injured_persons;
      try {
        let injuredPersons = [];
        if (injuredPersonsData && injuredPersonsData !== "null") {
          injuredPersons = JSON.parse(injuredPersonsData);
        }
        if (Array.isArray(injuredPersons)) {
          injuredPersons.forEach((person) => {
            if (person?.reportable_classification === "Non-Section 24") {
              classificationCounts["Non-Section 24"]++;
            } else if (person?.reportable_classification === "Section 24") {
              classificationCounts["Section 24"]++;
            }
          });
        }
      } catch (error) {
        console.error("Error parsing injured_persons:", error, record);
        // Handle error appropriately
      }
    }

    const data = Object.values(classificationCounts);
    const labels = Object.keys(classificationCounts);

    return {
      data,
      labels,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Persons classified by Experience who are involved in  IOD(Injured On Duty) incidents */
const personExperienceByIOD = async (dateFilter) => {
  try {
    // query to get injured persons
    const injuredPersonsQuery = `
      SELECT injured_persons
      FROM injury_on_duty
      WHERE ${dateFilter.getSQLWhereDateClause("incident_date")}
      AND deleted = 0;
    `;
    const [injuredRecords] = await db.query(injuredPersonsQuery);

    const experienceCounts = {
      "0-6 Months": new Set(),
      "6-12 Months": new Set(),
      "1-5 years": new Set(),
      "More than 5 years": new Set(),
    };

    for (const record of injuredRecords) {
      let injuredPersonsData = record?.injured_persons;
      try {
        let injuredPersons = [];
        if (injuredPersonsData && injuredPersonsData !== "null") {
          injuredPersons = JSON.parse(injuredPersonsData);
        }
        if (Array.isArray(injuredPersons)) {
          injuredPersons.forEach((person) => {
            const experienceType = person?.year_of_experience;
            const experienceNumber = parseInt(person?.experience_in_number);
            const userId = person?.user_id;

            if (userId) {
              let experienceInMonths = 0;
              if (experienceType === "Years" && !isNaN(experienceNumber)) {
                experienceInMonths = experienceNumber * 12;
              } else if (
                experienceType === "Months" &&
                !isNaN(experienceNumber)
              ) {
                experienceInMonths = experienceNumber;
              } else if (
                experienceType === "Days" &&
                !isNaN(experienceNumber)
              ) {
                experienceInMonths = Math.round(experienceNumber / 30); // Approximate conversion
              }

              if (experienceInMonths >= 0 && experienceInMonths <= 6) {
                experienceCounts["0-6 Months"].add(userId);
              } else if (experienceInMonths > 6 && experienceInMonths <= 12) {
                experienceCounts["6-12 Months"].add(userId);
              } else if (experienceInMonths > 12 && experienceInMonths <= 60) {
                experienceCounts["1-5 years"].add(userId);
              } else if (experienceInMonths > 60) {
                experienceCounts["More than 5 years"].add(userId);
              }
            }
          });
        }
      } catch (error) {
        console.error("Error parsing injured_persons:", error, record);
        // Handle error appropriately
      }
    }

    const data = [
      experienceCounts["0-6 Months"].size,
      experienceCounts["6-12 Months"].size,
      experienceCounts["1-5 years"].size,
      experienceCounts["More than 5 years"].size,
    ];
    const xaxis = [
      "0-6 Months",
      "6-12 Months",
      "1-5 years",
      "More than 5 years",
    ];

    return {
      data,
      xaxis,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of IOD incidents by Persons  */
const iodByPersons = async (dateFilter) => {
  try {
    // query to get injured persons
    const injuredPersonsQuery = `
      SELECT injured_persons
      FROM injury_on_duty
      WHERE ${dateFilter.getSQLWhereDateClause("incident_date")}
      AND deleted = 0;
    `;
    const [injuredRecords] = await db.query(injuredPersonsQuery);

    const injuredPersonCounts = {};
    const userIds = new Set();

    for (const record of injuredRecords) {
      let injuredPersonsData = record?.injured_persons;
      try {
        let injuredPersons = [];
        if (injuredPersonsData && injuredPersonsData !== "null") {
          injuredPersons = JSON.parse(injuredPersonsData);
        }
        if (Array.isArray(injuredPersons)) {
          injuredPersons.forEach((person) => {
            if (person?.user_id) {
              userIds.add(person.user_id);
            }
          });
        }
      } catch (error) {
        console.error("Error parsing injured_persons:", error, record);
        // Handle error appropriately
      }
    }

    const usersMap = {};
    if (userIds.size > 0) {
      const usersQuery = `
        SELECT id, CONCAT(name, ' ', surname) as notified_person
        FROM users
        WHERE deleted = '0' AND id IN (?)
      `;
      const [usersResult] = await db.query(usersQuery, [Array.from(userIds)]);
      usersResult.forEach((user) => {
        usersMap[user.id] = user.notified_person;
      });
    }

    for (const record of injuredRecords) {
      let injuredPersonsData = record?.injured_persons;
      try {
        let injuredPersons = [];
        if (injuredPersonsData && injuredPersonsData !== "null") {
          injuredPersons = JSON.parse(injuredPersonsData);
        }
        if (Array.isArray(injuredPersons)) {
          injuredPersons.forEach((person) => {
            if (person?.user_id && usersMap[person.user_id]) {
              const userName = usersMap[person.user_id];
              injuredPersonCounts[userName] =
                (injuredPersonCounts[userName] || 0) + 1;
            }
          });
        }
      } catch (error) {
        console.error("Error processing injured_persons:", error, record);
      }
    }

    const data = [
      {
        name: "Number of Incidents",
        type: "bar",
        data: Object.values(injuredPersonCounts),
        total: injuredRecords.length,
      },
    ];
    const xaxis = Object.keys(injuredPersonCounts);

    return {
      data,
      xaxis,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of IOD incidents by Categories  */
const iodByCategory = async (dateFilter) => {
  try {
    // query to get injured persons
    const injuredPersonsQuery = `
      SELECT injured_persons
      FROM injury_on_duty
      WHERE ${dateFilter.getSQLWhereDateClause("incident_date")}
      AND deleted = 0;
    `;
    const [injuredRecords] = await db.query(injuredPersonsQuery);

    const categoryCounts = {};

    for (const record of injuredRecords) {
      let injuredPersonsData = record?.injured_persons;
      try {
        let injuredPersons = [];
        if (injuredPersonsData && injuredPersonsData !== "null") {
          injuredPersons = JSON.parse(injuredPersonsData);
        }
        if (Array.isArray(injuredPersons)) {
          injuredPersons.forEach((person) => {
            const category = person?.injury_classification;
            if (category) {
              categoryCounts[category] = (categoryCounts[category] || 0) + 1;
            }
          });
        }
      } catch (error) {
        console.error("Error parsing injured_persons:", error, record);
        // Handle error appropriately
      }
    }

    const data = Object.values(categoryCounts);
    const labels = Object.keys(categoryCounts);

    return {
      data,
      labels,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of IOD incidents by Injury Type  */
const iodByType = async (dateFilter) => {
  try {
    // query to get injured persons
    const injuredPersonsQuery = `
      SELECT injured_persons
      FROM injury_on_duty
      WHERE ${dateFilter.getSQLWhereDateClause("incident_date")}
      AND deleted = 0;
    `;
    let [injuredRecords] = await db.query(injuredPersonsQuery);

    injuredRecords = await decodeAndParseFields(injuredRecords);
    const injuryTypeCounts = {};

    for (const record of injuredRecords) {
      let injuredPersonsData = record?.injured_persons;
      try {
        if (Array.isArray(injuredPersonsData)) {
          injuredPersonsData.forEach(async (person) => {
            const injuryType = person?.type_of_injury;
            const $ = cheerio.load(injuryType);
            const extractedText = $("p").text().trim();
            if (extractedText) {
              injuryTypeCounts[extractedText] =
                (injuryTypeCounts[extractedText] || 0) + 1;
            }
          });
        }
      } catch (error) {
        console.error("Error parsing injured_persons:", error, record);
        // Handle error appropriately
      }
    }

    const data = [
      {
        name: "Number of Incidents",
        type: "bar",
        data: Object.values(injuryTypeCounts),
        total: Object.values(injuryTypeCounts).reduce(
          (sum, count) => sum + count,
          0
        ),
      },
    ];
    const xaxis = Object.keys(injuryTypeCounts);

    return {
      data,
      xaxis,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Occupational Diseases by Category  */
const occupationalDiseasesByCategory = async (dateFilter) => {
  try {
    const getOccupationalDiseaseQuery = `
      SELECT occupationals
      FROM occupational_disease_form
      WHERE ${dateFilter.getSQLWhereDateClause("incident_date")}
      AND deleted = 0;
    `;
    const [occupationalDiseaseRecords] = await db.query(
      getOccupationalDiseaseQuery
    );

    const illnessTypeIds = new Set();
    const categoryCounts = {};

    for (const record of occupationalDiseaseRecords) {
      const occupationalsData = record?.occupationals;
      if (occupationalsData) {
        try {
          const occupationalsArray = JSON.parse(occupationalsData);
          if (Array.isArray(occupationalsArray)) {
            occupationalsArray.forEach((occupational) => {
              const occupationalTypes = occupational?.occupational_type;
              if (Array.isArray(occupationalTypes)) {
                occupationalTypes.forEach((id) => illnessTypeIds.add(id));
              }
            });
          }
        } catch (error) {
          console.error("Error parsing occupationals:", error, record);
        }
      }
    }

    const illnessTypeNamesMap = {};
    if (illnessTypeIds.size > 0) {
      const getOccupationalTypeQuery = `SELECT id, name FROM illness_type WHERE deleted = 0 AND id IN (?)`;
      const [illnessTypes] = await db.query(getOccupationalTypeQuery, [
        Array.from(illnessTypeIds),
      ]);
      illnessTypes.forEach((type) => {
        illnessTypeNamesMap[type.id] = type.name;
      });
    }

    for (const record of occupationalDiseaseRecords) {
      const occupationalsData = record?.occupationals;
      if (occupationalsData) {
        try {
          const occupationalsArray = JSON.parse(occupationalsData);
          if (Array.isArray(occupationalsArray)) {
            occupationalsArray.forEach((occupational) => {
              const occupationalTypes = occupational?.occupational_type;
              if (Array.isArray(occupationalTypes)) {
                occupationalTypes.forEach((id) => {
                  const illnessTypeName = illnessTypeNamesMap[id];
                  if (illnessTypeName) {
                    categoryCounts[illnessTypeName] =
                      (categoryCounts[illnessTypeName] || 0) + 1;
                  }
                });
              }
            });
          }
        } catch (error) {
          console.error("Error processing occupationals:", error, record);
        }
      }
    }

    const data = Object.values(categoryCounts);
    const labels = Object.keys(categoryCounts);

    return {
      data,
      labels,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Occupational Diseases by Description od Disease  */
const occupationalDiseasesByDescription = async (dateFilter) => {
  try {
    const getOccupationalDiseaseQuery = `
      SELECT occupationals
      FROM occupational_disease_form
      WHERE ${dateFilter.getSQLWhereDateClause("incident_date")}
      AND deleted = 0;
    `;
    const [occupationalDiseaseRecords] = await db.query(
      getOccupationalDiseaseQuery
    );

    const descriptionCounts = {};

    for (const record of occupationalDiseaseRecords) {
      const occupationalsData = record?.occupationals;
      if (occupationalsData) {
        try {
          const occupationalsArray = JSON.parse(occupationalsData);
          if (Array.isArray(occupationalsArray)) {
            occupationalsArray.forEach((occupational) => {
              const description = occupational?.occupational_description;
              const $ = cheerio.load(description);
              const extractedText = $("p").text().trim();
              if (extractedText) {
                descriptionCounts[extractedText] =
                  (descriptionCounts[extractedText] || 0) + 1;
              }
            });
          }
        } catch (error) {
          console.error("Error parsing occupationals:", error, record);
        }
      }
    }

    const data = [
      {
        name: "Number of Incidents",
        type: "bar",
        data: Object.values(descriptionCounts),
        total: Object.values(descriptionCounts).reduce(
          (sum, count) => sum + count,
          0
        ),
      },
    ];
    const xaxis = Object.keys(descriptionCounts);

    return {
      data,
      xaxis,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Occupational Diseases by Source of Exposure */
const occupationalDiseasesBySourceOfExposure = async (dateFilter) => {
  try {
    const getOccupationalDiseaseQuery = `
      SELECT occupationals
      FROM occupational_disease_form
      WHERE ${dateFilter.getSQLWhereDateClause("incident_date")}
      AND deleted = 0;
    `;
    const [occupationalDiseaseRecords] = await db.query(
      getOccupationalDiseaseQuery
    );

    const exposureRouteCounts = {};

    for (const record of occupationalDiseaseRecords) {
      const occupationalsData = record?.occupationals;
      if (occupationalsData) {
        try {
          const occupationalsArray = JSON.parse(occupationalsData);
          if (Array.isArray(occupationalsArray)) {
            occupationalsArray.forEach((occupational) => {
              const exposureRoute = occupational?.exposure_route;
              if (exposureRoute) {
                exposureRouteCounts[exposureRoute] =
                  (exposureRouteCounts[exposureRoute] || 0) + 1;
              }
            });
          }
        } catch (error) {
          console.error("Error parsing occupationals:", error, record);
        }
      }
    }

    const data = Object.values(exposureRouteCounts);
    const labels = Object.keys(exposureRouteCounts);

    return {
      data,
      labels,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Occupational Diseases by Clinic */
const occupationalDiseasesByClinic = async (dateFilter) => {
  try {
    // Query to get all organizations
    const getOrganizationQuery = `
      SELECT
        id, name
      FROM organization
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0
    `;
    const [organizations] = await db.query(getOrganizationQuery);
    const organizationMap = organizations.reduce((acc, org) => {
      acc[org.id] = org.name;
      return acc;
    }, {});

    const getOccupationalDiseaseQuery = `
      SELECT organization, occupationals
      FROM occupational_disease_form
      WHERE ${dateFilter.getSQLWhereDateClause("incident_date")}
      AND deleted = 0;
    `;
    const [occupationalDiseaseRecords] = await db.query(
      getOccupationalDiseaseQuery
    );

    const organizationCounts = {};

    for (const record of occupationalDiseaseRecords) {
      const organizationId = record?.organization;
      const occupationalsData = record?.occupationals;

      if (organizationId && occupationalsData) {
        try {
          const occupationalsArray = JSON.parse(occupationalsData);
          if (Array.isArray(occupationalsArray)) {
            const hasConfirmedExposure = occupationalsArray.some(
              (occupational) => occupational?.confirm_exposure === "1"
            );
            if (hasConfirmedExposure) {
              organizationCounts[organizationId] =
                (organizationCounts[organizationId] || 0) + 1;
            }
          }
        } catch (error) {
          console.error("Error parsing occupationals:", error, record);
        }
      }
    }

    const data = [
      {
        name: "Number of Incidents",
        type: "bar",
        data: Object.keys(organizationCounts).map(
          (orgId) => organizationCounts[orgId]
        ),
        total: Object.values(organizationCounts).reduce(
          (sum, count) => sum + count,
          0
        ),
      },
    ];
    const xaxis = Object.keys(organizationCounts).map(
      (orgId) => organizationMap[orgId] || `Organization ID: ${orgId}`
    );

    return {
      data,
      xaxis,
    };
  } catch (error) {
    throw error;
  }
};

/** No. of Asset Property Damage Incident based on Type */
const assetIncidentByType = async (dateFilter) => {
  try {
    const getAssetIncidentQuery = `
      SELECT asset_type
      FROM infrastructure_technical_form
      WHERE ${dateFilter.getSQLWhereDateClause("incident_date")}
      AND deleted = 0;
    `;
    const [assetIncidentRecords] = await db.query(getAssetIncidentQuery);

    const assetTypeIds = new Set();
    const assetTypeCounts = {};

    for (const record of assetIncidentRecords) {
      const assetTypeData = record?.asset_type;
      if (assetTypeData && Array.isArray(assetTypeData)) {
        assetTypeData.forEach((asset) => {
          const selectAssetType = asset?.select_asset_type;
          if (selectAssetType !== undefined && selectAssetType !== null) {
            assetTypeIds.add(selectAssetType);
          }
        });
      } else {
        console.warn("asset_type is not a valid array:", assetTypeData);
      }
    }

    const assetTypeNamesMap = {};
    if (assetTypeIds.size > 0) {
      const getAssetTypeQuery = `SELECT id, name FROM asset_type WHERE deleted = 0 AND id IN (?)`;
      const [assetTypes] = await db.query(getAssetTypeQuery, [
        Array.from(assetTypeIds),
      ]);
      assetTypes.forEach((type) => {
        assetTypeNamesMap[type.id] = type.name;
      });
    }

    for (const record of assetIncidentRecords) {
      const assetTypeData = record?.asset_type;
      if (assetTypeData && Array.isArray(assetTypeData)) {
        assetTypeData.forEach((asset) => {
          const selectAssetType = asset?.select_asset_type;
          const assetTypeName = assetTypeNamesMap[selectAssetType];
          if (assetTypeName) {
            assetTypeCounts[assetTypeName] =
              (assetTypeCounts[assetTypeName] || 0) + 1;
          }
        });
      }
    }

    const data = Object.values(assetTypeCounts);
    const xaxis = Object.keys(assetTypeCounts);

    return {
      data,
      xaxis,
    };
  } catch (error) {
    throw error;
  }
};

// get asset Incident by organization
const assetIncidentByOwner = async (dateFilter) => {
  const queryOrganizations = `SELECT id, name FROM organization WHERE deleted = "0"`;

  const [organizations] = await db.query(queryOrganizations);

  const organizationNames = organizations.map((org) => org.name);

  const queryInfrastructureByBusiness = `
        SELECT o.name as organization, COUNT(i.id) as count
        FROM infrastructure_technical_form i
        JOIN organization o ON i.organization = o.id WHERE ${dateFilter.getSQLWhereDateClause(
    "i.created_at"
  )} AND i.deleted = "0" AND i.asset_type IS NOT NULL
        GROUP BY o.name;
    `;
  const [infrastructureFormByBusiness] = await db.query(
    queryInfrastructureByBusiness
  );

  const colors = await Promise.all(
    organizations.map((org) => getColorForCategory(org.id, "organization"))
  );
  const infrastructure_technical_formData = organizationNames.map((orName) => {
    const found = infrastructureFormByBusiness.find(
      (m) => m.organization === orName
    );
    return found ? found.count : 0;
  });

  return {
    labels: organizationNames,
    data: infrastructure_technical_formData,
    colors,
  };
};

const assetIncidentBySeverity = async (dateFilter) => {
  const getAssetIncidentQuery = `
      SELECT asset_type
      FROM infrastructure_technical_form
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = 0 AND asset_type IS NOT NULL;
    `;

  const [assetIncidentRecords] = await db.query(getAssetIncidentQuery);

  const severityCounts = { "Not Set": 0 };

  const severityPromises = assetIncidentRecords.flatMap((assetIncident) => {
    if (!assetIncident.asset_type) return [];

    let assetTypes =
      typeof assetIncident.asset_type === "string"
        ? JSON.parse(assetIncident.asset_type)
        : assetIncident.asset_type;

    return assetTypes.map(async (record) => {
      let severity = record.damage_severity_level;
      if (severity) {
        const [[result]] = await db.query(
          `SELECT name FROM severity WHERE id = ? AND deleted = 0`,
          [severity]
        );
        let severityName = result ? result.name : "Not Set";
        severityCounts[severityName] = (severityCounts[severityName] || 0) + 1;
      } else {
        severityCounts["Not Set"] += 1;
      }
    });
  });

  await Promise.all(severityPromises);

  return {
    data: Object.values(severityCounts),
    labels: Object.keys(severityCounts),
  };
};

// get vehicle incident based on type
const vehicleIncidentByType = async (dateFilter) => {
  const query = `SELECT injured_persons FROM vehicle_incident_form WHERE ${dateFilter.getSQLWhereDateClause(
    "incident_date"
  )} AND deleted = "0" AND injured_persons IS NOT NULL`;

  const [records] = await db.query(query);

  let incidentCounts = {
    Theft: 0,
    Damage: 0,
    "Hi-jack": 0,
  };

  records.forEach((record) => {
    if (record.injured_persons) {
      const injuredPersons = JSON.parse(record.injured_persons);

      injuredPersons.forEach((person) => {
        if (person.vehicle_incident) {
          person.vehicle_incident.forEach((incident) => {
            const type = incident.vehicle_incident_type;
            if (incidentCounts.hasOwnProperty(type)) {
              incidentCounts[type] += 1;
            }
          });
        }
      });
    }
  });

  const data = {
    data: [
      {
        name: "Number of Vehicles",
        type: "bar",
        data: Object.values(incidentCounts),
        total: Object.values(incidentCounts).reduce(
          (sum, count) => sum + count,
          0
        ),
      },
    ],
    xaxis: Object.keys(incidentCounts),
  };
  return data;
};

const vehicleIncidentBySeverity = async (dateFilter) => {
  const query = `SELECT injured_persons FROM vehicle_incident_form WHERE ${dateFilter.getSQLWhereDateClause(
    "incident_date"
  )} AND deleted = "0" AND injured_persons IS NOT NULL`;

  const [records] = await db.query(query);

  const severityCounts = {
    "Not Set": 0,
  };

  const severityPromises = [];

  for (const record of records) {
    if (record.injured_persons) {
      const injuredPersons = JSON.parse(record.injured_persons);

      for (const person of injuredPersons) {
        if (person.vehicle_incident) {
          for (const incident of person.vehicle_incident) {
            let severity = incident.severity_of_damage;

            if (severity) {
              severityPromises.push(
                db
                  .query(
                    `SELECT name FROM severity WHERE id = '${severity}' AND deleted = "0"`
                  )
                  .then(([result]) => {
                    let severityName =
                      result.length > 0 ? result[0].name : "Not Set";
                    severityCounts[severityName] =
                      (severityCounts[severityName] || 0) + 1;
                  })
              );
            } else {
              severityCounts["Not Set"] += 1;
            }
          }
        }
      }
    }
  }

  await Promise.all(severityPromises);

  return {
    data: Object.values(severityCounts),
    labels: Object.keys(severityCounts),
  };
};

const vehicleIncidentByOwner = async (dateFilter) => {
  const queryOrganizations = `SELECT id, name FROM organization WHERE deleted = "0"`;

  const [organizations] = await db.query(queryOrganizations);

  const organizationCounts = {};
  organizations.forEach((org) => {
    organizationCounts[org.id] = 0;
  });

  const query = `
  SELECT organization, injured_persons 
  FROM vehicle_incident_form 
  WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
  AND deleted = "0" 
  AND organization IS NOT NULL 
  AND injured_persons IS NOT NULL;
`;

  const [records] = await db.query(query);
  records.forEach((record) => {
    const { organization, injured_persons } = record;

    if (injured_persons) {
      const injuredPersonsArray = JSON.parse(injured_persons);
      injuredPersonsArray.forEach((person) => {
        if (person.vehicle_incident) {
          organizationCounts[organization] += person.vehicle_incident.length;
        }
      });
    }
  });
  const organizationNames = organizations.map((org) => org.name);
  const incidentCounts = organizations.map(
    (org) => organizationCounts[org.id] || 0
  );

  const colors = await Promise.all(
    organizations.map((org) => getColorForCategory(org.id, "organization"))
  );

  return {
    data: incidentCounts,
    labels: organizationNames,
    colors,
  };
};

const vehicleIncidentByDamagedParts = async (dateFilter) => {
  const query = `
      SELECT injured_persons 
      FROM vehicle_incident_form 
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
      AND deleted = "0" 
      AND injured_persons IS NOT NULL;
  `;
  const [records] = await db.query(query);

  const partsCounts = {};
  const partNamePromises = [];
  const colorPromises = [];
  const partColors = {};

  for (const record of records) {
    if (record.injured_persons) {
      const injuredPersonsArray = JSON.parse(record.injured_persons);

      for (const person of injuredPersonsArray) {
        if (person.vehicle_incident) {
          for (const incident of person.vehicle_incident) {
            if (
              incident.car_parts_damaged &&
              Array.isArray(incident.car_parts_damaged)
            ) {
              for (const partId of incident.car_parts_damaged) {
                if (!partsCounts[partId]) {
                  partsCounts[partId] = 0;
                  partNamePromises.push(
                    db
                      .query(
                        `SELECT name FROM vehicle_parts WHERE id = '${partId}' AND deleted = "0"`
                      )
                      .then(([result]) => {
                        if (result.length > 0) {
                          const partName = result[0].name;
                          partsCounts[partName] = partsCounts[partId];
                          delete partsCounts[partId];
                          return { partId, partName };
                        }
                        return null;
                      })
                  );
                }
                partsCounts[partId] += 1;
              }
            }
          }
        }
      }
    }
  }

  // Wait for part names to resolve
  const resolvedParts = await Promise.all(partNamePromises);

  // Fetch colors for each part
  for (const part of resolvedParts) {
    if (part) {
      colorPromises.push(
        getColorForCategory(part.partId, "damage_part").then((color) => {
          partColors[part.partName] = color;
        })
      );
    }
  }

  // Wait for colors to resolve
  await Promise.all(colorPromises);

  const data = {
    data: [
      {
        name: "Number of Vehicles",
        type: "bar",
        data: Object.values(partsCounts),
        total: Object.values(partsCounts).reduce(
          (sum, count) => sum + count,
          0
        ),
      },
    ],
    xaxis: Object.keys(partsCounts),
    colors: Object.keys(partsCounts).map(
      (partName) => partColors[partName] || "#000000"
    ), // Default to black if no color found
  };

  return data;
};

const nearMissByOrganization = async (dateFilter) => {
  const queryOrganizations = `SELECT id, name FROM organization where deleted = "0"`;

  const [orgResult] = await db.query(queryOrganizations);

  const querySeverityByOrg = `
        SELECT o.name as organization, n.severity, COUNT(n.id) as count
        FROM near_miss_reporting n
        JOIN organization o ON n.organization = o.id WHERE ${dateFilter.getSQLWhereDateClause(
    "n.created_at"
  )} AND n.deleted = "0" AND n.severity IS NOT NULL
        GROUP BY o.name, n.severity;
    `;

  const [severityByOrganization] = await db.query(querySeverityByOrg);

  const organizationNames = orgResult.map((org) => org.name);

  const defaultSeverity = ["Low", "Medium", "High"];

  const severityMap = defaultSeverity.map((severity) => {
    return {
      severity,
      data: organizationNames.map((orgName) => {
        // Find matching record
        const found = severityByOrganization.find(
          (item) => item.organization === orgName && item.severity === severity
        );
        return found ? found.count : 0;
      }),
      total:
        severityByOrganization
          .filter((item) => item.severity === severity)
          .reduce((sum, item) => sum + item.count, 0) || 0,
    };
  });

  const data = {
    data: severityMap,
    xaxis: organizationNames,
  };

  return data;
};

const nearMissTrends = async (dateFilter) => {
  const query = `
    SELECT 
      YEAR(created_at) AS year, 
      severity, 
      COUNT(*) AS count
    FROM near_miss_reporting
    WHERE deleted = 0
    GROUP BY year, severity
    ORDER BY year ASC;
  `;

  const [records] = await db.query(query);

  const yearWiseData = {};
  const severityLevels = ["Low", "Medium", "High"];

  records.forEach(({ year, severity, count }) => {
    if (!yearWiseData[year]) {
      yearWiseData[year] = { Low: 0, Medium: 0, High: 0 };
    }
    yearWiseData[year][severity] = count;
  });

  return {
    data: severityLevels.map((severity) => ({
      name: severity,
      data: Object.keys(yearWiseData).map(
        (year) => yearWiseData[year][severity] || 0
      ),
    })),
    xaxis: Object.keys(yearWiseData),
  };
};

const nearMissByCategoryAndSeverity = async (dateFilter) => {
  const [categories] = await db.query(
    `SELECT id, name FROM incident_category WHERE deleted = 0;`
  );

  const [records] = await db.query(
    `SELECT near_miss_classification, severity FROM near_miss_reporting WHERE ${dateFilter.getSQLWhereDateClause(
      "created_at"
    )} AND deleted = 0;`
  );

  const categoryWiseData = {};
  const severityLevels = ["High", "Medium", "Low"];

  const categoryMap = {};
  categories.forEach(({ id, name }) => {
    categoryMap[id] = name;
    categoryWiseData[name] = { High: 0, Medium: 0, Low: 0 };
  });

  records.forEach(({ near_miss_classification, severity }) => {
    if (near_miss_classification) {
      let categoryIds = [];
      try {
        categoryIds = JSON.parse(near_miss_classification);
      } catch (e) {
        categoryIds = near_miss_classification
          .split(",")
          .map((id) => parseInt(id.trim()));
      }

      categoryIds.forEach((categoryId) => {
        const categoryName = categoryMap[categoryId];
        if (categoryName) {
          categoryWiseData[categoryName][severity] += 1;
        }
      });
    }
  });

  return {
    data: severityLevels.map((severity) => ({
      name: severity,
      type: "bar",
      data: Object.keys(categoryWiseData).map(
        (category) => categoryWiseData[category][severity]
      ),
    })),
    xaxis: Object.keys(categoryWiseData),
  };
};

const nearMissByProbability = async (dateFilter) => {
  const query = `SELECT probability, COUNT(*) as count FROM near_miss_reporting WHERE ${dateFilter.getSQLWhereDateClause(
    "created_at"
  )} AND deleted = "0" GROUP BY probability;`;

  const [records] = await db.query(query);

  const defaultProbability = ["Low", "Medium", "High"];

  const probabilityMap = defaultProbability.map((probability) => {
    return {
      probability,
      count:
        records.find((item) => item.probability === probability)?.count || 0,
    };
  });

  return {
    data: probabilityMap.map((item) => item.count),
    labels: probabilityMap.map((item) => item.probability),
  };
};

const securityIncidentByType = async (dateFilter) => {
  const securityTypeQuery = `
  SELECT st.id, st.name as type, COUNT(sf.id) as count 
  FROM security_incident_form sf
  JOIN security_incident_type st ON sf.type_of_security_incident = st.id WHERE ${dateFilter.getSQLWhereDateClause(
    "sf.created_at"
  )} AND sf.deleted = "0" AND sf.type_of_security_incident IS NOT NULL
  GROUP BY st.id, st.name;
`;

  const [records] = await db.query(securityTypeQuery);
  return formatDashboardData(records, "type", null, "xaxis");
};

const securityIncidentBySuspectType = async (dateFilter) => {
  const [records] = await db.query(
    `SELECT suspects_detail FROM security_incident_form WHERE deleted = 0 AND ${dateFilter.getSQLWhereDateClause(
      "created_at"
    )}`
  );

  const suspectCounts = {
    "Internal Employee": 0,
    Contractor: 0,
    External: 0,
    Unknown: 0,
  };

  records.forEach(({ suspects_detail }) => {
    if (suspects_detail) {
      let suspectsArray = suspects_detail || [];

      if (typeof suspects_detail === "string") {
        suspectsArray = JSON.parse(suspects_detail);
      }

      suspectsArray.forEach(({ suspects }) => {
        if (suspectCounts.hasOwnProperty(suspects)) {
          suspectCounts[suspects] += 1;
        }
      });
    }
  });

  const result = {
    data: [
      {
        name: "Number of Security",
        data: Object.values(suspectCounts),
      },
    ],
    xaxis: Object.keys(suspectCounts),
  };

  return result;
};

const securityIncidentOccurrenceAtHour = async (dateFilter) => {
  const [records] = await db.query(
    `SELECT time_of_incident FROM security_incident_form WHERE deleted = 0 AND ${dateFilter.getSQLWhereDateClause(
      "created_at"
    )}`
  );

  const hourCounts = Array(24).fill(0);

  records.forEach(({ time_of_incident }) => {
    if (time_of_incident) {
      const hour = parseInt(time_of_incident.split(":")[0], 10); // Extract hour
      if (!isNaN(hour) && hour >= 0 && hour <= 23) {
        hourCounts[hour] += 1;
      }
    }
  });

  return {
    data: hourCounts,
    labels: Array.from({ length: 24 }, (_, i) => i.toString()),
  };
};

const securityIncidentOccurrenceByDayOfWeek = async (dateFilter) => {
  try {
    const query = `
      SELECT
        id,
        date_of_incident
        FROM security_incident_form
        WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
        AND deleted = "0"
      `;
    const [records] = await db.query(query);

    const dayOfWeekCounts = {
      Sunday: 0,
      Monday: 0,
      Tuesday: 0,
      Wednesday: 0,
      Thursday: 0,
      Friday: 0,
      Saturday: 0,
    };

    for (const record of records) {
      const actualDate = record.date_of_incident;
      if (actualDate) {
        const date = new Date(actualDate);
        const dayIndex = date.getDay(); // 0 for Sunday, 1 for Monday, etc.
        const daysOfWeek = [
          "Sunday",
          "Monday",
          "Tuesday",
          "Wednesday",
          "Thursday",
          "Friday",
          "Saturday",
        ];
        const dayName = daysOfWeek[dayIndex];
        dayOfWeekCounts[dayName]++;
      }
    }

    const data = Object.values(dayOfWeekCounts);
    const xaxis = Object.keys(dayOfWeekCounts);

    return {
      data,
      xaxis,
    };
  } catch (error) {
    throw error;
  }
};

const getCostBreakDown = async (dateFilter) => {
  const [records] = await db.query(
    `SELECT asset_type, vehicle_details FROM investigation WHERE deleted = "0" AND ${dateFilter.getSQLWhereDateClause(
      "created_at"
    )} AND asset_type IS NOT NULL AND vehicle_details IS NOT NULL`
  );

  let estimatedTotal = 0;
  let actualTotal = 0;

  records.forEach((record) => {
    let assets =
      typeof record.asset_type === "string"
        ? JSON.parse(record.asset_type)
        : record.asset_type;
    assets.forEach((asset) => {
      if (asset.value) estimatedTotal += parseFloat(asset.value) || 0;
      if (asset.actual_value)
        actualTotal += parseFloat(asset.actual_value) || 0;
    });

    if (record.vehicle_details) {
      let vehicles =
        typeof record.vehicle_details === "string"
          ? JSON.parse(record.vehicle_details)
          : record.vehicle_details;
      vehicles.forEach((vehicle) => {
        if (vehicle.damage_value)
          estimatedTotal += parseFloat(vehicle.damage_value) || 0;
        if (vehicle.actual_value)
          actualTotal += parseFloat(vehicle.actual_value) || 0;
      });
    }
  });

  return {
    data: [estimatedTotal, actualTotal],
    labels: ["Estimated", "Actual"],
  };
};

const getEstimatedVsActualCostByYear = async (dateFilter) => {
  const query = `
      SELECT report_date, asset_type, vehicle_details 
      FROM investigation 
      WHERE deleted = "0" AND report_date IS NOT NULL AND asset_type IS NOT NULL AND vehicle_details IS NOT NULL;
    `;
  const [records] = await db.query(query);

  const yearWiseCost = {};

  records.forEach((record) => {
    const year = new Date(record.report_date).getFullYear();

    if (!yearWiseCost[year]) {
      yearWiseCost[year] = { estimated: 0, actual: 0 };
    }

    // Extract estimated & actual values from asset_type
    if (record.asset_type) {
      const assets = JSON.parse(record.asset_type);
      assets.forEach((asset) => {
        if (asset.value)
          yearWiseCost[year].estimated += parseFloat(asset.value) || 0;
        if (asset.actual_value)
          yearWiseCost[year].actual += parseFloat(asset.actual_value) || 0;
      });
    }

    // Extract estimated & actual values from vehicle_details
    if (record.vehicle_details) {
      const vehicles = JSON.parse(record.vehicle_details);
      vehicles.forEach((vehicle) => {
        if (vehicle.damage_value)
          yearWiseCost[year].estimated += parseFloat(vehicle.damage_value) || 0;
        if (vehicle.actual_value)
          yearWiseCost[year].actual += parseFloat(vehicle.actual_value) || 0;
      });
    }
  });

  const result = {
    data: [
      {
        name: "Estimated",
        data: Object.values(yearWiseCost).map((y) => y.estimated),
      },
      {
        name: "Actual",
        data: Object.values(yearWiseCost).map((y) => y.actual),
      },
    ],
    labels: Object.keys(yearWiseCost).map((y) => parseInt(y)),
  };

  return result;
};

const getActualCostByType = async (dateFilter) => {
  const [records] = await db.query(
    `SELECT asset_type, vehicle_details FROM investigation WHERE deleted = "0" AND ${dateFilter.getSQLWhereDateClause(
      "created_at"
    )} AND asset_type IS NOT NULL AND vehicle_details IS NOT NULL`
  );

  let totalAssetCost = 0;
  let totalVehicleCost = 0;

  records.forEach((record) => {
    // Sum actual costs from asset_type
    if (record.asset_type) {
      const assets = JSON.parse(record.asset_type);
      assets.forEach((asset) => {
        if (asset.actual_value)
          totalAssetCost += parseFloat(asset.actual_value) || 0;
      });
    }

    // Sum actual costs from vehicle_details
    if (record.vehicle_details) {
      const vehicles = JSON.parse(record.vehicle_details);
      vehicles.forEach((vehicle) => {
        if (vehicle.actual_value)
          totalVehicleCost += parseFloat(vehicle.actual_value) || 0;
      });
    }
  });

  const result = {
    data: [
      {
        name: "Actual Cost",
        data: [totalAssetCost, totalVehicleCost],
      },
    ],
    xaxis: ["Asset", "Vehicle"],
  };
  return result;
};
const getActualCostByProperty = async (dateFilter) => {
  const query = `SELECT asset_type FROM investigation WHERE deleted = "0" AND ${dateFilter.getSQLWhereDateClause(
    "created_at"
  )} AND asset_type IS NOT NULL`;
  const [records] = await db.query(query);

  let assetTypeCosts = {};

  records.forEach((record) => {
    let assets = record.asset_type;
    if (typeof record.asset_type == "string") {
      assets = JSON.parse(record.asset_type);
    }
    assets.forEach((asset) => {
      if (asset.select_asset_type && asset.actual_value) {
        const assetTypeId = asset.select_asset_type;
        const actualValue = parseFloat(asset.actual_value) || 0;

        assetTypeCosts[assetTypeId] =
          (assetTypeCosts[assetTypeId] || 0) + actualValue;
      }
    });
  });

  const assetTypeIds = Object.keys(assetTypeCosts);
  if (assetTypeIds.length === 0) return { data: [], xaxis: [] };

  const assetTypeQuery = `SELECT id, name FROM asset_type WHERE id IN (${assetTypeIds.join(
    ","
  )})`;
  const [assetTypes] = await db.query(assetTypeQuery);

  const assetTypeMap = {};
  assetTypes.forEach((type) => {
    assetTypeMap[type.id] = type.name;
  });

  const result = {
    data: [
      {
        name: "Actual Cost",
        data: assetTypeIds.map((id) => assetTypeCosts[id]),
      },
    ],
    xaxis: assetTypeIds.map((id) => assetTypeMap[id]),
  };
  return result;
};

const getEnvironmentOccurrenceByConsequences = async (dateFilter) => {
  const incidentTypeList = [
    "Air Quality Incidents",
    "Water Pollution Incidents",
    "Land and Soil Contamination Incidents",
    "Hazardous Material Incidents",
    "Energy and Resource-Related Incidents",
    "Waste Management Incidents",
    "Chemical and Toxic Incidents",
    "Noise and Vibration Incidents",
    "Biodiversity and Wildlife Incidents",
    "Emission of Dangerous Substances",
    "Environmental Degradation Incidents",
    "Environmental Regulatory Incidents",
  ];

  const query = `SELECT incident_type FROM environmental_incidents WHERE deleted = "0" AND ${dateFilter.getSQLWhereDateClause(
    "created_at"
  )} AND incident_type IS NOT NULL`;
  const [records] = await db.query(query);

  let incidentCounts = incidentTypeList.reduce((acc, type) => {
    acc[type] = 0;
    return acc;
  }, {});

  records.forEach((record) => {
    let incidents = record.incident_type;
    if (typeof record.incident_type == "string") {
      incidents = JSON.parse(record.incident_type);
    }
    incidents.forEach((incident) => {
      if (
        incident.incident_type &&
        incidentTypeList.includes(incident.incident_type)
      ) {
        incidentCounts[incident.incident_type] += 1;
      }
    });
  });

  return {
    data: Object.values(incidentCounts),
    labels: Object.keys(incidentCounts),
  };
};

const getEnvironmentOccurrenceByLeakageAndEmission = async (dateFilter) => {
  const query = `SELECT incident_type FROM environmental_incidents WHERE deleted = "0" AND ${dateFilter.getSQLWhereDateClause(
    "created_at"
  )} AND incident_type IS NOT NULL`;
  const [records] = await db.query(query);

  const leakage = [
    "Chemical Spill into Water Bodies",
    "Oil Spill",
    "Spill of Hazardous Waste on Land",
    "Spill of Industrial Waste",
    "Hazardous Chemical Spill",
    "Radioactive Material Spill",
    "Corrosive Substance Spill",
    "Sewage Leak",
    "Storage Tank Leak",
    "Explosive Material Leak",
    "Toxic Waste Leakage",
  ];

  const emission = [
    "Emission of Greenhouse Gases",
    "Fugitive Emissions",
    "Dust Emissions",
    "Odor Emissions",
    "Excessive Carbon Dioxide Emission",
    "Smoke Emissions",
    "Release of Dioxins",
    "Emission of Sulfur Oxides (SOx)",
    "Nitrogen Oxides (NOx) Emissions",
    "Emission of Mercury",
    "Release of CFCs or Ozone-Depleting Substances",
    "Emission of Hydrogen Sulfide",
    "Polychlorinated Biphenyl (PCB) Release",
    "Emission of Benzene or Other Carcinogens",
  ];

  const wasteType = [
    'Improper Waste Disposal',
    'Failure to Segregate Waste',
    'Inadequate Recycling Practices',
    'Failure to Dispose of Hazardous Waste',
    'Illegal Waste Dumping',
    'Failure in Wastewater Treatment',
    'Biohazardous Waste Release'
  ]

  let leakageCounts = leakage.reduce((acc, type) => {
    acc[type] = 0;
    return acc;
  }, {});

  let emissionCounts = emission.reduce((acc, type) => {
    acc[type] = 0;
    return acc;
  }, {});

  let wasteCounts = wasteType.reduce((acc, type) => {
    acc[type] = 0;
    return acc;
  }, {});

  records.forEach((record) => {
    let incidents = record.incident_type;
    if (typeof record.incident_type == "string") {
      incidents = JSON.parse(record.incident_type);
    }
    incidents.forEach((incident) => {
      if (incident.incident_type_option) {
        if (leakage.includes(incident.incident_type_option)) {
          leakageCounts[incident.incident_type_option] += 1;
        } else if (emission.includes(incident.incident_type_option)) {
          emissionCounts[incident.incident_type_option] += 1;
        } else if (wasteType.includes(incident.incident_type_option)) {
          wasteCounts[incident.incident_type_option] += 1;
        }
      }
    });
  });

  const leakageResult = {
    data: [
      {
        name: "Number of Environmental Incidents",
        data: Object.values(leakageCounts),
      },
    ],
    xaxis: Object.keys(leakageCounts),
  };
  const emissionResult = {
    data: [
      {
        name: "Number of Environmental Incidents",
        data: Object.values(emissionCounts),
      },
    ],
    xaxis: Object.keys(emissionCounts),
  };
  const wasteResult = {
    data: [
      {
        name: "Number of Environmental Incidents",
        data: Object.values(wasteCounts),
      },
    ],
    xaxis: Object.keys(wasteCounts),
  };

  return { leakageResult, emissionResult, wasteResult };
};


/** *********************  ACTION MODULE ***************************** */

export const getActionDashboardData = async (req, res, next) => {
  const { all, filter } = req.query;

  if (!filter) {
    return sendResponse(res, 404, "Filter not found");
  }

  const currentMonth = moment().format("YYYY-MM");

  req.query.filter = JSON.parse(filter);

  const dateRange = req.query.filter?.date_type || "All-Time";
  const start_date = req.query.filter?.start_date;
  const end_date = req.query.filter?.end_date;
  const dateFilter = getDateRangeForSQL(dateRange, start_date, end_date);

  const { actionStatus, openedActions, overdueActions } =
    await getStatusActionData(dateFilter);

  const categoryAction = await getCategoryActionData(dateFilter);

  const queryCurrentMonth = `
        SELECT COUNT(*) as count 
        FROM custom_action_creation 
        WHERE DATE_FORMAT(created_at, '%Y-%m') = ? AND deleted = "0";
    `;

  const [currentMonthCount] = await db.query(queryCurrentMonth, [currentMonth]);

  const queryCompletedCurrentMonth = `
        SELECT COUNT(*) as count 
        FROM custom_action_creation 
        WHERE status = 'Completed' AND DATE_FORMAT(created_at, '%Y-%m') = ? AND deleted = "0";
    `;

  const [completedCurrentMonthCount] = await db.query(
    queryCompletedCurrentMonth,
    [currentMonth]
  );

  const actionModuleInstances = await getActionInstanceModules(dateFilter);

  const actionByBusiness = await getActionByBusiness(dateFilter);

  const finalData = {
    actionStatus,
    categoryAction,
    overdue: overdueActions,
    opened: openedActions,
    currentMonth: currentMonthCount[0].count,
    currentMonthCompleted: completedCurrentMonthCount[0].count,
    actionModuleInstances,
    actionByBusiness: actionByBusiness,
  };

  return sendResponse(res, 200, finalData);
};

// get status based action count
async function getStatusActionData(dateFilter) {
  const statusQuery = `SELECT status, COUNT(*) as count FROM custom_action_creation WHERE ${dateFilter.getSQLWhereDateClause(
    "created_at"
  )} AND deleted = "0" AND status IS NOT NULL GROUP BY status`;

  const predefinedStatuses = [
    "Pending",
    "To-Do",
    "In-Progress",
    "Completed",
    "Over-Due",
  ];

  const [statusResult] = await db.query(statusQuery);

  const actionByStatusProcessed = predefinedStatuses.map((status) => {
    const found = statusResult.find((item) => item.status === status);
    return { status, count: found ? found.count : 0 };
  });
  let openedActions = 0;
  let overdueActions = 0;
  actionByStatusProcessed.forEach((action) => {
    if (action.status !== "Over-Due") {
      openedActions += action.count;
    } else {
      overdueActions = action.count;
    }
  });
  const actionStatus = formatDashboardData(actionByStatusProcessed, "status");
  return { actionStatus, openedActions, overdueActions };
}

// get category based action count
async function getCategoryActionData(dateFilter) {
  const categoryQuery = `
  SELECT c.id, c.name as category, COUNT(ca.id) as count 
  FROM custom_action_creation ca
  JOIN category c ON ca.tag = c.id WHERE ${dateFilter.getSQLWhereDateClause(
    "ca.created_at"
  )} AND ca.deleted = "0" AND ca.tag IS NOT NULL
  GROUP BY c.id, c.name;
`;

  const [categoryResult] = await db.query(categoryQuery);

  const categoryColors = await Promise.all(
    categoryResult.map((category) =>
      getColorForCategory(category.id, "action_category")
    )
  );

  return formatDashboardData(categoryResult, "category", categoryColors);
}

// get action instance module count
async function getActionInstanceModules(dateFilter) {
  const queryModule = `
        SELECT s.id, s.title as sub_module, COUNT(a.id) as count 
        FROM custom_action_creation a
        JOIN sidebar s ON a.sub_module = s.id WHERE ${dateFilter.getSQLWhereDateClause(
    "a.created_at"
  )} AND a.deleted = "0" AND a.sub_module IS NOT NULL
        GROUP BY s.id, s.title;
    `;

  const [moduleResult] = await db.query(queryModule);

  const moduleColors = await Promise.all(
    moduleResult.map((module) =>
      getColorForCategory(module.id, "action_sub_module")
    )
  );

  const formattedModule = formatDashboardData(
    moduleResult,
    "sub_module",
    moduleColors
  );

  return formattedModule;
}

// get action by business
async function getActionByBusiness(dateFilter) {
  const queryOrganizations = `SELECT id, name FROM organization where deleted = "0"`;

  const [orgResult] = await db.query(queryOrganizations);

  const queryStatusByOrg = `
        SELECT o.name as organization, a.status, COUNT(a.id) as count
        FROM custom_action_creation a
        JOIN organization o ON a.organization = o.id WHERE ${dateFilter.getSQLWhereDateClause(
    "a.created_at"
  )} AND a.deleted = "0" AND a.status IS NOT NULL
        GROUP BY o.name, a.status;
    `;

  const predefinedStatuses = [
    "Pending",
    "To-Do",
    "In-Progress",
    "Completed",
    "Over-Due",
  ];
  const [statusByOrganization] = await db.query(queryStatusByOrg);

  const organizationNames = orgResult.map((org) => org.name);

  // Create a map for structured data
  const statusMap = predefinedStatuses.map((status) => {
    return {
      status,
      data: organizationNames.map((orgName) => {
        // Find matching record
        const found = statusByOrganization.find(
          (item) => item.organization === orgName && item.status === status
        );
        return found ? found.count : 0;
      }),
      total:
        statusByOrganization
          .filter((item) => item.status === status)
          .reduce((sum, item) => sum + item.count, 0) || 0,
    };
  });

  return {
    data: statusMap,
    xaxis: organizationNames,
  };
}

// format data based on passed key
const formatDashboardData = (data, key, colors = null, label = "labels") => {
  const result = {
    [label]: data.map((item) => item[key]),
    data: data.map((item) => item.count),
  };

  if (colors) {
    result.colors = colors;
  }

  return result;
};

/** *********************  MEETING MODULE ***************************** */

export async function getMeetingDashboardData(req, res) {
  const { all, filter } = req.query;

  if (!filter) {
    return sendResponse(res, 404, "Filter not found");
  }

  req.query.filter = JSON.parse(filter);

  const dateRange = req.query.filter?.date_type || "All-Time";
  const start_date = req.query.filter?.start_date;
  const end_date = req.query.filter?.end_date;
  const dateFilter = getDateRangeForSQL(dateRange, start_date, end_date);

  const meetingByBusiness = await getMeetingByBusiness(dateFilter);
  const { attendancePercentage, attendanceStatus } =
    await getAttendancePercentageAndAttendanceByStatus(dateFilter);

  const conductedPercentage = await getConductedMeeting(dateFilter);
  const meetingType = await getMeetingType(dateFilter);
  const attendanceBusiness = await getAttendanceByBusiness(dateFilter);
  const attendanceDate = await getAttendanceByDate(dateFilter);
  const plannedVsCompletedAttendance = await getPlannedVsCompletedAttendance(
    dateFilter
  );
  const overduePlannedMeetings = await getOverduePlannedMeetings(dateFilter);
  const finalData = {
    meetingByBusiness,
    attendancePercentage,
    conductedPercentage,
    meetingType,
    attendanceStatus,
    attendanceBusiness,
    attendanceDate,
    plannedVsCompletedAttendance,
    overduePlannedMeetings,
  };
  return sendResponse(res, 200, finalData);
}

const getMeetingByBusiness = async (dateFilter) => {
  const queryOrganizations = `SELECT id, name FROM organization WHERE deleted = "0"`;

  const [organizations] = await db.query(queryOrganizations);

  const organizationNames = organizations.map((org) => org.name);

  const queryMeetingsByBusiness = `
        SELECT o.name as organization, COUNT(m.id) as count
        FROM meeting m
        JOIN organization o ON m.organization = o.id WHERE ${dateFilter.getSQLWhereDateClause(
    "m.created_at"
  )} AND m.deleted = "0"
        GROUP BY o.name;
    `;
  const [meetingResults] = await db.query(queryMeetingsByBusiness);

  const meetingData = organizationNames.map((orName) => {
    const found = meetingResults.find((m) => m.organization === orName);
    return found ? found.count : 0;
  });

  const result = {
    data: [
      {
        name: "Number of Meetings",
        data: meetingData,
      },
    ],
    xaxis: organizationNames,
  };

  return result;
};

const getAttendancePercentageAndAttendanceByStatus = async (dateFilter) => {
  const queryMeetings = `SELECT participants FROM meeting_recording WHERE ${dateFilter.getSQLWhereDateClause(
    "created_at"
  )} AND deleted = "0"`;

  const [meetings] = await db.query(queryMeetings);

  const defaultStatuses = ["present", "absent", "apologised", "Not Set"];

  let totalParticipants = 0;
  let presentParticipants = 0;

  let participantMap = {}; // Stores count of each status per participant
  let statusMap = {}; // Stores total count per status
  let participantsList = new Set(); // Stores unique participants

  for (const meeting of meetings) {
    let participants = meeting.participants;
    if (typeof participants == "string") {
      participants = JSON.parse(participants);
    }

    totalParticipants += participants.length;
    presentParticipants += participants.filter(
      (p) => p.status === "present"
    ).length;

    participants.forEach((p) => {
      const fullName = `${p.name} ${p.surname}`;
      participantsList.add(fullName);

      // Determine status: Use existing status or default to "Not Set"
      const status = p.status ? p.status : "Not Set";

      // Initialize participant in map
      if (!participantMap[fullName]) {
        participantMap[fullName] = {};
      }

      // Initialize status for participant
      if (!participantMap[fullName][status]) {
        participantMap[fullName][status] = 0;
      }

      participantMap[fullName][status] += 1;

      // Track total count for each status
      if (!statusMap[status]) {
        statusMap[status] = 0;
      }
      statusMap[status] += 1;
    });
  }

  const percentagePresent =
    totalParticipants > 0 ? (presentParticipants / totalParticipants) * 100 : 0;

  const xaxis = Array.from(participantsList);
  let formattedData = defaultStatuses.map((status) => {
    return {
      status,
      data: xaxis.map((name) => participantMap[name]?.[status] || 0),
      total: statusMap[status] || 0,
    };
  });

  const attendanceStatus = {
    data: formattedData,
    xaxis,
  };

  return {
    attendancePercentage: percentagePresent.toFixed(2) + "%",
    attendanceStatus,
  };
};

// const getConductedMeeting = async (dateFilter) => {
//   const [[{ totalMeetings }]] = await db.query(
//     `SELECT COUNT(*) AS totalMeetings FROM meeting WHERE ${dateFilter.getSQLWhereDateClause(
//       "created_at"
//     )} AND deleted = "0"`
//   );

//   const [[{ conductedMeetings }]] = await db.query(
//     `SELECT COUNT(*) AS conductedMeetings FROM meeting_recording WHERE ${dateFilter.getSQLWhereDateClause(
//       "created_at"
//     )} AND deleted = "0"`
//   );

//   const conductedPercentage =
//     totalMeetings > 0 ? (conductedMeetings / totalMeetings) * 100 : 0;

//   return conductedPercentage.toFixed(2) + "%";
// };

const getConductedMeeting = async (dateFilter) => {
  const [[{ totalMeetings }]] = await db.query(
    ` SELECT COUNT(*) AS totalMeetings
      FROM meeting
      WHERE ${dateFilter.getSQLWhereDateClause("created_at")}
        AND deleted = "0"`
  );

  // Early exit if no meetings were scheduled in the period.
  // Prevents division by zero and ensures logical consistency.
  if (totalMeetings === 0 || totalMeetings == null) {
    return "0.00%";
  }


  const [[{ conductedMeetingsCount }]] = await db.query(
    ` SELECT COUNT(DISTINCT m.id) AS conductedMeetingsCount
      FROM meeting m
      JOIN meeting_recording mr ON m.id = mr.scheduled_meeting
      WHERE ${dateFilter.getSQLWhereDateClause("m.created_at")} -- Filter based on meeting creation date
        AND m.deleted = "0"  -- Ensure the meeting itself wasn't deleted
        AND mr.deleted = "0" -- Ensure the recording wasn't deleted
      GROUP BY m.id` // Group by meeting ID to count distinct meetings
  );

  const conductedCount = conductedMeetingsCount || 0;
  const conductedPercentage = (conductedCount / totalMeetings) * 100;

  return conductedPercentage.toFixed(2) + "%";
};

// get meeting by category // no field like category
// const getMeetingCategory = async (dateFilter) => {}

const getMeetingType = async (dateFilter) => {
  const query = `SELECT meeting_hierarchy, COUNT(*) as count FROM meeting WHERE ${dateFilter.getSQLWhereDateClause(
    "created_at"
  )} AND deleted = "0" AND meeting_hierarchy IS NOT NULL GROUP BY meeting_hierarchy`;

  const [hierarchyResult] = await db.query(query);

  const meetingHierarchies = [
    "Executive Meeting",
    "Operational Meeting",
    "General Meeting",
    "Project Meeting",
    "Management Meeting",
    "Compliance Meeting",
  ];

  const meetingHierarchyProcessed = meetingHierarchies.map((type) => {
    const found = hierarchyResult.find(
      (item) => item.meeting_hierarchy === type
    );
    return { type, count: found ? found.count : 0 };
  });

  const meetingType = formatDashboardData(meetingHierarchyProcessed, "type");
  return meetingType;
};

const getAttendanceByBusiness = async (dateFilter) => {
  const queryOrganizations = `SELECT id, name FROM organization WHERE deleted = "0"`;

  const [organizations] = await db.query(queryOrganizations);

  const organizationNames = organizations.map((org) => org.name);

  const queryMeetings = `SELECT participants FROM meeting_recording WHERE ${dateFilter.getSQLWhereDateClause(
    "created_at"
  )} AND deleted = "0"`;

  const [meetings] = await db.query(queryMeetings);

  const defaultStatuses = ["present", "absent", "apologised", "Not Set"];
  let attendanceData = {};

  // Initialize structure with default 0 counts
  defaultStatuses.forEach((status) => {
    attendanceData[status] = {
      status,
      data: Array(organizationNames.length).fill(0),
      total: 0,
    };
  });

  meetings.forEach((row) => {
    if (row.participants) {
      const participants = JSON.parse(row.participants);
      participants.forEach((p) => {
        const orgIndex = organizations.findIndex(
          (org) => org.id == JSON.parse(p.organization)
        );
        if (orgIndex !== -1) {
          const participantStatus = defaultStatuses.includes(p.status)
            ? p.status
            : "Not Set";
          if (attendanceData[participantStatus]) {
            attendanceData[participantStatus].data[orgIndex] += 1;
            attendanceData[participantStatus].total += 1;
          }
        }
      });
    }
  });

  const responseData = {
    data: Object.values(attendanceData),
    xaxis: organizationNames,
  };
  return responseData;
};

const getFinancialYear = (dateString) => {
  const date = new Date(dateString);
  const year = date.getFullYear();
  const month = date.getMonth() + 1; // JavaScript months start at 0

  if (month >= 4) {
    return `${year}/${(year + 1).toString().slice(-2)}`; // e.g., April 2024 → "2024/25"
  } else {
    return `${year - 1}/${year.toString().slice(-2)}`; // e.g., March 2024 → "2023/24"
  }
};

const getAttendanceByDate = async (dateFilter) => {
  const [meetingRecords] = await db.query(
    `SELECT created_at, participants FROM meeting_recording WHERE deleted = "0"`
  );

  let attendanceData = {};

  const defaultStatuses = ["present", "absent", "apologised", "Not Set"];

  meetingRecords.forEach((row) => {
    if (row.participants) {
      const participants = JSON.parse(row.participants);
      const financialYear = getFinancialYear(row.created_at);

      // Ensure the financial year is initialized
      if (!attendanceData[financialYear]) {
        attendanceData[financialYear] = {};
        defaultStatuses.forEach((status) => {
          attendanceData[financialYear][status] = 0;
        });
      }

      // Count attendance by status
      participants.forEach((p) => {
        const participantStatus = defaultStatuses.includes(p.status)
          ? p.status
          : "Not Set";
        if (attendanceData[financialYear][participantStatus] !== undefined) {
          attendanceData[financialYear][participantStatus] += 1;
        }
      });
    }
  });

  // Convert data into structured format
  let structuredData = {
    data: [],
    xaxis: Object.keys(attendanceData).sort(), // Sorting years for better visualization
  };

  defaultStatuses.forEach((status) => {
    let statusData = {
      status,
      data: structuredData.xaxis.map(
        (year) => attendanceData[year][status] || 0
      ),
      total: structuredData.xaxis.reduce(
        (sum, year) => sum + (attendanceData[year][status] || 0),
        0
      ),
    };
    structuredData.data.push(statusData);
  });

  return structuredData;
};

const getPlannedVsCompletedAttendance = async (dateFilter) => {
  const [plannedMeetings] = await db.query(
    `SELECT created_at FROM meeting WHERE deleted = "0"`
  );

  const [completedMeetings] = await db.query(
    `SELECT created_at FROM meeting_recording WHERE deleted = "0";`
  );

  let meetingData = {};

  // Process planned meetings
  plannedMeetings.forEach((row) => {
    const financialYear = getFinancialYear(row.created_at);
    if (!meetingData[financialYear]) {
      meetingData[financialYear] = { planned: 0, completed: 0 };
    }
    meetingData[financialYear].planned += 1;
  });

  // Process completed meetings
  completedMeetings.forEach((row) => {
    const financialYear = getFinancialYear(row.created_at);
    if (!meetingData[financialYear]) {
      meetingData[financialYear] = { planned: 0, completed: 0 };
    }
    meetingData[financialYear].completed += 1;
  });

  let structuredData = {
    data: [
      {
        name: "Planned Meetings",
        data: [],
      },
      {
        name: "Completed Meetings",
        data: [],
      },
    ],
    xaxis: Object.keys(meetingData).sort(),
  };

  structuredData.xaxis.forEach((year) => {
    structuredData.data[0].data.push(meetingData[year].planned);
    structuredData.data[1].data.push(meetingData[year].completed);
  });

  return structuredData;
};

const getOverduePlannedMeetings = async (dateFilter) => {
  const today = moment();

  const [meetings] = await db.query(
    `SELECT meeting.id, meeting.planned_meeting_date_to 
    FROM meeting 
    LEFT JOIN meeting_recording mr ON meeting.id = mr.scheduled_meeting 
    WHERE mr.scheduled_meeting IS NULL 
    AND meeting.planned_meeting_date_to IS NOT NULL;`
  );
  // Initialize overdue count object
  let overdueData = {
    "0 to 30 Days": 0,
    "31 to 60 Days": 0,
    "61 to 90 Days": 0,
    "90+ Days": 0,
    Overdue: 0,
  };

  meetings.forEach((row) => {
    const meetingDate = moment(row.planned_meeting_date_to);
    const diffDays = today.diff(meetingDate, "days");

    if (diffDays > 0) {
      if (diffDays <= 30) {
        overdueData["0 to 30 Days"] += 1;
      } else if (diffDays <= 60) {
        overdueData["31 to 60 Days"] += 1;
      } else if (diffDays <= 90) {
        overdueData["61 to 90 Days"] += 1;
      } else {
        overdueData["90+ Days"] += 1;
      }
      overdueData["Overdue"] += 1;
    }
  });

  return {
    labels: Object.keys(overdueData),
    data: Object.values(overdueData),
  };
};
