import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  insertActivityLog,
  searchConditionRecord,
  makeJoins,
  whereCondition,
  countQueryCondition,
  settingsUpdater,
  deleteSettingRecord,
  encodeSingle_statement,
  decodeSingle_statement,
  uniqueIdGenerator,
} from "../helper/general.js";
import InspectionLocation from "../sequelize/InspectionLocationSchema.js";

/** Function to create or update InspectionLocation */
export const createUpdateInspectionLocation = async (req, res) => {
  const { id, name, description } = req.body;
  const encodedDescription = await encodeSingle_statement(description);

  if (id) {
    const result = await settingsUpdater(
      InspectionLocation,
      req.body.organization,
      req.body,
      req.user
    );
    if (!result) {
      return sendResponse(res, 404, "No data found");
    }
    await insertActivityLog(
      req.user.sessionid,
      "update",
      "InspectionLocation",
      `This user updated InspectionLocation with id: ${id}`
    );
    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const insertPromises = req.body.organization.map(async (org) => {
      const [checkName] = await db.query(
        `SELECT name FROM inspection_location WHERE name = ? AND deleted = 0 AND organization = ?`,
        [name, org]
      );
      if (checkName.length === 0) {
         const unique_id = await uniqueIdGenerator(
            org,
            req?.body?.department,
            "INSL",
            "inspection_location",
            "unique_id",
            "unique_id"
          );
         
        await db.query(
          `INSERT INTO inspection_location (type, name, description, created_by, organization,unique_id) VALUES (? ,?, ?, ?, ?, ?)`,
          [req.body.type, name, encodedDescription, req.user.sessionid, org,unique_id]
        );
        await insertActivityLog(
          req.user.sessionid,
          "create",
          "InspectionLocation",
          `This user created a new InspectionLocation '${name}' for organization ${org}`
        );
      }
    });
    await Promise.all(insertPromises);
    return sendResponse(res, 200, "Record created successfully");
  }
};

/** Function to view all or single InspectionLocation */
export const viewInspectionLocation = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "inspection_location",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  const searchFields = [
    "inspection_location.name",
    "inspection_location.description",
    "users.name",
    "organization.name",
  ];
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchFields
  );
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = inspection_location.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = inspection_location.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const fetchQuery = `SELECT inspection_location.*, CONCAT(users.name, ' ', users.surname) AS created_by, organization.name AS organization_name FROM inspection_location ${joinsRecord} WHERE inspection_location.deleted = 0 ${searchCondition} ${condition}`;
  let [records] = await db.query(fetchQuery);

  for (const record of records) {
    record.description = await decodeSingle_statement(record.description);
    if (req.query.grouped == "true") {
      const [organizations] = await db.query(
        `SELECT organization FROM inspection_location WHERE deleted = 0 AND name = ?`,
        [record.name]
      );
      const arr = organizations.map((item) => item.organization);
      record.organizations = arr;
    }
  }

  const totalRecord = await countQueryCondition(fetchQuery);
  return sendResponse(res, 200, records, totalRecord);
};

/** Function to delete InspectionLocation */
export const deleteInspectionLocation = async (req, res) => {
  const { id } = req.params;
  const deleteRecord = await deleteSettingRecord("inspection_location", id);
  if (deleteRecord) {
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "InspectionLocation",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  } else {
    return sendResponse(res, 404, "Record not found");
  }
};
