import db from "../db-config.js";
import {
  encodeSingle_statement,
  decodeSingle_statement,
  getOrganizationAccordingToDepartment,
  uploadFile,
  updateQueryBuilder,
  createQueryBuilder,
  whereCondition,
  searchConditionRecord,
  makeJoins,
  countQueryCondition,
  decodeAndParseFields,
  insertActivityLog,
  deleteRecord,
  getRecord,
  getKnowledgeComment,
  getSupervisor,
  processesSingleDDRMDocument,
  insertNotification,
  uniqueIdGenerator,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import KnowledgeComment from "../sequelize/knowledgeCommentSchema.js";
import KnowledgeManagementRecording from "../sequelize/knowledgeManagementRecordingsSchema.js";

export const createUpdateKnowledgeManagementRecording = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];
  const { id, description, comments, department, sidebar_id = 218 } = req.body;

  let reviewer = req.body?.reviewer || [];
  reviewer = reviewer.map((element) => element?.user_id);
  if (reviewer.length == 0) {
    return sendResponse(res, 200, "At least one Reviewer is required");
  }
  req.body.reviewer = reviewer;
  if (req.body.knowledge_management_type == 0) {
    req.body.current_status = "reviewing";
  } else {
    req.body.current_status = "approved";
  }
  const status = id ? "updated" : "created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }

  if (!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "KnowledgeManagementRecording",
      "knowledge_management_recording",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  if (req.files && req.files.supporting_document) {
    const file = req.files.supporting_document;
    const ddrm_id = await processesSingleDDRMDocument(
      "knowledge_management_recording",
      sidebar_id,
      file,
      req
    );
    req.body.ddrm_id = ddrm_id;
  }

  req.body.description = await encodeSingle_statement(description);
  req.body.comments = await encodeSingle_statement(comments);
  req.body.keywords = JSON.stringify(req.body.keywords);
  const { query, values } = id
    ? updateQueryBuilder(KnowledgeManagementRecording, req.body)
    : createQueryBuilder(KnowledgeManagementRecording, req.body);
  await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    "create",
    "Knowledge Management Recording",
    `This user ${status} a new Knowledge Management Recording Record for organization ${req.body.organization}`
  );
  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getKnowledgeManagementRecording = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "knowledge_management_recording",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = [
    "knowledge_management_recording.lesson_best_practice_id",
    "knowledge_management_recording.title",
    "knowledge_management_recording.description",
    "knowledge_management_recording.recommendations",
    "knowledge_management_recording.key_insights",
    "knowledge_management_recording.keywords",
    "knowledge_management_recording.implementation_steps",
    "knowledge_management_recording.comments",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = knowledge_management_recording.organization",
    },
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = knowledge_management_recording.created_by",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = knowledge_management_recording.department",
    },
    {
      type: "left",
      targetTable: "users AS u1",
      onCondition: "u1.id = knowledge_management_recording.author",
    },
    {
      type: "left",
      targetTable: "file_classification",
      onCondition:
        "file_classification.id = knowledge_management_recording.file_classification",
    },
    {
      type: "left",
      targetTable: "best_practice",
      onCondition: "best_practice.id = knowledge_management_recording.category",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = knowledge_management_recording.ddrm_id",
    },
    {
      type: "left",
      targetTable: "users as approver",
      onCondition: "approver.id = knowledge_management_recording.approver",
    },
    {
      type: "left",
      targetTable: "users as author",
      onCondition: "author.id = knowledge_management_recording.author",
    },
  ];

  const joinCondition = await makeJoins(joins);
  let knowledgeComments = [];
  const knowledgeManagementRecordingQuery = `SELECT knowledge_management_recording.* , repository.url as  supporting_document , organization.name AS organization_name , department.name AS department_name , users.name AS created_by_name , users.surname as created_by_surname , users.profile as created_by_profile  , CONCAT(u1.name , ' '  , u1.surname) AS author_name , u1.profile AS authors_profile, file_classification.name AS file_classification_name ,
  best_practice.name as category_name, CONCAT(approver.name , ' ' , approver.surname) as approver_name, approver.profile as approver_profile, CONCAT(author.name , ' ' , author.surname) as author_name, author.profile as author_profile FROM knowledge_management_recording ${joinCondition} WHERE knowledge_management_recording.deleted = 0 ${searchCondition} ${condition} `;

  let [knowledgeManagementRecording] = await db.query(
    knowledgeManagementRecordingQuery
  );

  knowledgeManagementRecording = await decodeAndParseFields(
    knowledgeManagementRecording
  );
  knowledgeManagementRecording = knowledgeManagementRecording.map(
    async (element) => {
      if(element.authors_profile) {
        element.authors_profile = `${process.env.HARMONY_BACKEND_URL}/${element.authors_profile.replace(/\\/g, "/")}`;
      }
      if(element.approver_profile) {
        element.approver_profile = `${process.env.HARMONY_BACKEND_URL}/${element.approver_profile.replace(/\\/g, "/")}`;
      }
      const stakeholder = element?.stakeholder || [];
      let stakeholder_list = [];
      if (stakeholder.length > 0) {
        [stakeholder_list] = await db.query(
          `SELECT id , CONCAT(name , ' ' , surname) as name from users WHERE id in (${stakeholder})`
        );
      }
      let stakeholder_name = stakeholder_list.map((element) => element.name);
      element.stakeholder_name = stakeholder_name;
      const reviewers = element?.reviewer || [];
      // console.log("element: ", reviewers);
      for (let i = 0; i < reviewers.length; i++) {
        const reviewer = reviewers[i];
        let employeeDetails = [];
        if (reviewer) {
          [employeeDetails] = await db.query(
            `SELECT users.id as user_id , CONCAT(users.name , ' ' , users.surname) as name , roles.name as role_name, department.name as department from users Left join roles on roles.id = users.role Left join department on department.id = users.department WHERE users.id = ${reviewer} AND users.deleted = "0"`
          );
        }

        reviewers[i] = { ...employeeDetails[0] };
      }
      return {
        ...element,
      };
    }
  );

  knowledgeManagementRecording = await Promise.all(
    knowledgeManagementRecording
  );
  if (id) knowledgeComments = await getKnowledgeComment(id);
  /**Count of all Policy */
  const dataWithComments = knowledgeManagementRecording.map((item) => {
    let isCommented = false;
    return {
      ...item,
      comments_by_users: knowledgeComments.filter((comment) => {
        if (req.user.sessionid === comment.created_by) {
          isCommented = true;
        }
        if (
          req.user.sessionid === item.approver ||
          req.user.sessionid === item.created_by ||
          req.user.isSuperAdmin == 1
        ) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            reject_reason: comment.reject_reason,
            steps: comment.steps,
            status: comment.status,
          };
        } else if (req.user.sessionid === comment.created_by) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            reject_reason: comment.reject_reason,
            steps: comment.steps,
            status: comment.status,
          };
        }
      }),
      isCommented,
    };
  });
  const totalRecord = await countQueryCondition(
    knowledgeManagementRecordingQuery
  );

  return sendResponse(res, 200, dataWithComments, totalRecord);
};

export const deleteKnowledgeManagementRecording = async (req, res) => {
  const { id } = req.params;
  await deleteRecord("knowledge_management_recording", id);

  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Knowledge Management Recording",
    `This user deleted a new Knowledge Management Recording Record for organization ${id}`
  );

  return sendResponse(res, 200, "Record deleted successfully");
};

export const getKnowledgeManagementRecordingApprovalWorkflow = async (
  req,
  res
) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "knowledge_management_recording",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const approvalCondition = req.user.isSuperAdmin
    ? ""
    : `AND (JSON_CONTAINS(knowledge_management_recording.reviewer, "${req.user.sessionid}")  OR knowledge_management_recording.approver = ${req.user.sessionid}) `;
  const searchTableName = [
    "knowledge_management_recording.lesson_best_practice_id",
    "knowledge_management_recording.title",
    "knowledge_management_recording.description",
    "knowledge_management_recording.recommendations",
    "knowledge_management_recording.key_insights",
    "knowledge_management_recording.keywords",
    "knowledge_management_recording.implementation_steps",
    "knowledge_management_recording.comments",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = knowledge_management_recording.organization",
    },
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = knowledge_management_recording.created_by",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = knowledge_management_recording.department",
    },
    {
      type: "left",
      targetTable: "users AS u1",
      onCondition: "u1.id = knowledge_management_recording.author",
    },
    {
      type: "left",
      targetTable: "file_classification",
      onCondition:
        "file_classification.id = knowledge_management_recording.file_classification",
    },
    {
      type: "left",
      targetTable: "best_practice",
      onCondition: "best_practice.id = knowledge_management_recording.category",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = knowledge_management_recording.ddrm_id",
    },
  ];
  let knowledgeComments = [];
  const joinCondition = await makeJoins(joins);

  const knowledgeManagementRecordingQuery = `SELECT knowledge_management_recording.* , repository.url as  supporting_document , organization.name AS organization_name , department.name AS department_name , users.name AS created_by_name , users.surname as created_by_surname , users.profile as created_by_profile  , CONCAT(u1.name , ' '  , u1.surname) AS author_name , u1.profile AS authors_profile, file_classification.name AS file_classification_name ,
  best_practice.name as category_name FROM knowledge_management_recording ${joinCondition} WHERE knowledge_management_recording.deleted = 0 AND knowledge_management_recording.knowledge_management_type = 0 ${approvalCondition} ${searchCondition} ${condition}`;

  let [knowledgeManagementRecording] = await db.query(
    knowledgeManagementRecordingQuery
  );

  knowledgeManagementRecording = await decodeAndParseFields(
    knowledgeManagementRecording
  );
  knowledgeManagementRecording = knowledgeManagementRecording.map(
    async (element) => {
      const stakeholder = element?.stakeholder || [];
      let stakeholder_list = [];
      if (stakeholder.length > 0) {
        [stakeholder_list] = await db.query(
          `SELECT id , CONCAT(name , ' ' , surname) as name from users WHERE id in (${stakeholder})`
        );
      }
      let stakeholder_name = stakeholder_list.map((element) => element.name);
      element.stakeholder_name = stakeholder_name;
      const reviewers = element?.reviewer || [];
      // console.log("element: ", reviewers);
      for (let i = 0; i < reviewers.length; i++) {
        const reviewer = reviewers[i];
        let employeeDetails = [];
        if (reviewer) {
          [employeeDetails] = await db.query(
            `SELECT users.id as user_id , CONCAT(users.name , ' ' , users.surname) as name , roles.name as role_name, department.name as department from users Left join roles on roles.id = users.role Left join department on department.id = users.department WHERE users.id = ${reviewer} AND users.deleted = "0"`
          );
        }

        reviewers[i] = { ...employeeDetails[0] };
      }

      return {
        ...element,
      };
    }
  );

  knowledgeManagementRecording = await Promise.all(
    knowledgeManagementRecording
  );

  if (id) knowledgeComments = await getKnowledgeComment(id);

  /**Count of all Policy */
  const dataWithComments = knowledgeManagementRecording.map((item) => {
    let isCommented = false;
    return {
      ...item,
      comments_by_users: knowledgeComments.filter((comment) => {
        if (req.user.sessionid === comment.created_by) {
          isCommented = true;
        }
        if (
          req.user.sessionid === item.approver ||
          req.user.sessionid === item.created_by ||
          req.user.isSuperAdmin == 1
        ) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            reject_reason: comment.reject_reason,
            steps: comment.steps,
            status: comment.status,
          };
        } else if (req.user.sessionid === comment.created_by) {
          return {
            id: comment.created_by,
            name: comment.name,
            profile: comment.profile,
            date: comment.created_at,
            comments: comment.comments,
            reject_reason: comment.reject_reason,
            steps: comment.steps,
            status: comment.status,
          };
        }
      }),
      isCommented,
    };
  });

  const totalRecord = await countQueryCondition(
    knowledgeManagementRecordingQuery
  );

  return sendResponse(res, 200, dataWithComments, totalRecord);
};

/**Function to create comments on knowledge management */
export const createComment = async (req, res) => {
  let {
    status,
    approver,
    knowledge_recording_id,
    steps: implementation_steps,
    comments,
  } = req.body;
  if (!status) {
    return sendResponse(res, 400, "Status is required");
  }
  const [record] = await getRecord(
    "knowledge_management_recording",
    "id",
    knowledge_recording_id
  );

  if (!approver && status == "reviewing") {
    const reviewerList = JSON.parse(record?.reviewer || "[]");
    if (reviewerList.includes(req.user.sessionid)) {
      const supervisor_id = await getSupervisor(req.user.sessionid);
      console.log("supervisor_id: ", supervisor_id);
      if (!supervisor_id) {
        return sendResponse(res, 400, "No Supervisor Found For This User");
      }
      // make the direct manger or supervisor as reviewer of the record
      reviewerList.push(supervisor_id);
      await db.query(
        "Update knowledge_management_recording set reviewer = ? , current_status = 'reviewing' where id = ?",
        [JSON.stringify(reviewerList), knowledge_recording_id]
      );
      return sendResponse(
        res,
        200,
        "Record Sent to Your supervisor for Review"
      );
    }
  }

  if (approver && status === "approval") {
    const reviewerList = JSON.parse(record?.reviewer || "[]");
    if (!reviewerList.includes(req.user.sessionid) && !req.user.isSuperAdmin) {
      return sendResponse(res, 400, "Access Denied");
    } else {
      await db.query(
        "Update knowledge_management_recording set approver = ? , current_status = 'approval' where id = ?",
        [approver, req.body.knowledge_recording_id]
      );

      // notify the user that the record is sent for approval
      const url = ``;
      await insertNotification(
        "Record Sent for Approval",
        url,
        approver,
        "url",
        req.body.sessionid
      );
      return sendResponse(res, 200, "Record Sent for Approval");
    }
  }
  req.body.created_by = req.user.sessionid;
  const { query, values } = createQueryBuilder(KnowledgeComment, req.body);
  if (status == "rejected") {
    /** Update the knowledge status on the user basis if reviewer to draft or if approver then reject it*/
    const reviewer = JSON.parse(record?.reviewer || "[]");

    if (reviewer.includes(req.user.sessionid)) {
      await db.query(
        "Update knowledge_management_recording set current_status = 'rejected' where id = ?",
        [knowledge_recording_id]
      );
    }
    if (record?.approver == req.user.sessionid || req.user.isSuperAdmin == 1) {
      await db.query(
        "Update knowledge_management_recording set current_status = 'rejected' where id = ?",
        [knowledge_recording_id]
      );
    }
  }

  if (status == "approved") {
    if (record?.approver == req.user.sessionid || req.user.isSuperAdmin == 1) {
      await db.query(
        "Update knowledge_management_recording set current_status = 'approved' where id = ?",
        [knowledge_recording_id]
      );
      return sendResponse(res, 200, "Record Approved");
    }
  }
  const insertedData = await db.query(query, values);
  return sendResponse(res, 200, "Comment on knowledge management stored");
};
