import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  decodeSingle_statement,
  deleteRecord,
  getCompanyName,
  getOrganizationAccordingToDepartment,
  getRecord,
  getUserDetails,
  insertActivityLog,
  insertMeetingActionNotes,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import MeetingRecording from "../sequelize/MeetingRecordingSchema.js";
import { sendResponse } from "../helper/wrapper.js";
import moment from "moment";
import sendEmail from "../helper/sendEmail.js";
import { nanoid } from "nanoid";
import Joi from "joi";
import { meetingNotesStatusChangeSchema } from "../validationsRecord/validationSchema.js";

/**Function to create and update Meeting Recording  */
export const createUpdateMeetingRecording = async (req, res) => {
  let {
    id,
    scheduled_meeting,
    approval_status,
    meeting_notes = [],
    organization,
    department,
  } = req.body;
  /**Check record if organization is not coming then fetch organization according to department */

  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }
  meeting_notes = meeting_notes.map((note) => ({
    ...note,
    actions: note.actions.map((action) => ({
      ...action,
      action_id: nanoid(),
    })),
  }));
  req.body.meeting_notes = meeting_notes;
  /**If id comes in body then it will update the query */
  if (id) {
    if (approval_status) {
      await db.query(
        `UPDATE meeting_recording SET  status = '${approval_status}' WHERE id = ${id}`
      );
    } else {
      /**Update record for Meeting Recording */
      req.body.updated_by = req.user.sessionid;
      const { query, values } = updateQueryBuilder(MeetingRecording, req.body);

      await db.query(query, values);
      // await insertMeetingActionNotes(
      //   scheduled_meeting,
      //   req.user.sessionid,
      //   meeting_notes,
      //   id
      // );
    }

    /**Insert Activity  */
    insertActivityLog(req.user.sessionid, "update", "Meeting Recording ", id);

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const [meeting] = await db.query(
      `SELECT * FROM meeting_recording WHERE scheduled_meeting = ${scheduled_meeting}`
    );
    if (meeting.length > 0) {
      return sendResponse(res, 400, "Meeting already recorded");
    }
    const unique_id = await uniqueIdGenerator(
      organizationId,
      department,
      "MR",
      "meeting_recording",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
    req.body.created_by = req.user.sessionid;
    /**Insert record for Meeting Recording */
    const { query, values } = createQueryBuilder(MeetingRecording, req.body);
    const [createMeetingRecording] = await db.query(query, values);

    // send email to participants or assigned
    let meetingRecordData = await getMeetingRecordById(
      createMeetingRecording.insertId
    );
    let actionData = [];

    meetingRecordData[0]?.notes?.forEach((note) => {
      if (note.actions && Array.isArray(note.actions)) {
        const filteredActions = note?.actions?.map((action) => ({
          action_discussion_name: action.action_discussion_name,
          due_date: action.due_date,
          assigned_name: action?.assigned_name,
        }));

        actionData.push(...filteredActions);
      }
    });

    const companyDetails = await getCompanyName(
      meetingRecordData[0].organization
    );

    const actualStartDate = moment(
      meetingRecordData[0].actual_meeting_date_from
    ).format("YYYY-MM-DD");
    const actualStartTime = moment(
      meetingRecordData[0].actual_meeting_date_from
    ).format("hh:mm A");
    const actualEndDate = moment(
      meetingRecordData[0].actual_meeting_date_to
    ).format("YYYY-MM-DD");
    const actualEndTime = moment(
      meetingRecordData[0].actual_meeting_date_to
    ).format("hh:mm A");

    let meetingScriber = `${meetingRecordData[0]?.meeting_scriber} ${meetingRecordData[0]?.meeting_scriber_surname}`;
    for (let user of meetingRecordData[0].participants) {
      const sendRecordArray = {
        templateFileUrl: "mail_for_meeting_actions_assigned_template.html",
        templateName: "Meeting Action",
        action_owner: `${user?.name} ${user?.surname}`,
        meeting_name: meetingRecordData[0].meeting_title,
        actual_start_date: actualStartDate,
        actual_start_time: actualStartTime,
        actual_end_time: actualEndTime,
        actionData: `
           <table border="1" cellspacing="0" cellpadding="5" style="width:100%; border-collapse: collapse;">
              <thead>
                <tr style="background-color: #004080; color: white;">
                  <th>Action Discussion</th>
                  <th>Allocated To</th>
                  <th>Due Date</th>
                </tr>
              </thead>
              <tbody>
              ${actionData
                .map(
                  (action) => `
                  <tr>
                    <td>${action?.action_discussion_name || " "}</td>
                    <td>${action?.assigned_name || " "}</td>
                    <td>${action?.due_date || " "}</td>
                  </tr>
                `
                )
                .join("")}
              </tbody>
            </table>
          `,
        meeting_scriber: meetingScriber,
        meeting_scriber_email: meetingRecordData[0].meeting_scriber_email,
        company_name: companyDetails?.name,
        company_website: companyDetails?.business_logo,
      };

      const info = await sendEmail(
        "info@harmonyandhelp.com",
        user.email,
        `Action Items Assigned: ${meetingRecordData[0].meeting_title}`,
        sendRecordArray
      );
    }

    /**Insert record for activity log */
    insertActivityLog(
      req.user.sessionid,
      "create",
      "Meeting Recording ",
      createMeetingRecording.insertId
    );

    // await insertMeetingActionNotes(
    //   scheduled_meeting,
    //   req.user.sessionid,
    //   meeting_notes,
    //   createMeetingRecording.insertId
    // );

    return sendResponse(res, 200, "Record Created Successfully");
  }
};

/**Function to view all  meeting recording */
export const viewAllMeetingRecording = async (req, res) => {
  const condition = await whereCondition({
    table: "meeting_recording",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id: req.params.id,
    user: req.user,
  });

  /** If value come with any search condition then search that word */
  const searchTableName = [
    "meeting_recording.status",
    "meeting.meeting_title",
    "meeting.venue",
    "meeting.meeting_hierarchy",
    "organization.name",
    "meeting.meeting_title",
    "meeting.meeting_title",
    "CONCAT(meetingScriber.name , ' ' , meetingScriber.surname)",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = meeting_recording.created_by",
    },
    {
      type: "left",
      targetTable: "meeting",
      onCondition: "meeting.id = meeting_recording.scheduled_meeting",
    },
    {
      type: "left",
      targetTable: "users as meetingOwner",
      onCondition: "meetingOwner.id = meeting.meeting_owner",
    },
    {
      type: "left",
      targetTable: "users as meetingScriber",
      onCondition: "meetingScriber.id = meeting_recording.meeting_scriber",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = meeting_recording.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = meeting_recording.department",
    },
    {
      type: "left",
      targetTable: "meeting_notes",
      onCondition: "meeting_notes.meeting_id = meeting_recording.department",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all meetings */
  const meetingsQuery = `SELECT meeting_recording.unique_id,meeting_recording.meeting_notes,meeting_recording.external_meeting_guest,meeting_recording.discussion_have_action,meeting_recording.external_guest_details,meeting_recording.status as approval_status , meeting_recording.participants , meetingScriber.name as meeting_scriber,meetingScriber.surname as meeting_scriber_surname,meetingScriber.profile as meeting_scriber_profile , meeting_recording.scheduled_meeting , meeting_recording.organization , meeting_recording.meeting_scriber as meeting_scriber_id ,  meetingOwner.name as meeting_owner , meetingOwner.surname as meeting_owner_surname , meetingOwner.profile as  meeting_owner_profile , meeting.meeting_title ,
    meeting.meeting_hierarchy,meeting_recording.actual_meeting_date_from ,meeting_recording.actual_meeting_date_to , meeting_recording.meeting_duration , meeting.venue , meeting.planned_meeting_date_from,meeting.planned_meeting_date_to,meeting.meeting_owner as meeting_owner_id , organization.name as organization_name, organization.business_logo , 
    organization.background_image, organization.header_image, organization.footer_image, organization.font_size, organization.pick_color, meeting_recording.id,meeting.meeting_id as meeting_id,meeting_recording.created_by as created_by_id,users.name as created_by ,users.surname as created_by_surname , users.profile as created_by_profile
     FROM meeting_recording 
     ${joinsRecord} where meeting_recording.deleted = 0 ${searchCondition} ${condition}`;
  const [meetings] = await db.query(meetingsQuery);

  /**Get meeting notes according to meeting id */
  for (const meetingsNotes of meetings) {
    meetingsNotes.notes = JSON.parse(meetingsNotes.meeting_notes);
    meetingsNotes.external_guest_details = JSON.parse(
      meetingsNotes.external_guest_details
    );
    // /** Structuring for meeting participants */
    // console.log("meetingasdasd",meetingsNotes)
    const participants = JSON.parse(meetingsNotes.participants);
    const participantsArray = [];

    for (const participant of participants) {
      const [user] = await db.query(`SELECT users.name, 
        users.id, 
        users.surname, 
        users.phone,
        users.unique_id,
        roles.name as role_name,
        organization.name as organization_name,
        users.profile FROM users
        LEFT JOIN roles ON roles.id = users.role
        LEFT JOIN organization ON organization.id = users.my_organization
        where users.id = ${participant.id} AND users.deleted = "0"`);
      participantsArray.push({ ...participant, ...user[0] });
    }

    meetingsNotes.participants = participantsArray;
  }
  const result = meetings;

  const totalRecord = await countQueryCondition(meetingsQuery);

  return sendResponse(res, 200, result, totalRecord);
};

/**Function to delete a specific Meeting */
export const deleteMeetingRecording = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("meeting_recording", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(req.user.sessionid, "delete", "Meeting Recording", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

/**Function to view all  meeting recording */
export const viewAllMeetingRecordingById = async (req, res) => {
  const condition = await whereCondition({
    table: "meeting",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id: req.params.meetingId,
    user: req.user,
    columnName: "meeting_id",
  });

  /** If value come with any search condition then search that word */
  const searchTableName = [
    "meeting.meeting_title",
    "meeting.venue",
    "meeting.meeting_hierarchy",
    "organization.name",
    "meeting.meeting_title",
    "meeting.meeting_title",
    "CONCAT(u1.name , ' ' , u1.surname)",
    "meeting_notes.meeting_discussion_name",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = meeting_recording.created_by",
    },
    {
      type: "left",
      targetTable: "meeting",
      onCondition: "meeting.id = meeting_recording.scheduled_meeting",
    },
    {
      type: "left",
      targetTable: "users as meetingOwner",
      onCondition: "meetingOwner.id = meeting.meeting_owner",
    },
    {
      type: "left",
      targetTable: "users as meetingScriber",
      onCondition: "meetingScriber.id = meeting_recording.meeting_scriber",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = meeting_recording.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = meeting_recording.department",
    },
    {
      type: "left",
      targetTable: "meeting_notes",
      onCondition: "meeting_notes.meeting_id = meeting_recording.department",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all meetings */
  const meetingsQuery = `SELECT meeting.meeting_hierarchy,meeting_recording.status,meeting_recording.participants,meeting_recording.external_meeting_guest,meeting_recording.external_guest_details, meeting_recording.unique_id, meetingScriber.name as meeting_scriber , meeting_recording.scheduled_meeting , meeting_recording.organization , meeting_recording.meeting_scriber as meeting_scriber_id ,  meetingOwner.name as meeting_owner , meeting.meeting_title ,meeting_recording.actual_meeting_date_from ,meeting_recording.actual_meeting_date_to , meeting_recording.meeting_duration,meeting_recording.meeting_notes, meeting.venue , meeting.planned_meeting_date_from,meeting.planned_meeting_date_to,meeting.meeting_owner as meeting_owner_id , organization.name as organization_name, meeting_recording.id,meeting.meeting_id as meeting_id,meeting_recording.created_by as created_by_id,users.name as created_by
     FROM meeting_recording 
     ${joinsRecord} where meeting_recording.deleted = 0 ${searchCondition} ${condition}`;
  const [meetings] = await db.query(meetingsQuery);
  /**Get meeting notes according to meeting id */
  for (const meetingsNotes of meetings) {
    meetingsNotes.notes = JSON.parse(meetingsNotes.meeting_notes);
    meetingsNotes.external_guest_details = JSON.parse(
      meetingsNotes.external_guest_details
    );
    // /** Structuring for meeting participants */
    const participants = JSON.parse(meetingsNotes.participants);
    const participantsArray = [];

    for (const participant of participants) {
      const [user] = await db.query(`SELECT users.name, 
        users.id, 
        users.surname, 
        users.phone,
        users.unique_id,
        roles.name as role_name,
        organization.name as organization_name,
        users.profile FROM users
        LEFT JOIN roles ON roles.id = users.role
        LEFT JOIN organization ON organization.id = users.my_organization
        where users.id = ${participant.id} AND users.deleted = "0"`);
      participantsArray.push({ ...participant, ...user[0] });
    }

    meetingsNotes.participants = participantsArray;
  }
  const result = meetings;
  const totalRecord = await countQueryCondition(meetingsQuery);
  return sendResponse(res, 200, result, totalRecord);
};

/**Function to view all  meeting recording */
export const getMeetingRecordActionItemOld = async (req, res) => {
  const condition = await whereCondition({
    table: "meeting",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id: req.params.id,
    user: req.user,
  });

  /** If value come with any search condition then search that word */
  const searchTableName = [
    "meeting.meeting_title",
    "meeting.venue",
    "meeting.meeting_hierarchy",
    "organization.name",
    "meeting.meeting_title",
    "meeting.meeting_title",
    "CONCAT(u1.name , ' ' , u1.surname)",
    "meeting_notes.meeting_discussion_name",
    "meeting_notes.meeting_discussion",
    "meeting_notes.meeting_action_items",
    "CONCAT(raised.name , ' ' , raised.surname)",
    "CONCAT(meeting_owner.name , ' ' , meeting_owner.surname)",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = meeting_notes.assigned_to",
    },
    {
      type: "left",
      targetTable: "users as raised",
      onCondition: "raised.id = meeting_notes.raised_by",
    },
    {
      type: "left",
      targetTable: "meeting",
      onCondition: "meeting.id = meeting_notes.meeting_id",
    },
    {
      type: "left",
      targetTable: "meeting_recording",
      onCondition:
        "meeting_recording.scheduled_meeting = meeting_notes.meeting_id",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = meeting_recording.meeting_scriber",
    },
    {
      type: "left",
      targetTable: "users as meeting_owner",
      onCondition: "meeting_owner.id = meeting.meeting_owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = meeting_recording.organization",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all meetings */
  const meetingsNotesQuery = `SELECT meeting_recording.actual_meeting_date_from,
     meeting_recording.meeting_scriber,
     u1.name as meeting_scriber, 
     meeting_owner.name as meeting_chairperson,
     meeting_owner.surname as meeting_chairperson_surname,
     meeting.venue,
     meeting_notes.notes_time, raised.name as raised_by  , raised.surname as raised_by_surname , 
     raised.profile as raised_by_profile, meeting_notes.raised_by as  raised_by_id ,  
     meeting.meeting_id,meeting.meeting_title,meeting_notes.id,
     meeting_notes.agenda_item,meeting_notes.meeting_discussion_name,
     meeting_notes.completed_date,
     meeting_recording.organization,
     organization.name as organization_name,
     meeting_notes.meeting_discussion,meeting_notes.action_discussion_name,meeting_notes.meeting_action_items ,meeting_notes. meeting_action_description ,meeting_notes. assigned_to as assigned_to_id , users.name as assigned_to , users.surname as assigned_to_surname ,users.profile as assigned_to_profile ,meeting_notes.due_date ,meeting_notes.meeting_action_progress ,meeting_notes.status FROM meeting_notes
     ${joinsRecord} where meeting_notes.deleted = 0 ${searchCondition} ${condition}`;
  const [meetingsNotes] = await db.query(meetingsNotesQuery);
  for (const notes of meetingsNotes) {
    notes.meeting_discussion = await decodeSingle_statement(
      notes.meeting_discussion
    );
    notes.meeting_action_description = await decodeSingle_statement(
      notes.meeting_action_description
    );
  }
  const totalRecord = await countQueryCondition(meetingsNotesQuery);
  return sendResponse(res, 200, meetingsNotes, totalRecord);
};

/**Function to view all  meeting recording */
export const getMeetingRecordActionItem = async (req, res) => {
  try {
    const { meeting_id: meetingRecordId, action_id: actionId } = req.query;
    const { page = 1, pageSize = 10, all = "false" } = req.query;
    const offset = (page - 1) * pageSize;

    // Fetch meeting record
    const meetingRecordingQuery = `
      SELECT meeting_recording.*, meeting.meeting_title, meeting.meeting_id,
      meeting.venue,
      organization.name as organization_name,
      meeting_owner.name as meeting_chairperson,
      meeting_owner.surname as meeting_chairperson_surname
      FROM meeting_recording 
      LEFT JOIN meeting ON meeting_recording.scheduled_meeting = meeting.id 
      LEFT JOIN users as meeting_owner on meeting.meeting_owner = meeting_owner.id
      LEFT JOIN organization on meeting_recording.organization = organization.id
      WHERE meeting_recording.deleted = 0 AND meeting_recording.id = ?
    `;
    let [meetingRecordingRecord] = await db.query(meetingRecordingQuery, [
      meetingRecordId,
    ]);

    if (!meetingRecordingRecord.length) {
      return sendResponse(res, 404, { message: "Meeting record not found." });
    }

    meetingRecordingRecord = await decodeAndParseFields(meetingRecordingRecord);
    const meetingNotes = meetingRecordingRecord[0]?.meeting_notes || [];
    let actions = [];

    for (let note of meetingNotes) {
      const raised_by = note.raised_by;
      if (raised_by) {
        const raisedByQuery = ` SELECT 
            users.name,
            users.surname,
            users.id,
            users.profile,
            roles.name AS role_name, 
            department.name AS department_name 
            FROM users 
            LEFT JOIN roles ON roles.id = users.role 
            LEFT JOIN department ON department.id = users.department 
            WHERE users.deleted = "0" AND users.id = ?
          `;
        const [raisedByDetails] = await db.query(raisedByQuery, [raised_by]);
        if (raisedByDetails.length > 0) {
          note.raised_by = raisedByDetails[0]?.name;
          note.raised_by_surname = raisedByDetails[0]?.surname;
          note.raised_by_id = raisedByDetails[0]?.id;
          note.raised_by_role_name = raisedByDetails[0]?.role_name;
          note.raised_by_department_name = raisedByDetails[0]?.department_name;
        }
      }
      if (note.actions && Array.isArray(note.actions)) {
        for (let action of note.actions) {
          if (action?.assigned_to) {
            const employeeQuery = ` SELECT 
              users.name,
              users.surname,
              users.id,
              users.profile,
              roles.name AS role_name, 
              department.name AS department_name 
              FROM users 
              LEFT JOIN roles ON roles.id = users.role 
              LEFT JOIN department ON department.id = users.department 
              WHERE users.deleted = "0" AND users.id = ?
            `;
            const [employeeDetails] = await db.query(employeeQuery, [
              action.assigned_to,
            ]);

            if (employeeDetails.length > 0) {
              action.assigned_to = employeeDetails[0]?.name;
              action.assigned_to_surname = employeeDetails[0]?.surname;
              action.assigned_to_id = employeeDetails[0]?.id;
              action.assigned_to_role_name = employeeDetails[0]?.role_name;
              action.assigned_to_department_name =
                employeeDetails[0]?.department_name;
              action.assigned_to_profile = employeeDetails[0]?.profile;
            } else {
              action.assigned_to = "";
              action.assigned_to_surname = "";
              action.assigned_to_id = "";
              action.assigned_to_role_name = "";
              action.assigned_to_department_name = "";
              action.assigned_to_profile = "";
            }
            if (actionId) {
              if (action.action_id == actionId) {
                actions.push({
                  ...action,
                  meeting_title: meetingRecordingRecord[0]?.meeting_title || "",
                  agenda_item: note.agenda_item,
                  meeting_discussion: note.meeting_discussion,
                  meeting_discussion_name: note.meeting_discussion_name,
                  meeting_id: meetingRecordingRecord[0]?.meeting_id,
                  organization_name:
                    meetingRecordingRecord[0]?.organization_name,
                  raised_by: note?.raised_by,
                  raised_by_surname: note?.raised_by_surname,
                  notes_time: note?.notes_time,
                });
              }
            } else {
              actions.push({
                ...action,
                meeting_title: meetingRecordingRecord[0]?.meeting_title || "",
                agenda_item: note.agenda_item,
                meeting_discussion: note.meeting_discussion,
                meeting_discussion_name: note.meeting_discussion_name,
                meeting_id: meetingRecordingRecord[0]?.meeting_id,
                organization_name: meetingRecordingRecord[0]?.organization_name,
                raised_by: note?.raised_by,
                raised_by_surname: note?.raised_by_surname,
                notes_time: note?.notes_time,
              });
            }
          }
        }
      }
    }

    // Apply pagination only if all is not "true"
    let responseActions = actions;
    if (all !== "true") {
      responseActions = actions.slice(offset, offset + parseInt(pageSize));
    }

    return sendResponse(res, 200, responseActions, actions.length);
  } catch (error) {
    console.error("Error fetching meeting record action items:", error);
    return sendResponse(res, 500, { message: "Internal Server Error" });
  }
};

/**Function to update notes status */
export const meetingNotesStatusChange = async (req, res) => {
  const {
    action_id: actionId,
    meeting_id: meetingRecordId,
    meeting_action_progress,
    status,
  } = req.body;

  const { error } = meetingNotesStatusChangeSchema.validate(req.body);
  if (error) {
    return res.status(400).json({ error: error.details[0].message });
  }

  const [meetingRecord] = await getRecord(
    "meeting_recording",
    "id",
    meetingRecordId
  );
  let meeting_notes = JSON.parse(meetingRecord.meeting_notes);

  meeting_notes = meeting_notes.map((note) => ({
    ...note,
    actions: Array.isArray(note.actions)
      ? note.actions.map((action) =>
          action.action_id == actionId
            ? { ...action, status, meeting_action_progress }
            : action
        )
      : note.actions,
  }));

  await db.query(
    `UPDATE meeting_recording SET meeting_notes = ? WHERE id = ?`,
    [JSON.stringify(meeting_notes), meetingRecordId]
  );
  return sendResponse(res, 200, "Record updated successfully");
};

// ! Not in use on frontend
export const updateApprovalStatus = async (req, res) => {
  /**Select all meeting approval record data  */
  const [getMeetingApprovalRecord] = await db.query(
    `SELECT id,due_date FROM meeting_notes WHERE deleted = "0" AND status != "Completed" AND due_date != "0000-00-00"`
  );

  // Function to get status based on due date
  async function getStatus(dueDate) {
    const timeDifference = dueDate.getTime() - Date.now();
    return timeDifference > 24 * 60 * 60 * 1000 ? "overdue" : "not overdue";
  }

  // Update status for records where due date is more than 24 hours from current date
  await Promise.all(
    getMeetingApprovalRecord.map(async (record) => {
      const status = await getStatus(new Date(record.due_date));
      if (status === "overdue") {
        await db.query(
          `UPDATE meeting_notes SET status = "overdue" WHERE id = ?`,
          [record.id]
        );
      }
    })
  );

  return sendResponse(res, 200, "Record updated successfully");
};

export const getMeetingRecordById = async (meetingRecordId) => {
  const condition = await whereCondition({
    table: "meeting_recording",
    id: meetingRecordId,
  });

  const searchTableName = [
    "meeting_recording.status",
    "meeting.meeting_title",
    "meeting.venue",
    "meeting.meeting_hierarchy",
    "organization.name",
    "meeting.meeting_title",
    "meeting.meeting_title",
    "CONCAT(meetingScriber.name , ' ' , meetingScriber.surname)",
    // "CONCAT(u1.name , ' ' , u1.surname)",
  ];

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = meeting_recording.created_by",
    },
    {
      type: "left",
      targetTable: "meeting",
      onCondition: "meeting.id = meeting_recording.scheduled_meeting",
    },
    {
      type: "left",
      targetTable: "users as meetingOwner",
      onCondition: "meetingOwner.id = meeting.meeting_owner",
    },
    {
      type: "left",
      targetTable: "users as meetingScriber",
      onCondition: "meetingScriber.id = meeting_recording.meeting_scriber",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = meeting_recording.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = meeting_recording.department",
    },
    {
      type: "left",
      targetTable: "meeting_notes",
      onCondition: "meeting_notes.meeting_id = meeting_recording.department",
    },
  ];

  const joinsRecord = await makeJoins(joins);
  const meetingsQuery = `SELECT meeting_recording.unique_id,meeting_recording.meeting_notes,
  meeting_recording.external_meeting_guest,meeting_recording.discussion_have_action,
  meeting_recording.external_guest_details,meeting_recording.status as approval_status , 
  meeting_recording.participants , meetingScriber.name as meeting_scriber,
  meetingScriber.surname as meeting_scriber_surname,meetingScriber.profile as meeting_scriber_profile , 
  meetingScriber.email as meeting_scriber_email, meeting_recording.scheduled_meeting , 
  meeting_recording.organization , meeting_recording.meeting_scriber as meeting_scriber_id ,  
  meetingOwner.name as meeting_owner , meetingOwner.surname as meeting_owner_surname , 
  meetingOwner.profile as  meeting_owner_profile , meetingOwner.email as meeting_owner_email, meeting.meeting_title ,
  meeting.meeting_hierarchy, meeting.agenda , meeting_recording.actual_meeting_date_from ,
    meeting_recording.actual_meeting_date_to , meeting_recording.meeting_duration , meeting.venue , 
    meeting.planned_meeting_date_from,meeting.planned_meeting_date_to,meeting.meeting_owner as meeting_owner_id ,
    organization.name as organization_name, organization.business_logo , 
    organization.background_image, organization.header_image, organization.footer_image, organization.font_size,
    organization.pick_color, meeting_recording.id,meeting.meeting_id as meeting_id,
    meeting_recording.created_by as created_by_id,users.name as created_by ,users.surname as created_by_surname , 
    users.profile as created_by_profile
    FROM meeting_recording 
     ${joinsRecord} where meeting_recording.deleted = 0 ${condition}`;
  const [meetings] = await db.query(meetingsQuery);
  for (const meetingsNotes of meetings) {
    meetingsNotes.notes = JSON.parse(meetingsNotes.meeting_notes);
    meetingsNotes.external_guest_details = JSON.parse(
      meetingsNotes.external_guest_details
    );

    const participants = JSON.parse(meetingsNotes.participants);
    const participantsArray = [];

    for (const participant of participants) {
      const [user] = await db.query(`SELECT users.name, 
        users.id, 
        users.email,
        users.surname, 
        users.phone,
        users.unique_id,
        roles.name as role_name,
        organization.name as organization_name,
        users.profile FROM users
        LEFT JOIN roles ON roles.id = users.role
        LEFT JOIN organization ON organization.id = users.my_organization
        where users.id = ${participant.id} AND users.deleted = "0"`);
      participantsArray.push({ ...participant, ...user[0] });
    }

    meetingsNotes.participants = participantsArray;

    if (meetingsNotes.notes && Array.isArray(meetingsNotes.notes)) {
      for (const note of meetingsNotes.notes) {
        if (note.actions && Array.isArray(note.actions)) {
          for (const action of note.actions) {
            if (action.assigned_to) {
              const [assignedUserResult] = await db.query(`SELECT 
                users.name, 
                users.surname,
                users.email,
                users.profile
                FROM users 
                WHERE users.id = ${action.assigned_to} AND users.deleted = "0"`);

              if (assignedUserResult && assignedUserResult.length > 0) {
                const assignedUser = assignedUserResult[0];
                action.assigned_name = `${assignedUser.name} ${assignedUser.surname}`;
                action.assigned_profile = assignedUser.profile;
                action.assigned_email = assignedUser.email;
              }
            }
          }
        }
      }
    }
  }

  for (const meeting of meetings) {
    const agendas = JSON.parse(meeting?.agenda || `[]`);
    for (const agenda of agendas) {
      const userId = agenda.allocated_to;
      let userName = "";
      if (userId) {
        [userName] = await db.query(
          `SELECT CONCAT(name , ' ' , surname) AS name, profile FROM users WHERE id = ${userId}`
        );
      }
      agenda.allocated_name = userName[0]?.name;
      agenda.allocated_profile = userName[0]?.profile;
    }
    meeting.agenda = agendas;
  }

  const result = meetings;
  return result;
};
