import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  insertActivityLog,
  searchConditionRecord,
  makeJoins,
  countQueryCondition,
  deleteSettingRecord,
  decodeSingle_statement,
  uniqueIdGenerator,
  encodeAndStringifyFields,
  getFirstCreatedAndAllOrganizationIds,
  decodeAndParseFields,
} from "../helper/general.js";
import PpeType from "../sequelize/PpeTypeSchema.js";
import {
  createQueryBuilder,
  updateQueryBuilder,
} from "../helper/queryBuilder.js";

export const createUpdatePpeType = async (req, res) => {
  const { id, organization } = req.body;
  if (id && organization.length === 0) {
    return sendResponse(
      res,
      404,
      "Please Select at least one business Structure"
    );
  }
  if (id) {
    const [findPpeType] = await db.query(
      "SELECT * FROM ppe_type WHERE id = ? AND deleted = 0",
      [id]
    );
    if (findPpeType[0].is_static === 1) {
      return sendResponse(res, 404, "Restricted!");
    }
  }
  if (!id) {
    const result = await getFirstCreatedAndAllOrganizationIds();

    if (!result.firstOrgId) {
      return sendResponse(res, 404, "No organization found.");
    }
    const unique_id = await uniqueIdGenerator(
      result.firstOrgId,
      req.body?.department || null,
      "PpeType",
      "ppe_type",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
    req.body.organization =
      organization.length > 0 ? organization : result.organizationIds;
  }

  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id
    ? updateQueryBuilder(PpeType, req.body)
    : createQueryBuilder(PpeType, req.body);
  const [result] = await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    status,
    "PpeType",
    id ? id : result.insertId
  );
  return sendResponse(res, 200, `Record ${status} Successfully`);
};

/**Function to view all and single Disability */
export const viewPpeType = async (req, res) => {
  const { id } = req.params;
  const { dropdown } = req.query;
  const page = parseInt(req.query.page) || 1;
  const pageSize = parseInt(req.query.pageSize) || 10;
  const offset = (page - 1) * pageSize;
  const orderBy = req.query.orderBy || "DESC";
  let filter = req.query.filter || JSON.stringify({});
  let whereCondition = "";
  if (filter) {
    filter = JSON.parse(filter);
    if (id) {
      filter.id = id;
    }
    const filterTable = "ppe_type";
    const whereFilterRecord = Object.entries(filter)
      .filter(([key, value]) => value !== "" && value !== null)
      .map(([key, value]) => {
        if (key === "created_at") {
          return `Date(${filterTable}.${key}) = '${value}'`;
        } else {
          if (key === "organization") {
            return `JSON_CONTAINS(${filterTable}.${key}, '${value}')`;
          }
          return `${filterTable}.${key} = '${value}'`;
        }
      })
      .join(" AND ");

    if (whereFilterRecord !== "") {
      whereCondition = `AND ${whereFilterRecord}`;
    }
  }
  const searchTableName = [
    "ppe_type.name",
    "ppe_type.description",
    "CONCAT(users.name , ' ' , users.surname)",
  ];
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = ppe_type.created_by",
    },
    {
      type: "left",
      targetTable: "ppe_type AS parent_ppe_type",
      onCondition: "parent_ppe_type.id = ppe_type.parent",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const PpeTypeQuery = `SELECT ppe_type.unique_id, ppe_type.id, ppe_type.created_by as created_by_id, ppe_type.name,parent_ppe_type.name AS parent_name , ppe_type.description, users.name as created_by, users.surname as created_by_surname, users.profile as created_by_profile, ppe_type.organization as organizations,ppe_type.organization as organization, ppe_type.parent FROM ppe_type ${joinsRecord} WHERE ppe_type.deleted = 0 ${searchCondition} ${whereCondition} ORDER BY ppe_type.id ${orderBy} LIMIT ${pageSize} OFFSET ${offset}`;
  let [PpeType] = await db.query(PpeTypeQuery);
  PpeType = await decodeAndParseFields(PpeType);
  async function fetchSubChildren(parentId) {
    const [subChildren] = await db.query(
      `SELECT * FROM ppe_type WHERE deleted = 0 AND parent = ?`,
      [parentId]
    );

    // Only populate sub-children if any are found, and no need to recurse if not needed
    for (const child of subChildren) {
      child.description = await decodeSingle_statement(child.description);
      child.sub_children = await fetchSubChildren(child.id);
    }
    return subChildren;
  }

  const nestedPpeType = [];

  for (const item of PpeType) {
    item.description = await decodeSingle_statement(item.description);
    if (item.parent === null) {
      item.sub_children = await fetchSubChildren(item.id);
      nestedPpeType.push(item);
    } else if (dropdown != 1 || id) {
      nestedPpeType.push(item);
    }
  }

  const totalRecord = await countQueryCondition(PpeTypeQuery);
  return sendResponse(res, 200, nestedPpeType, totalRecord);
};

export const viewPpeTypeDropDown = async (req, res) => {
  try {
    // First get ALL parent records (where parent is NULL)
    const parentQuery = `
        SELECT 
          id, 
          name, 
          parent, 
          organization
        FROM ppe_type
        WHERE deleted = 0 
        ORDER BY name ASC
      `;

    // Get all parent records without any LIMIT

    // Fetch all parent records
    let [allRecords] = await db.query(parentQuery);
    allRecords = await decodeAndParseFields(allRecords);

    return sendResponse(res, 200, allRecords);
  } catch (error) {
    console.error("Error fetching ppe_type dropdown data:", error);
    return sendResponse(res, 500, {
      error: "Internal Server Error",
      details:
        process.env.NODE_ENV === "development"
          ? {
              message: error.message,
              stack: error.stack,
            }
          : undefined,
    });
  }
};
export const getParentPpeTypeList = async (req, res) => {
  let { filter } = req.query;
  filter = typeof filter === "string" ? JSON.parse(filter) : filter;
  let { organizations } = filter;
  if (!organizations || organizations.length == 0) {
    const [arrOrg] = await db.query(
      `SELECT id FROM organization WHERE deleted = 0`
    );
    organizations = arrOrg.map((item) => item.id);
  }
  organizations =
    typeof organizations == "string"
      ? JSON.parse(organizations)
      : organizations;
  const ppe_typeQuery = `
      SELECT *
      FROM ppe_type
      WHERE 
          ${organizations
            .map(() => `JSON_CONTAINS(organization, CAST(? AS JSON), '$')`)
            .join(" AND ")}
          AND deleted = 0;
  `;

  const params = [...organizations];
  const [parentPpeTypeList] = await db.query(ppe_typeQuery, params);
  return sendResponse(res, 200, parentPpeTypeList);
};

/**Function to delete a specific Disability */
export const deletePpeType = async (req, res) => {
  const { id } = req.params;
  const [findPpeType] = await db.query(
    "SELECT * FROM ppe_type WHERE id = ? AND deleted = 0",
    [id]
  );
  if (findPpeType[0].is_static === 1) {
    return sendResponse(res, 404, "Restricted!");
  }
  const deleteRecordQuery = await deleteSettingRecord("ppe_type", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(req.user.sessionid, "delete", "ppe_type", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};
