import db from "../db-config.js";
import moment from "moment";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  getRecord,
  insertActivityLog,
  //! insertNotification,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";

import { sendResponse } from "../helper/wrapper.js";
import StakeholderActivityRecording from "../sequelize/StakeholderActivityRecordingSchema.js";
import fs from "fs/promises";
import path from "path";
import ical from "ical-generator";
import StakeholderActivityScheduling from "../sequelize/StakeholderActivitySchedulingSchema.js";
import { uploadToDDRM } from "../helper/ddrmUploader.js";

export const createUpdateActivityRecording = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];
  const { department, stakeholder, sidebar = 243 } = req.body;
  const id = req.body.id;
  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  const file = req.files && req.files.file;
  // if (file) {
  //   req.body.ddrm_id = await uploadToDDRM(sidebar, file, req);
  // }

  // req.body.ddrm_id = req.body.file;

  /// already existance  check

  if (stakeholder) {
    const [record] = await getRecord(
      "stakeholder_activity_recording",
      "stakeholder",
      stakeholder
    );
    if (record && record.id != id) {
      return sendResponse(res, 400, "Record already exist");
    }
  }
  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }

  let arr = [];

  // let created_user = req.user.sessionid;
  // if (req.body.send_for_approval) {
  //   if (created_user == 1) {
  //     req.body.for_approval = 1;
  //     req.body.approval_status = "Approved";
  //   } else {
  //     const [teams] = await db.query(`SELECT * FROM teams WHERE deleted = 0`);
  //     for (const team of teams) {
  //       let teamMembers = JSON.parse(team.team_members);
  //       // console.log(teamMembers);
  //       // console.log(team.team_leader);
  //       if (teamMembers.includes(created_user)) {
  //         arr.push(team.team_leader);
  //         break;
  //       }
  //     }
  //     if (arr.length == 0 || arr[0] == null) {
  //       req.body.for_approval = 1;
  //       req.body.approval_status = "Approved";
  //     } else {
  //       req.body.send_to_employee = arr;
  //       req.body.for_approval = 1;
  //     }
  //   }
  // }

  // console.log(">>>>>>>>>>>>",req.body)

  req.body = await encodeAndStringifyFields(req.body);

  const unique_id = await uniqueIdGenerator(
    req.body.organization,
    department,
    "StakeholderActivityRecording",
    "stakeholder_activity_recording",
    "unique_id",
    "unique_id"
  );
  req.body.unique_id = unique_id;

  const { query, values } = id
    ? updateQueryBuilder(StakeholderActivityRecording, req.body)
    : createQueryBuilder(StakeholderActivityRecording, req.body);

  // console.log(query, values);
  // console.log(`before execution`);
  let [result] = await db.query(query, values);

  // if (arr.length > 0) {
  //   for (let i = 0; i < arr.length; i++) {
  //     const publicUrl = `stakeholder-activity-recording/approval-workflow/view/${
  //       id ? id : result.insertId
  //     }`;
  //     await insertNotification(
  //       "Stakeholder Activity Recording",
  //       publicUrl,
  //       arr[i],
  //       "url",
  //       req.user.sessionid
  //     );
  //   }
  // }

  await insertActivityLog(
    req.user.sessionid,
    status,
    "Stakeholder Activity Recording",
    id ? id : result.insertId
  );

  await StakeholderActivityScheduling.update(
    {
      response_received: true,
    },
    {
      where: {
        id: req.body.stakeholder,
      },
    }
  );

  // if (id) {
  //   await deleteRecord(StakeholderActivityNotes, id, "activity_recording_id");
  // }
  // delete req.body.id;
  // for (const notes of activity_notes) {
  //   notes[id ? "updated_by" : "created_by"] = req.user.sessionid;
  //   notes.activity_recording_id = id ? id : result.insertId;
  //   const { query, values } = createQueryBuilder(
  //     StakeholderActivityNotes,
  //     notes
  //   );
  //   await db.query(query, values);
  // }

  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getStakeholderActivityRecording = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "stakeholder_activity_recording",
    user: req.user,
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });
  const searchTableName = [
    "stakeholder_activity_recording.engagement_details",
    "stakeholder_activity_recording.engagement_method",
    "stakeholder_activity_recording.engagement_plan_objective",
    "stakeholder_activity_recording.frequency_of_engagement",
    "stakeholder_activity_recording.responsible_person",
    "stakeholder_activity_recording.approval_status",
    "stakeholder_activity_recording.status",
    "stakeholder_activity_recording.engagement_status",
    "stakeholder_activity_recording.feedback_status",
    "stakeholder_activity_recording.sentiment_rating",
    "stakeholder_activity_recording.feedback_priority",
  ];

  /** If value come with any search condition then search that word */
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const filter = req.query.filter && JSON.parse(req.query.filter);

  //TODO : Could be used in future
  // !let ownerCondition = "";
  // !if (filter?.for_approval == 1) {
  // !  ownerCondition = req.user.isSuperAdmin
  // ! ? ""
  // !    : `AND JSON_CONTAINS(stakeholder_activity_recording.send_to_employee, ${req.user.sessionid} )`;
  // !}

  const joins = [
    {
      type: "left",
      targetTable: "stakeholder_activity_scheduling",
      onCondition:
        "stakeholder_activity_recording.stakeholder = stakeholder_activity_scheduling.id",
    },
    {
      type: "left",
      targetTable: "stakeholder_engagement_plan",
      onCondition:
        "stakeholder_engagement_plan.id = stakeholder_activity_scheduling.stakeholder",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = stakeholder_activity_recording.organization",
    },
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = stakeholder_activity_recording.created_by",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = stakeholder_activity_recording.ddrm_id",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const stakeholderActivityRecordingQuery = `SELECT stakeholder_activity_recording.*  ,  
  stakeholder_activity_scheduling.stakeholder AS engagement_id,
  stakeholder_engagement_plan.stakeholder_name AS stakeholder_name,
  stakeholder_activity_scheduling.engagement_plan,
  stakeholder_activity_scheduling.engagement_details,
  stakeholder_activity_scheduling.engagement_method,
  stakeholder_activity_scheduling.engagement_plan_objective,
  stakeholder_activity_scheduling.frequency_of_engagement,
  stakeholder_activity_scheduling.responsible_person,
  repository.url AS file,
  stakeholder_activity_scheduling.identification_id , organization.name AS organization_name , CONCAT(users.name , ' ' , users.surname) AS created_by_name , users.profile AS created_by_profile  FROM stakeholder_activity_recording ${joinCondition} WHERE stakeholder_activity_recording.deleted = 0 ${searchCondition} 
  ${condition}`;
  //! ${ownerFilter}

  let [stakeholderActivityRecording] = await db.query(
    stakeholderActivityRecordingQuery
  );

  for (const element of stakeholderActivityRecording) {
    const stakeholderId = element.stakeholder_name;

    const id = !isNaN(Number(stakeholderId));

    if (id) {
      const [user] = await db.query(
        `SELECT CONCAT(name , ' ' , surname) AS name ,id  , profile  from users where id = ${Number(
          stakeholderId
        )}`
      );
      element.stakeholder_person_id = user[0]?.id;
      element.stakeholder_person_name = user[0]?.name;
      element.stakeholder_person_profile = user[0]?.profile;
      element.file = element.file ? element.file : "";
    } else {
      element.stakeholder_person_name = stakeholderId;
      element.file = element.file ? element.file : "";
    }
  }

  stakeholderActivityRecording = await decodeAndParseFields(
    stakeholderActivityRecording
  );

  const totalRecord = await countQueryCondition(
    stakeholderActivityRecordingQuery
  );

  return sendResponse(res, 200, stakeholderActivityRecording, totalRecord);
};

export const deleteStakeholderActivityRecording = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(StakeholderActivityRecording, id);
  const activityRecording = await StakeholderActivityRecording.findByPk(id);
  console.log(activityRecording);
  const stakeholder = activityRecording.stakeholder;
  await StakeholderActivityScheduling.update(
    {
      response_received: false,
    },
    {
      where: {
        id: stakeholder,
      },
    }
  );
  // await StakeholderActivityScheduling
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Stakeholder Activity Recording",
    id
  );
  return sendResponse(res, 200, "Record deleted successfully");
};

export const approveRejectStakeholderActivityRecording = async (req, res) => {
  const { id, approval_status } = req.body;
  const { query, values } = updateQueryBuilder(
    StakeholderActivityRecording,
    req.body
  );
  await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    "approve",
    "Stakeholder Activity Recording",
    id
  );
  return sendResponse(res, 200, `Record ${approval_status} successfully`);
};

export const fetchRecordingScheduleData = async (req, res) => {
  // const [recordingSchedule] = await db.query(
  //   `SELECT * FROM stakeholder_activity_scheduling WHERE deleted = 0`
  // );
  const [recordingSchedule] = await db.query(
    `SELECT * FROM stakeholder_activity_scheduling 
    LEFT JOIN stakeholder_engagement_plan ON stakeholder_activity_scheduling.engagement_plan = stakeholder_engagement_plan.engagement_plan_name
    WHERE stakeholder_activity_scheduling.deleted = 0`
  );

  console.log("recordingSchedule: ", recordingSchedule);

  // Create a new calendar
  const calendar = ical({
    name: "Stakeholder Activity Recording Calendar",
    prodId: { company: "Harmony and Help", product: "SSHEQ" },
    timezone: "Asia/Kolkata",
    url: "https://ssheq.thewingshield.com/",
  });

  recordingSchedule.forEach((schedule) => {
    const engagement_details = JSON.parse(schedule.engagement_details);
    let eventOptions = {
      id: schedule.id,
      organizer: { name: "Harmony & Help", email: "info@harmonyandhelp.com" },
    };
    if (engagement_details) {
      for (let engagement of engagement_details) {
        const planned_activity_date_from = engagement?.engagement_start_date
          ? engagement?.engagement_start_date
          : engagement.actual_start_date;
        const planned_activity_date_to = engagement?.engagement_start_date
          ? engagement?.engagement_start_date
          : engagement.actual_end_date;
        // const startDate = moment(planned_activity_date_from, "DD-MM-YYYY", true); // Use strict parsing
        // const endDate = moment(planned_activity_date_to, "DD-MM-YYYY", true);

        let startDate, endDate;
        const isISODate = (dateString) =>
          /^\d{4}-\d{2}-\d{2}$/.test(dateString);

        // Validate or format the dates
        if (isISODate(planned_activity_date_from)) {
          // If already in YYYY-MM-DD, parse directly
          startDate = moment(planned_activity_date_from, "YYYY-MM-DD", true);
        } else {
          // Convert from other formats to YYYY-MM-DD
          startDate = moment(planned_activity_date_from, "DD-MM-YYYY", true);
        }

        if (isISODate(planned_activity_date_to)) {
          // If already in YYYY-MM-DD, parse directly
          endDate = moment(planned_activity_date_to, "YYYY-MM-DD", true);
        } else {
          // Convert from other formats to YYYY-MM-DD
          endDate = moment(planned_activity_date_to, "DD-MM-YYYY", true);
        }
        if (!startDate.isValid() || !endDate.isValid()) {
          console.warn(`Invalid date for schedule ID: ${schedule.id}`);
          return; // Skip invalid entries
        }
        eventOptions.start = startDate.toDate();
        eventOptions.end = endDate.toDate();
        eventOptions.summary = "meeting_title";
        eventOptions.description = "meeting_title";
        eventOptions.location = "venue";
        const event = calendar.createEvent(eventOptions);
      }
    }

    // If the event is recurring, set the recurrence rule
  });

  // Ensure the directory exists
  const calendarDir = path.join(process.cwd(), "public", "stakeholder");
  await fs.mkdir(calendarDir, { recursive: true });

  // Save the calendar to an .ics file
  const icsContent = calendar.toString();
  console.log("icsContent: ", icsContent);
  const writeFilePath = path.join(
    calendarDir,
    "stakeholder_activity_recording.ics"
  );

  await fs.writeFile(writeFilePath, icsContent);

  res.status(200).json({
    status: true,
    message: "Stakeholder activity Recording saved to .ICS file",
    data: {
      filePath: "stakeholder/stakeholder_activity_recording.ics",
    },
  });
};

export const feedbackAnalysis = async (req, res) => {
  // Total Feedbacks received
  const [allRecord] = await db.query(
    `SELECT * FROM stakeholder_activity_recording WHERE deleted = 0`
  );
  const totalFeedbackCount = allRecord.length;

  // Positive Feedback vs Neutral Feedbacks vs Negative Feedbacks
  const [barGraphData] = await db.query(`
SELECT 
    fs.feedback_status,
    COUNT(sar.feedback_status) AS count,
    ROUND(
        (COUNT(sar.feedback_status) * 100.0 / 
         (SELECT COUNT(*) FROM stakeholder_activity_recording)), 
        2
    ) AS percentage
FROM 
    (SELECT 'Positive' AS feedback_status
     UNION ALL
     SELECT 'Negative'
     UNION ALL
     SELECT 'Neutral') fs
LEFT JOIN 
    stakeholder_activity_recording sar
ON 
    fs.feedback_status = sar.feedback_status
GROUP BY 
    fs.feedback_status;

`);

  // Feedback by Stakeholder
  const [allStakeholderFeedback] = await db.query(`
  SELECT COUNT(*) as stakeholder_feedback 
  FROM stakeholder_activity_recording 
  WHERE deleted = 0 AND stakeholder IS NOT NULL
`);
  const stakeholderFeedbackCount =
    allStakeholderFeedback[0]?.stakeholder_feedback || 0;

  // Feedback by Engagement Method
  const [allEngagementMethodFeedback] = await db.query(`
  SELECT COUNT(*) as engagement_method_feedback 
  FROM stakeholder_activity_recording 
  WHERE deleted = 0 AND engagement_method IS NOT NULL
`);
  //! should be changed where condition for only engagement method or stakeholder // in case of engagement method: stakeholder is null , in case of stakeholder: engagement method is null
  const engagementMethodFeedbackCount =
    allEngagementMethodFeedback[0]?.engagement_method_feedback || 0;

  // Overall Stakeholder Rating
  const [allStakeholderRating] = await db.query(`
 SELECT 
    statuses.feedback_status,
    CASE 
        WHEN statuses.feedback_status = 'Positive' THEN 
            SUM(CASE WHEN sar.feedback_status = 'Positive' THEN 1 ELSE 0 END)
        WHEN statuses.feedback_status = 'Neutral' THEN 
            SUM(CASE WHEN sar.feedback_status = 'Neutral' THEN 1 ELSE 0 END)
        WHEN statuses.feedback_status = 'Negative' THEN 
            SUM(CASE WHEN sar.feedback_status = 'Negative' THEN 1 ELSE 0 END)
        ELSE 0
    END AS count
FROM 
    (SELECT 'Positive' AS feedback_status
     UNION ALL
     SELECT 'Neutral'
     UNION ALL
     SELECT 'Negative') statuses
LEFT JOIN 
    stakeholder_activity_recording sar
ON 
    statuses.feedback_status = sar.feedback_status
   AND sar.deleted = 0 
   AND sar.stakeholder IS NOT NULL
GROUP BY 
    statuses.feedback_status;
`);

  // Overall Stakeholder Sentiment Rating
  const [allStakeholderSentimentRating] = await db.query(`
  SELECT 
      ROUND (AVG(
          CASE 
              WHEN sentiment_rating = 'Very Positive' THEN 2
              WHEN sentiment_rating = 'Positive' THEN 1
              WHEN sentiment_rating = 'Neutral' THEN 0
              WHEN sentiment_rating = 'Negative' THEN -1
              WHEN sentiment_rating = 'Very Negative' THEN -2
          END
      ), 2) AS overall_rating
  FROM 
      stakeholder_activity_recording 
  WHERE deleted = 0 AND stakeholder IS NOT NULL;
`);
  const overallSentimentRating =
    allStakeholderSentimentRating[0]?.overall_rating || null;

  // Overall Impact (Individual vs Team vs Company)
  const [overallImpact] = await db.query(`
    WITH possible_values AS (
        SELECT 'Individual' AS scope
        UNION ALL
        SELECT 'Team'
        UNION ALL
        SELECT 'Company'
    )
    SELECT 
        pv.scope, 
        COALESCE(COUNT(sar.feedback_impact_scope), 0) AS count
    FROM 
        possible_values pv
    LEFT JOIN 
        stakeholder_activity_recording sar 
    ON 
        sar.feedback_impact_scope LIKE CONCAT('%', pv.scope, '%')
    GROUP BY 
        pv.scope;
`);

  // Send the response
  return res.status(200).json({
    success: true,
    data: {
      totalFeedbacks: totalFeedbackCount,
      feedbackDistribution: barGraphData,
      stakeholderFeedbackCount,
      engagementMethodFeedbackCount,
      stakeholderRatings: allStakeholderRating,
      overallSentimentRating,
      overallImpact,
    },
  });
};
