import db from "../db-config.js";
import StakeholderEngagementPlan from "../sequelize/StakeholderEngagementPlanSchema.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  getRecord,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";

const methodFirst = [
  "Newsletters",
  "Reports",
  "Notice Boards",
  "Feedback Forms",
  "Phone Calls",
  "E-mails",
  "Loyalty Programs",
  "Media Campaigns",
];

const methodSecond = [
  "Press Releases",
  "Physical Letters",
  "Website Updates",
  "Social Media Posts",
  "Intranet Portals",
];

const methodThird = [
  "Focus Groups",
  "Public Consultation Sessions",
  "Exit Interviews",
  "Workshops",
  "Collective Bargaining",
  "Meetings",
  "Awards and Recognition",
  "Corporate Events",
  "Corporate Social Investments",
];

const methodFourth = ["Voting and Balloting"];

const methodStructureObj = {
  method_first: [
    {
      topic_name: "",
      actual_start_date: "",
      actual_frequency: "",
      actual_end_date: "",
    },
  ],
  method_second: [
    {
      topic_name: "",
      actual_start_date: "",
      actual_frequency: "",
      actual_end_date: "",
    },
  ],

  method_third: [
    {
      title: "",
      hierarchy: "",
      participants: "",
      host: "",
      dateFrom: "",
      dateTo: "",
      venue: "",
      agenda: [{ title: "", allocated_to: "" }],
    },
  ],

  method_fourth: [
    {
      subject_name: [""],
      actual_start_date: "",
      actual_frequency: "",
      actual_end_date: "",
    },
  ],
};

// export const createUpdateStakeholderEngagementPlan = async (req, res) => {
//   const {
//     id,
//     department,
//     no_of_cycles,
//     engagement_frequency,
//     engagement_start_date,
//   } = req.body;
//   let status = id ? "Update" : "Create";
//   req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

//   if (department) {
//     req.body.organization = (
//       await getOrganizationAccordingToDepartment(department)
//     )[0].organization;
//   }
//   const idCheck = id ? ` AND id != ${id}` : "";
//   const [check] = await db.query(
//     `SELECT * FROM stakeholder_engagement_plan WHERE stakeholder_name = '${req.body.stakeholder_name}' AND engagement_methods = ${req.body.engagement_methods} AND organization = ${req.body.organization} ${idCheck} AND deleted = 0`
//   );

//   if (check.length > 0) {
//     return sendResponse(
//       res,
//       403,
//       "Record Already Exist with the given engagement plan"
//     );
//   }

//   req.body = await encodeAndStringifyFields(req.body);

//   const createRecords = [];
//   let startDate = new Date(engagement_start_date);
//   const endOfYear = new Date(startDate.getFullYear(), 11, 31); // End of the current year

//   // Loop to create records based on no_of_cycles or until the end of the year
//   let cycleCount = no_of_cycles || Infinity;
//   let cycleNumber = 0;

//   while (cycleNumber < cycleCount && startDate <= endOfYear) {
//     const cycleData = {
//       ...req.body,
//       engagement_start_date: startDate.toISOString().split("T")[0],
//     };
//     createRecords.push(cycleData);

//     // Increment the date for the next cycle
//     console.log("engagement_frequency: ", engagement_frequency);
//     switch (engagement_frequency) {
//       case "Daily":
//         startDate.setDate(startDate.getDate() + 1);
//         break;
//       case "Weekly":
//         startDate.setDate(startDate.getDate() + 7);
//         break;
//       case "Bi-weekly":
//         startDate.setDate(startDate.getDate() + 14);
//         break;
//       case "Monthly":
//         startDate.setMonth(startDate.getMonth() + 1);
//         break;
//       case "Bi-Monthly":
//         startDate.setMonth(startDate.getMonth() + 2);
//         break;
//       case "Quarterly":
//         startDate.setMonth(startDate.getMonth() + 3);
//         break;
//       case "Bi-Annually":
//         startDate.setMonth(startDate.getMonth() + 6);
//         break;
//       case "Annually":
//         startDate.setFullYear(startDate.getFullYear() + 1);
//         break;
//       default:
//         return sendResponse(res, 400, "Invalid engagement frequency.");
//     }

//     cycleNumber += 1;
//   }

//   for (const cycle of createRecords) {
//     const { query, values } = id
//       ? updateQueryBuilder(StakeholderEngagementPlan, cycle)
//       : createQueryBuilder(StakeholderEngagementPlan, cycle);
//     await db.query(query, values);
//   }

//   await insertActivityLog(
//     req.user.sessionid,
//     status,
//     "Stakeholder Engagement Plan",
//     id
//   );

//   return sendResponse(res, 200, `Record ${status}d Successfully`);
// };

// changes explained by evan sir
export const createUpdateStakeholderEngagementPlan = async (req, res) => {
  const {
    id,
    department,
    no_of_cycles,
    engagement_frequency,
    engagement_start_date,
  } = req.body;
  let status = id ? "Update" : "Create";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }
  const idCheck = id ? ` AND id != ${id}` : "";
  const [check] = await db.query(
    `SELECT * FROM stakeholder_engagement_plan WHERE stakeholder_name = '${req.body.stakeholder_name}' AND engagement_methods = "${req.body.engagement_methods}" AND organization = ${req.body.organization} ${idCheck} AND deleted = 0`
  );

  if (check.length > 0) {
    return sendResponse(
      res,
      403,
      "Record Already Exist with the given engagement plan"
    );
  }

  req.body = await encodeAndStringifyFields(req.body);

  if (!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "StakeholderEngagementPlan",
      "stakeholder_engagement_plan",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  const { query, values } = id
    ? updateQueryBuilder(StakeholderEngagementPlan, req.body)
    : createQueryBuilder(StakeholderEngagementPlan, req.body);
  await db.query(query, values);

  await insertActivityLog(
    req.user.sessionid,
    status,
    "Stakeholder Engagement Plan",
    id
  );

  return sendResponse(res, 200, `Record ${status}d Successfully`);
};

export const getStakeholderEngagementPlan = async (req, res) => {
  const { id } = req.params;
  const { stakeholder } = req.query;
  const condition = await whereCondition({
    table: "stakeholder_engagement_plan",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
    groupBy: req.query.groupBy,
  });
  // const stakeholderFilter = stakeholder ? `AND `
  const searchTableName = [
    "stakeholder_engagement_plan.classification_type",
    "stakeholder_engagement_plan.priority_level",
    "stakeholder_engagement_plan.stakeholder_name",
    "stakeholder_engagement_plan.recommended_engagement_levels",
    "stakeholder_engagement_plan.engagement_plan_name",
    "stakeholder_engagement_plan.engagement_plan_objectives",
    "stakeholder_engagement_plan.engagement_frequency",
    "stakeholder_engagement_plan.engagement_start_date",
    "stakeholder_engagement_plan.no_of_cycles",
    "stakeholder_engagement_plan.type_of_resources",
    "organization.name",
    "department.name",
    "CONCAT(users.name , ' ' , users.surname)",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "stakeholder_classification",
      onCondition:
        "stakeholder_classification.id = stakeholder_engagement_plan.stakeholder_category",
    },
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = stakeholder_engagement_plan.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = stakeholder_engagement_plan.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = stakeholder_engagement_plan.department",
    },
    {
      type: "left",
      targetTable: "users as responsible",
      onCondition:
        "responsible.id = stakeholder_engagement_plan.responsible_person",
    },
    // {
    //   type: "left",
    //   targetTable: "engagement_method AS em",
    //   onCondition: "em.id = stakeholder_engagement_plan.engagement_methods",
    // },
    {
      type: "left",
      targetTable: "engagement_levels AS el",
      onCondition: "el.id = stakeholder_engagement_plan.engagement_level",
    },
    {
      type: "left",
      targetTable: "currency",
      onCondition: "currency.id = stakeholder_engagement_plan.currency",
    },
  ];
  const joinRecord = await makeJoins(joins);

  const stakeholderEngagementPlanQuery = `SELECT stakeholder_engagement_plan.* , organization.name as organization_name , department.name AS department_name , CONCAT(users.name , ' ' , users.surname) AS created_by_name, users.profile AS created_by_profile, CONCAT(responsible.name , ' ' , responsible.surname) as responsible_person_name , responsible.profile AS responsible_person_profile, el.name AS engagement_level_name, currency.name AS currency_name , stakeholder_classification.stakeholder_category AS stakeholder_category_name , stakeholder_classification.stakeholder_type FROM stakeholder_engagement_plan ${joinRecord} WHERE stakeholder_engagement_plan.deleted = 0 ${searchCondition} ${condition}`;

  let [stakeholderEngagementPlan] = await db.query(
    stakeholderEngagementPlanQuery
  );
  stakeholderEngagementPlan = await decodeAndParseFields(
    stakeholderEngagementPlan
  );
  // console.log(stakeholderEngagementPlan);

  for (const element of stakeholderEngagementPlan) {
    const hr = element.human_resource_requirements;
    if (hr) {
      for (const h of hr) {
        const id = h.role;
        if (id) {
          const [role] = await db.query(
            `SELECT name FROM roles WHERE id = ${id}`
          );
          h.role_name = role[0]?.name;
        }
      }
    }

    const stakeholderId = element.stakeholder_name;
    // console.log(stakeholderId);
    const id = !isNaN(Number(stakeholderId));
    // console.log(id);
    if (id) {
      const [user] = await db.query(
        `SELECT CONCAT(name , ' ' , surname) AS name ,id  , profile  from users where id = ${Number(
          stakeholderId
        )}`
      );
      element.stakeholder_person_id = user[0]?.id;
      element.stakeholder_person_name = user[0]?.name;
      element.stakeholder_person_profile = user[0]?.profile;
    } else {
      element.stakeholder_person_name = stakeholderId;
    }
  }

  const totalRecord = await countQueryCondition(stakeholderEngagementPlanQuery);

  return sendResponse(res, 200, stakeholderEngagementPlan, totalRecord);
};

export const deleteStakeholderEngagementPlan = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(StakeholderEngagementPlan, id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Stakeholder Engagement Plan",
    id
  );
  return sendResponse(res, 200, "Record deleted successfully");
};

export const getActivityStructureOnFrequency = async (req, res) => {
  const { id } = req.params;
  let { page = 1, pageSize = 10 } = req.query;
  page = parseInt(page, 10);
  pageSize = parseInt(pageSize, 10);
  const [record] = await getRecord("stakeholder_engagement_plan", "id", id);
  // console.log('record: ', record);
  if (!record) {
    return sendResponse(res, 404, "Record not found");
  }
  const startDate = new Date(record.engagement_start_date);
  const frequency = record.engagement_frequency;
  const no_of_cycles = record.no_of_cycles;
  let activityStructure = getActivityStructure(
    startDate,
    frequency,
    no_of_cycles,
    record.engagement_methods
  );
  const totalRecords = activityStructure.length;
  const totalPages = Math.ceil(totalRecords / pageSize);
  const currentPage = page;
  const startIndex = (currentPage - 1) * pageSize;
  // console.log( `Total Records: ${totalRecords}, Total Pages: ${totalPages}, Current Page: ${currentPage}, Start Index: ${startIndex}`)
  activityStructure = activityStructure.slice(
    startIndex,
    startIndex + pageSize
  );
  return sendResponse(res, 200, activityStructure, totalRecords);
};

function getActivityStructure(
  startDate,
  frequency,
  no_of_cycles,
  engagement_methods
) {
  let key;
  if (methodFirst.includes(engagement_methods)) {
    key = "method_first";
  } else if (methodSecond.includes(engagement_methods)) {
    key = "method_second";
  } else if (methodThird.includes(engagement_methods)) {
    key = "method_third";
  } else if (methodFourth.includes(engagement_methods)) {
    key = "method_fourth";
  }
  const selectedStructure = methodStructureObj[key];

  let currentDate = new Date(startDate);
  const endOfYear = new Date(startDate.getFullYear(), 11, 31); // End of the current year
  let cycleCount = no_of_cycles || Infinity;
  let cycleNumber = 0;
  const activityStructure = [];
  while (cycleNumber < cycleCount && currentDate <= endOfYear) {
    activityStructure.push({
      ...selectedStructure[0],
      actual_start_date: currentDate.toLocaleDateString("en-CA"),
      actual_frequency: frequency,
    });
    switch (frequency) {
      case "Daily":
        currentDate.setDate(currentDate.getDate() + 1);
        break;
      case "Bi-weekly":
        currentDate.setDate(currentDate.getDate() + 3);
        break;
      case "Weekly":
        currentDate.setDate(currentDate.getDate() + 7);
        break;
      case "Bi-Monthly":
        currentDate.setDate(currentDate.getDate() + 15);
        break;
      case "Monthly":
        currentDate.setMonth(currentDate.getMonth() + 1);
        break;
      case "Quarterly":
        currentDate.setMonth(currentDate.getMonth() + 3);
        break;
      case "Bi-Annually":
        currentDate.setMonth(currentDate.getMonth() + 6);
        break;
      case "Annually":
        currentDate.setFullYear(currentDate.getFullYear() + 1);
        break;
      default:
        break;
    }
    cycleNumber += 1;
  }
  return activityStructure;
}
