import TrainingEffectivenessOutcome from "../sequelize/TrainingEffectivenessOutcomeSchema.js";
import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  updateQueryBuilder,
  deleteRecord,
  insertActivityLog,
  makeJoins,
  whereCondition,
  searchConditionRecord,
  getOrganizationAccordingToDepartment,
  decodeAndParseFields,
  encodeAndStringifyFields,
  uniqueIdGenerator,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";

export const createUpdateEffectivenessOutcome = async (req, res) => {
  const { id, department } = req.body;
  let status = id ? "Updated" : "Created";

  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0]?.organization;
  }

  if(!id){
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "TrainingEffectivenessOutcome",
      "effectiveness_outcome",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id
    ? updateQueryBuilder(TrainingEffectivenessOutcome, req.body)
    : createQueryBuilder(TrainingEffectivenessOutcome, req.body);
  await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    "create",
    `This user ${status} a new Training Effectiveness Outcome Record`,
    id
  );

  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getEffectivenessOutcome = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "effectiveness_outcome",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "effectiveness_outcome.employee_name",
    "effectiveness_outcome.surname",
    "effectiveness_outcome.employee_id",
    "effectiveness_outcome.role_name",
    // "effectiveness_outcome.training_awareness_program",
    "training_need_analysis.training_name",
    "effectiveness_outcome.training_date",
    "effectiveness_outcome.training_venue",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = effectiveness_outcome.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = effectiveness_outcome.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = effectiveness_outcome.department",
    },
    {
      type: "left",
      targetTable: "roles",
      onCondition: "roles.id = effectiveness_outcome.role_name",
    },
    {
      type: "left",
      targetTable: "users AS employee",
      onCondition: "employee.id = effectiveness_outcome.employee_name",
    },
    {
      type: "left",
      targetTable: "roles AS employee_role",
      onCondition: "employee_role.id = employee.role",
    },
    {
      type: "left",
      targetTable: "attendance_recording",
      onCondition: "attendance_recording.id = effectiveness_outcome.training_awareness",
    },
    {
      type: "left",
      targetTable: "training_schedule",
      onCondition: "training_schedule.id = attendance_recording.training_program",
    },
    {
      type: "left",
      targetTable: "training_need_analysis",
      onCondition: "training_need_analysis.id = training_schedule.training_program",
    },
  ];
  const joinCondition = await makeJoins(joins);

  const effectivenessOutcomeQuery = `SELECT effectiveness_outcome.* , training_need_analysis.training_name as training_awareness_name, organization.name as organization_name, department.name as department_name, users.name as created_by_name, users.surname as created_by_surname , users.profile as created_by_profile , roles.name as roles_name , employee.name AS employee_name_name , employee.surname AS surname , employee.unique_id AS employee_id , employee_role.name AS roles_name FROM effectiveness_outcome ${joinCondition} WHERE effectiveness_outcome.deleted = 0 ${searchCondition} ${condition}`;

  let [effectivenessOutcome] = await db.query(effectivenessOutcomeQuery);
  effectivenessOutcome = await decodeAndParseFields(effectivenessOutcome);

  // each section calculation

  for (let i = 0; i < effectivenessOutcome.length; i++) {
    const effectivenessOutcomeResponse =
      effectivenessOutcome[i].responses || [];
    let overallSum = 0;
    let overallCount = 0;
    for (let j = 0; j < effectivenessOutcomeResponse.length; j++) {
      let categorySum = 0;
      let categoryCount = 0;
      const section = effectivenessOutcomeResponse[j];
      // Loop through the keys to find numeric scores
      for (const key in section) {
        if (!isNaN(key) && !isNaN(parseFloat(section[key]))) {
          categorySum += parseFloat(section[key]);
          categoryCount++;
        }
      }

      // Calculate category average
      if (categoryCount > 0) {
        const categoryAverage = categorySum / categoryCount;
        effectivenessOutcomeResponse[j].category_average = Math.round(
          categoryAverage
        );

        // Add to overall sum and count
        overallSum += categorySum;
        overallCount += categoryCount;
      }
      else{
        effectivenessOutcomeResponse[j].category_average = 0;
      }
    }
    // Calculate overall average
    if (overallCount > 0) {
      const overallAverage = overallSum / overallCount;
      effectivenessOutcome[i].overall_average = Math.round(overallAverage);
    }
    else{
      effectivenessOutcome[i].overall_average = 0;
    }
  }
  const totalRecord = await countQueryCondition(effectivenessOutcomeQuery);

  return sendResponse(res, 200, effectivenessOutcome, totalRecord);
};

export const deleteEffectivenessOutcome = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(TrainingEffectivenessOutcome, id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Training Effectiveness Outcome Record has been deleted by this user",
    id
  );
  return sendResponse(res, 200, "Record deleted successfully");
};
