import Excel from "exceljs";
import path from "path";
import db from "../db-config.js";
import {
  decodeSingle_statement,
  decodeTheEditorContent,
  whereCondition,
  storeError,
  textExtractor,
} from "./general.js";
import asyncHandler from "express-async-handler";
import fs from "fs";

const controlEffectivenessObj = {
  np_1: "Not Practicable",
  n_0: "None",
  "ri_0.5": "Requires Improvement",
  s_1: "Satisfactory",
};

const inherentFrequencyObj = {
  7: "Continuous",
  6: "Daily",
  5: "Weekly",
  4: "Monthly",
  3: "Quarterly",
  2: "Yearly",
  1: "10 Yearly",
};

const inherentConsequenceObj = {
  1: "None",
  3: "Insignificant",
  7: "Noticeable",
  15: "ImOrganisationant",
  40: "Serious",
  100: "Disaster",
};

const inherentProbabilityObj = {
  5: "Expected",
  4: "Probable",
  3: "Unusual",
  2: "Remote",
  1: "Improbable",
};

const controlEffectivenessClassificationObj = {
  99: "Satisfactory",
  90: "Satisfactory",
  80: "Satisfactory",
  79: "Requires Improvement",
  70: "Requires Improvement",
  60: "Requires Improvement",
  50: "Requires Improvement",
  49: "Unsatisfactory",
  40: "Unsatisfactory",
  30: "Unsatisfactory",
  20: "Unsatisfactory",
  10: "Unsatisfactory",
  9: "None",
};

function controlEffectiveClassificationFunc(value) {
  if (value >= 80) {
    return "Satisfactory";
  } else if (value > 49 && value < 80) {
    return "Requires Improvement";
  } else if (value <= 49 && value >= 10) {
    return "Unsatisfactory";
  } else if (value < 10) {
    return "None";
  }
}

export const getOperationalRiskAssessmentRegisterExcel = asyncHandler(
  async (req, res) => {
    try {
      const workbook = new Excel.Workbook();
      const worksheet = workbook.addWorksheet("Operational Risk Assessment");

      // Colors for styling
      const white = "FFFFFF";
      const black = "000000";
      const blue = "0000FF";
      const darkBlue = "000080";
      const violet = "9400D3";
      const red = "FF0000";
      const gray = "808080";
      const lightGray = "D3D3D3";
      const green = "008000";

      // Header setup (columns)
      const headers = [
        { header: "Risk ID", key: "riskId", width: 20 },
        { header: "Frequency", key: "frequency", width: 25 },
        { header: "Probability", key: "probability", width: 25 },
        { header: "Consequence", key: "consequences", width: 25 },
        { header: "Inherent Risk Rating", key: "inherentRiskRating", width: 25 },
        { header: "Risk Ranking", key: "riskRanking", width: 25 },
        { header: "Substitution", key: "substitution", width: 25 },
        { header: "Engineering Controls", key: "engineeringControls", width: 25 },
        { header: "Administrative Controls", key: "administrativeControls", width: 25 },
        { header: "Last Resort", key: "lastResort", width: 25 },
        { header: "Risk Ranking", key: "riskRankings", width: 25 },
        { header: "Is there any opportunity for risk", key: "opportunityIdentification", width: 25 },
        { header: "If yes state opportunity", key: "opportunityDescription", width: 25 },
        { header: "Is the risk a priority (YES/NO)", key: "priorityConfirmation", width: 25 },
      ];

      // Adjust column widths based on header length with null check
      headers.forEach((header, index) => {
        const columnIndex = index + 1;

        // Check if header value exists and calculate width accordingly
        const headerText = header.header || '';  // Default to empty string if header is null or undefined
        worksheet.getColumn(columnIndex).width = Math.max(headerText.length + 15, header.width || 20);
      });

      // Insert the first three rows at the top of the worksheet
      worksheet.insertRow(1, []);
      worksheet.mergeCells("A1:Z1");
      worksheet.getCell("A1").value = "OPERATIONAL RISK ASSESSMENT REGISTER";
      worksheet.getCell("A1").alignment = { horizontal: "center" };
      worksheet.getCell("A1").font = { size: 16, bold: true };


      // Add the headers starting from row 4 (shifted down by 3 rows)
      worksheet.addRow(headers.map((header) => header.header)); // Headers are added on row 4

      // Fetch the data from the database
      const { id } = req.params;
      const condition = await whereCondition({
        table: "operational_risk_identification",
        page: req.query.page,
        all: req.query.all,
        pageSize: req.query.pageSize,
        filter: req.query.filter,
        id,
        grouped: req.query.grouped,
        user: req.user,
      });

      const [dataFetch] = await db.query(
        `SELECT id as riskId, frequency, probability, consequences, risk_rating as inherentRiskRating, 
        risk_ranking as riskRanking, substitution, engineering_control as engineeringControl, 
        administrative_control as administrativeControl, last_resort as lastResort, 
        risk_ranking as riskRankings, opportunity_identification as opportunityIdentification, 
        opportunity_description as opportunityDescription, priority_confirmation as priorityConfirmation 
        FROM operational_risk_identification
        WHERE deleted = 0 ${condition}`
      );

      // Decode content and prepare rows for filling
      for (const item of dataFetch) {
        item.frequency = inherentFrequencyObj[item.frequency];
        item.probability = inherentProbabilityObj[item.probability];
        item.consequences = inherentConsequenceObj[item.consequences];
        item.substitution = controlEffectivenessObj[item.substitution];
        item.engineeringControl = controlEffectivenessObj[item.engineeringControl];
        item.administrativeControl = controlEffectivenessObj[item.administrativeControl];
        item.lastResort = controlEffectivenessObj[item.lastResort];
        item.priorityConfirmation = +item.priorityConfirmation === 1 ? "YES" : "NO";
        item.opportunityIdentification = item.opportunityIdentification == 1 ? "YES" : "NO";
        if (item.opportunityIdentification === "YES") {
          item.opportunityDescription = await decodeTheEditorContent(
            item.opportunityDescription,
            "description"
          );
        }

        item.impact = await decodeSingle_statement(item?.impact);
        item.impact = textExtractor(item?.impact || "");
      }

      // Insert data rows starting from row 5 (since rows 1, 2, 3 are taken)
      dataFetch.forEach((row, index) => {
        // console.log("row",row)
        const rowIndex = index + 5; // Start from row 5
        worksheet.getCell(rowIndex, 1).value = row["riskId"];
        worksheet.getCell(rowIndex, 2).value = row["frequency"];
        worksheet.getCell(rowIndex, 3).value = row["probability"];
        worksheet.getCell(rowIndex, 4).value = row["consequences"];
        worksheet.getCell(rowIndex, 5).value = row["inherentRiskRating"];
        worksheet.getCell(rowIndex, 6).value = row["riskRanking"];
        worksheet.getCell(rowIndex, 7).value = row["substitution"];
        worksheet.getCell(rowIndex, 8).value = row["engineeringControl"];
        worksheet.getCell(rowIndex, 9).value = row["administrativeControl"];
        worksheet.getCell(rowIndex, 10).value = row["frequency"];
        worksheet.getCell(rowIndex, 11).value = row["probability"];
        worksheet.getCell(rowIndex, 12).value = row["consequences"];
        worksheet.getCell(rowIndex, 13).value = row["inherentRiskRating"];
        worksheet.getCell(rowIndex, 14).value = row["riskRanking"];
        worksheet.getCell(rowIndex, 24).value = row["opportunityIdentification"];
        worksheet.getCell(rowIndex, 25).value = row["opportunityDescription"];
        worksheet.getCell(rowIndex, 26).value = row["priorityConfirmation"];
      });

      const second = worksheet.getRow(2); 
      second.height = 20;
      
      // Loop over all columns to handle merging and alignment
      for (let col = 1; col <= headers.length; col++) {
        // Get the cell in row 2 (this should be your header row)
        const cell = worksheet.getCell(2, col); // Row 2 is the header row after the first merged cell
      
        // Merging logic (adjusting based on the column range)
        if (col >= 10 && col <= 14) {
          // Merge cells from row 3 to row 4 for columns 10 to 14
          worksheet.mergeCells(3, col, 4, col);
        } else if (col >= 16 && col <= 23) {
          // Merge cells from row 3 to row 4 for columns 16 to 23 (with an additional condition for col >= 20)
          if (col >= 20) {
            worksheet.mergeCells(3, col, 4, col);
          }
        } else {
          // Merge rows 2 to 4 for all other columns
          worksheet.mergeCells(2, col, 4, col);
        }
      
        // Safe length check on cell.value to avoid errors
        const cellValue = cell.value || '';  // Default to empty string if cell.value is null or undefined
        if (cellValue.length > 20 && cellValue.length < 50) {
          // Adjust cell alignment and rotation if the header text length is between 20 and 50 characters
          cell.alignment = {
            vertical: "middle",
            horizontal: "center",
            textRotation: 180,
          };
        } else {
          // Default alignment for shorter text
          cell.alignment = { vertical: "middle", horizontal: "center" };
        }
      }
      
      const cwd = process.cwd();

      worksheet.getCell("J3").value = "Frequency";
      worksheet.getCell("K3").value = "Probability";
      worksheet.getCell("L3").value = "Consequences";
      worksheet.getCell("M3").value = "Inherent Risk Rating";
      worksheet.getCell("N3").value = "Risk Ranking";
      worksheet.mergeCells("J2:N2");
      worksheet.getCell("J2").value = "Inherent Risk Assessment";
      worksheet.getCell("J2").alignment = { horizontal: "center" };

      worksheet.getCell("P3").value = "Substitution";
      worksheet.getCell("Q3").value = "Engineering Controls";
      worksheet.getCell("R3").value = "Administrative Controls";
      worksheet.getCell("S3").value = "Last Resort";
      worksheet.mergeCells("P2:V2");
      worksheet.getCell("P2").value = "Residual Risk Assessment";
      worksheet.getCell("P2").alignment = { horizontal: "center" };

      worksheet.getCell("P4").value = "30%";
      worksheet.getCell("Q4").value = "25%";
      worksheet.getCell("R4").value = "20%";
      worksheet.getCell("S4").value = "15%";

      worksheet.getCell("T3").value = "Control Effectiveness";
      worksheet.getCell("U3").value = "Residual Risk Rating";
      worksheet.getCell("V3").value = "Control Effectiveness Classification";
      worksheet.getCell("W2").value = "Risk Ranking";
      worksheet.getCell("X2").value = "Is there any opportunity for risk";
      worksheet.getCell("Y2").value = "If yes state opportunity";
      worksheet.getCell("Z2").value = "Is the risk a priority (YES/NO)";

      // Applying styles to the worksheet
      worksheet.getColumn(16).width = 30;
      worksheet.getColumn(17).width = 35;
      worksheet.getColumn(18).width = 35;
      worksheet.getColumn(19).width = 25;
      worksheet.getColumn(20).width = 32;
      worksheet.getColumn(21).width = 32;
      worksheet.getColumn(22).width = 33;
      worksheet.getColumn(23).width = 33;
      worksheet.getColumn(24).width = 33;
      worksheet.getColumn(25).width = 33;
      worksheet.getColumn(26).width = 33;

      worksheet.eachRow({ includeEmpty: false }, function (row, rowNumber) {
        // Iterate over each cell in the row
        row.eachCell({ includeEmpty: false }, function (cell, colNumber) {
          // Set the border for each cell
          cell.border = {
            top: { style: "medium" }, // Thin top border
            left: { style: "medium" }, // Thin left border
            bottom: { style: "medium" }, // Thin bottom border
            right: { style: "medium" }, // Thin right border
            // You can adjust the style to 'medium' or 'thick' for darker or bolder borders
          };
          cell.alignment = { vertical: "middle", horizontal: "center" };
        });
      });

      const heading = worksheet.getRow(1);

      heading.height = 40;

      heading.eachCell((cell) => {
        cell.fill = {
          type: "pattern",
          pattern: "solid",
          fgColor: { argb: "90EE90" },
        };
        cell.font = {
          size: 16,
        };
      });

      const subHeading = worksheet.getRow(2);
      subHeading.eachCell((cell) => {
        cell.font = {
          size: 14,
        };
      });

      //  for (let i = 2; i <= 4; i++) {
      //    worksheet.getRow(i).fill = {
      //      type: "pattern",
      //      pattern: "solid",
      //      fgColor: { argb: "FF808080" }, // Gray color
      //    };
      //  }
      applyCss(worksheet, 2, 1, 26, gray, black);
      applyCss(worksheet, 3, 1, 26, gray, black);
      applyCss(worksheet, 4, 1, 26, gray, black);
      const headerRow = worksheet.getRow(2);
      headerRow.height = 20;
      // Define the file path where you want to save the Excel file

      const directoryPath = path.join(cwd, "public", "excel");
      const filePath = path.join(
        directoryPath,
        "Operational_Risk_Assessment_Register.xlsx"
      );
      // Ensure the directory exists
      if (!fs.existsSync(directoryPath)) {
        fs.mkdirSync(directoryPath, { recursive: true });
      }

      // Save the workbook
      await workbook.xlsx.writeFile(filePath);

      return res.status(200).json({
        status: true,
        message: "Excel file created successfully",
        data: `excel/Operational_Risk_Assessment_Register.xlsx`,
      });
    } catch (error) {
      storeError(error);
      return res.status(500).json({
        status: false,
        message: error.message,
      });
    }
  }
);

function applyCss(worksheet, row, startCol, endCol, BgColor, textColor) {
  for (let col = startCol; col <= endCol; col++) {
    const cell = worksheet.getCell(row, col); // Get cell at row 2, current column
    cell.fill = {
      type: "pattern",
      pattern: "solid",
      fgColor: { argb: BgColor },
    };
    cell.font = {
      color: { argb: textColor },
    };
  }
}
