import axios from "axios";
import dotenv from "dotenv";
import readline from "readline";

dotenv.config();
const { SWAGGER_URL } = process.env;

// Replace with your actual values
const COLLECTION_NAME = "NEW COLLECTION"; // Your custom collection name

// Define different credentials for team members
const credentials = {
  ravi: {
    POSTMAN_API_KEY: process.env.POSTMAN_API_KEY_RAVI,
    POSTMAN_COLLECTION_ID: process.env.POSTMAN_COLLECTION_ID_RAVI,
  },
  harsh: {
    POSTMAN_API_KEY: process.env.POSTMAN_API_KEY_HARSH,
    POSTMAN_COLLECTION_ID: process.env.POSTMAN_COLLECTION_ID_HARSH,
  },
  tariq: {
    POSTMAN_API_KEY: process.env.POSTMAN_API_KEY_TARIQ,
    POSTMAN_COLLECTION_ID: process.env.POSTMAN_COLLECTION_ID_TARIQ,
  },
  // Add more users as needed
};

// Function to get user input from the terminal
const askUserInput = (question) => {
  const rl = readline.createInterface({
    input: process.stdin,
    output: process.stdout,
  });

  return new Promise((resolve) =>
    rl.question(question, (answer) => {
      rl.close();
      resolve(answer);
    })
  );
};

const updatePostmanCollection = async (collectionName, postmanApiKey, collectionId) => {
  try {
    // Fetch Swagger JSON
    const swaggerResponse = await axios.get(SWAGGER_URL);
    const swaggerJson = swaggerResponse.data;

    if (!swaggerJson.paths) {
      throw new Error("Invalid Swagger JSON: 'paths' property is missing.");
    }

    // Prepare Postman collection
    const postmanCollection = {
      info: {
        name: collectionName,
        schema: "https://schema.getpostman.com/json/collection/v2.1.0/collection.json",
      },
      item: [],
    };

    const tagGroups = {};

    // Group endpoints by tags
    Object.keys(swaggerJson.paths).forEach((path) => {
      const pathItem = swaggerJson.paths[path];
      Object.keys(pathItem).forEach((method) => {
        const request = pathItem[method];
        const tags = request.tags || [];

        tags.forEach((tag) => {
          if (!tagGroups[tag]) {
            tagGroups[tag] = {
              name: tag,
              item: [],
            };
          }

          const requestItem = {
            name: request.summary || `${method.toUpperCase()} ${path}`,
            request: {
              method: method.toUpperCase(),
              header: [
                {
                  key: "Content-Type",
                  value: "application/json",
                },
              ],
              url: {
                raw: `{{baseUrl}}${path}`,
                host: ["{{baseUrl}}"],
                path: path.split("/").filter(Boolean),
              },
              body: request.requestBody
                ? {
                    mode: "raw",
                    raw: JSON.stringify(request?.requestBody?.content["application/json"]?.example, null, 2),
                    options: {
                      raw: {
                        language: "json",
                      },
                    },
                  }
                : undefined,
            },
          };

          // Add test script for login route
          if (path === "/users/login" && method === "post") {
            requestItem.event = [
              {
                listen: "test",
                script: {
                  type: "text/javascript",
                  exec: [
                    "var res = pm.response.json();",
                    "pm.environment.set('harmony_token', res.token);",
                    "pm.environment.set('token', res.token);",
                  ],
                },
              },
            ];
          }

          tagGroups[tag].item.push(requestItem);
        });
      });
    });

    // Add tags as folders to Postman collection
    Object.values(tagGroups).forEach((tagGroup) => {
      postmanCollection.item.push({
        name: tagGroup.name,
        item: tagGroup.item,
      });
    });

    // Update Postman collection using Postman API
    await axios.put(
      `https://api.getpostman.com/collections/${collectionId}`,
      { collection: postmanCollection },
      { headers: { "X-Api-Key": postmanApiKey } }
    );

    console.log("Postman collection updated successfully!");
  } catch (error) {
    console.error("Error updating Postman collection:", error);
  }
};

const run = async () => {
  const user = await askUserInput(
    `Please enter the user to use credentials for (available: ${Object.keys(credentials).join(", ")}): `
  );

  if (!credentials[user]) {
    console.error("Invalid user specified.");
    return;
  }

  const { POSTMAN_API_KEY, POSTMAN_COLLECTION_ID } = credentials[user];
  const collectionName = process.argv[2] || COLLECTION_NAME;

  await updatePostmanCollection(collectionName, POSTMAN_API_KEY, POSTMAN_COLLECTION_ID);
};

run();
