import { Server } from "socket.io";
import db from "./db-config.js";
import { storeError } from "./helper/general.js";
import jwt from "jsonwebtoken";

const handleError = (error) => {
  storeError(error);
};

const handleConnection = (io) => async (socket) => {
  try {
    // const token = socket.handshake.auth.token;

    const token = socket.handshake.auth.token;
    if (token) {
      const decodedToken = jwt.verify(token, process.env.JWT_KEY);
      const { sessionid } = decodedToken;
      socket.user = decodedToken;
      await db.query("UPDATE users SET socket_id = ? WHERE id = ?", [
        socket.id,
        sessionid,
      ]);
    } else {
      // console.error("Token is required, please login again");
      // socket.emit("error", "Token is required, please login again");
      return;
    }
  } catch (error) {
    handleError(error);
    socket.emit("error", "Authentication error");
    return;
  }

  // Enhanced debugging inside the event handler
  socket.on("send-message", async (data) => {
    if (!data || typeof data !== "object") {
      console.error("Invalid data format received");
      return;
    }

    const { senderId, receiverId, message } = data;

    if (!senderId || !receiverId || !message) {
      console.error("Missing required fields in send-message data");
      return;
    }

    try {
      let chatRoomId;
      const findChatRoom = await db.query(
        "SELECT * FROM chat_room WHERE (sender_id = ? AND receiver_id = ?) OR (sender_id = ? AND receiver_id = ?) AND deleted = 0",
        [senderId, receiverId, receiverId, senderId]
      );
      if (findChatRoom.length === 0) {
        const query = await db.query(
          "INSERT INTO chat_room (sender_id, receiver_id) VALUES (?, ?)",
          [senderId, receiverId]
        );
        chatRoomId = query.insertId;
      } else {
        chatRoomId = findChatRoom[0].id;
      }

      await db.query(
        "INSERT INTO messages (chat_room_id, sender_id, message) VALUES (?, ?, ?)",
        [chatRoomId, senderId, message]
      );

      socket.emit("message-sent", { senderId, receiverId, message });

      const receiverSocket = await db.query(
        "SELECT socket_id FROM users WHERE id = ?",
        [receiverId]
      );

      if (receiverSocket?.length > 0 && receiverSocket[0].socket_id !== null) {
        io.to(receiverSocket[0].socket_id).emit("message-received", {
          senderId,
          receiverId,
          message,
        });
      } else {
        // console.log("Receiver is not connected");
      }
    } catch (error) {
      handleError(error);
    }
  });

  /**Socket for getting users who is online */
  socket.on("checkOnlineUsers", async () => {
    try {
      const checkOnlineUsers = await db.query(
        "SELECT id FROM users WHERE deleted = 0 AND socket_id IS NOT NULL"
      );
      const onlineUsers =
        checkOnlineUsers.length > 0
          ? checkOnlineUsers.map((user) => user.id)
          : [];
      io.emit("OnlineUsers", onlineUsers);
    } catch (error) {
      handleError(error);
    }
  });

  socket.on("disconnect", async () => {
    // console.log(`User ${socket.id} disconnected`);
    try {
      const { sessionid } = socket.user;
      await db.query("UPDATE users SET socket_id = NULL WHERE id = ?", [
        sessionid,
      ]);

      const checkOnlineUsers = await db.query(
        "SELECT id FROM users WHERE deleted = 0 AND socket_id IS NOT NULL"
      );
      const onlineUsers =
        checkOnlineUsers.length > 0
          ? checkOnlineUsers.map((user) => user.id)
          : [];
      io.emit("OnlineUsers", onlineUsers);
    } catch (error) {
      handleError(error);
    }
  });
};

const initializeSocket = (server) => {
  try {
    const io = new Server(server, {
      cors: {
        origin: "*",
      },
    });

    io.on("connection", handleConnection(io));
  } catch (error) {
    handleError(error);
  }
};

export default initializeSocket;
