// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateBanks,
  deleteBanks,
  viewBanks,
} from "../controller/banksController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";
/**
 * @swagger
 * tags:
 *   name: Bank
 *   description: API operations for managing all Bank
 */
const banksRoutes = Router();

/**
 * @swagger
 * /api/banks/create-update-bank:
 *   post:
 *     summary: Create or update an Bank
 *     tags: [Bank]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Policy Impact Name"
 *             description: "Description of the bank"
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Bank created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
banksRoutes.post(
  "/create-update-bank",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 324, "banks")(
      req,
      res,
      next
    );
  },
  controllerWrapper(createUpdateBanks)
);

/**
 * @swagger
 * /api/banks/get-bank:
 *   get:
 *     summary: Get list of all Banks
 *     tags: [Bank]
 *     responses:
 *       200:
 *         description: List of banks
 *       500:
 *         description: Internal server error
 */
banksRoutes.get(
  "/get-bank",
  verifyToken,
  sideBarPermissionCheck("view", 324, "banks"),
  controllerWrapper(viewBanks)
);

/**
 * @swagger
 * /api/banks/get-bank/{id}:
 *   get:
 *     summary: Get a specific record of Bank
 *     tags: [Bank]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the bank
 *     responses:
 *       200:
 *         description: Bank data
 *       404:
 *         description: Bank not found
 *       500:
 *         description: Internal server error
 */
banksRoutes.get(
  "/get-bank/:id",
  verifyToken,
  sideBarPermissionCheck("view", 324, "banks"),
  controllerWrapper(viewBanks)
);

/**
 * @swagger
 * /api/banks/delete-bank/{id}:
 *   delete:
 *     summary: Delete a specific record of Bank
 *     tags: [Bank]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the bank to delete
 *     responses:
 *       200:
 *         description: Bank successfully deleted
 *       404:
 *         description: Bank not found
 *       500:
 *         description: Internal server error
 */
banksRoutes.delete(
  "/delete-bank/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 324, "banks"),
  controllerWrapper(deleteBanks)
);

export default banksRoutes;
